/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2014 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.util;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.StringWriter;
import java.io.UnsupportedEncodingException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.apache.http.HttpResponse;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.gson.Gson;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.openexchange.guard.config.Config;
import com.openexchange.guard.database.Access;
import com.openexchange.guard.encr.EncrLib;
import com.openexchange.guard.encr.GuardKeys;
import com.openexchange.guard.exceptions.GuardMissingParameter;


public class Core {
	
	private static Logger logger = LoggerFactory.getLogger(Core.class);
	private static PrivateKey passwordKey = null;
	private static String pubkey = null;

    /**
     * Send answer to response stream
     * 
     * @param response
     * @param answer String to put in HTML
     */
    public static void sendOK(HttpServletResponse response, String answer) {
        response.setContentType("text/html; charset=UTF-8");
        response.addHeader("Access-Control-Allow-Origin", "*");
        response.setStatus(HttpServletResponse.SC_OK);
        try {
            response.getWriter().println(answer);
        } catch (IOException e) {
        	logger.error("Error sending OK message", e);
        }
    }

    /**
     * Send answer to response stream
     * 
     * @param response
     * @param answer String to put in HTML
     */
    public static void sendOK(HttpServletResponse response, String type, byte[] answer) {
        response.setContentType(type);

        response.addHeader("Access-Control-Allow-Origin", "*");
        response.setStatus(HttpServletResponse.SC_OK);
        try {
            OutputStream out = response.getOutputStream();
            out.write(answer);
            out.close();
        } catch (IOException e) {
        	logger.error("Error sending OK message", e);
        }
    }

    public static void sendFail(HttpServletResponse response, String answer) {
        response.setContentType("text/html");
        response.addHeader("Access-Control-Allow-Origin", "*");
        response.setStatus(HttpServletResponse.SC_NOT_ACCEPTABLE);
        try {
            response.getWriter().println(answer);
        } catch (IOException e) {
        	logger.error("Error sending NOT ACCEPTABLE message", e);
        }

    }
    
    public static void sendError(HttpServletResponse response, String answer) {
        response.setContentType("text/html");
        response.addHeader("Access-Control-Allow-Origin", "*");
        response.setStatus(HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
        try {
            response.getWriter().println(answer);
        } catch (IOException e) {
        	logger.error("Error sending Error message", e);
        }

    }

    public static JsonObject getJSON(HttpServletRequest request) {
        StringBuffer sb = new StringBuffer();
        String line = null;
        try {
            BufferedReader reader = request.getReader();
            while ((line = reader.readLine()) != null) {
                sb.append(line);
            }
            reader.close();
        } catch (Exception e) {
            logger.error("Error reading from request", e);
        } 
        JsonParser jsonParser = new JsonParser();
        JsonObject json = (JsonObject) jsonParser.parse(sb.toString());
        return (checkEpass(json));
    }
    
    public static JsonObject getJSON (String data) {
    	JsonParser jsonParser = new JsonParser();
        JsonObject json = (JsonObject) jsonParser.parse(data);
        return (json);
    }

    /**
     * Checks for error in reply string
     * 
     * @param response
     * @return true if contains error
     */
    public static boolean checkFail(String response) {
        if (response == null) return(true);
        if (response.contains("\"data\":null")) {
        	logger.debug("No error");
        } else logger.debug(response);
        if (response.contains("error")) {
            logger.info(response);
            return (true);
        }
        return (false);
    }
    
    public static boolean checkFail(JsonObject resp) {
    	Gson gson = new Gson();
    	return (checkFail(gson.toJson(resp)));
    }
    
    public static String getError (String response) {
    	try {
    		int i = response.indexOf("{");
    		int j = response.lastIndexOf("}");
    		response = response.substring(i, j+ 1);
    		JsonParser parser = new JsonParser();
    		JsonObject resp = (JsonObject)parser.parse(response);
    		if (resp.has("error_desc")) {
    			return (resp.get("error_desc").getAsString());
    		}
    		else return ("");
    	} catch (Exception ex) {
    		return ("");
    	}
    }

    public static boolean isEncrItem(byte[] data) {
        String head = new String(data, 0, data.length > 100 ? 100 : data.length);
        try {
            Pattern p = Pattern.compile("\\bID: [A-F0-9]{8}(?:-[A-F0-9]{4}){3}-[A-Z0-9]{12}\\b", Pattern.CASE_INSENSITIVE);
            Matcher m = p.matcher(head);
            if (m.find()) {
                return (true);
            }
        } catch (Exception e) {
            logger.error("Error determining whether item is encrypted", e);
        }
        return (false);
    }

    public static String dbformat(String data) {
        if (data == null)
            return (data);
        data = data.replace("'", "\'");
        data = data.replace("`", "\'");
        data = data.replace("\\", "\\\\");
        return (data);
    }
    
    public static String gettime (long start) {
    	return (((System.nanoTime() - start) / 1000000) + "ms");
    }
    
    /**
     * Format response into a JSON object
     * 
     * @param response
     * @return
     * @throws UnsupportedEncodingException
     * @throws IllegalStateException
     * @throws IOException
     */
    public static JsonObject getJson(CloseableHttpResponse response) throws UnsupportedEncodingException, IllegalStateException, IOException {
        BufferedReader reader = new BufferedReader(new InputStreamReader(response.getEntity().getContent(), "UTF-8"));
        String json = reader.readLine();
        JsonParser parser = new JsonParser();
        JsonObject result = (JsonObject) parser.parse(json);
        response.close();
        return (result);
    }
    
    public static String getXMLString(org.w3c.dom.Document doc) throws TransformerException {
        StringWriter writer = new StringWriter();
        StreamResult result = new StreamResult(writer);
        TransformerFactory tf = TransformerFactory.newInstance();
        Transformer transformer = tf.newTransformer();
        DOMSource domSource = new DOMSource(doc);
        transformer.transform(domSource, result);
        String XML = writer.toString();
        if (XML.contains("?>")) {
            XML = XML.substring(XML.indexOf("?>") + 2);
        }
        return (XML);
    }
    
    private static JsonObject checkEpass (JsonObject result) {
    	if (result == null) return(null);
    	result = decr(result, "e_password", "password");
    	result = decr(result, "e_encr_password", "encr_password");
    	result = decr(result, "e_ox_password", "ox_password");
    	result = decr(result, "e_extrapass", "extrapass");
        return (result);
    }
    
    private static JsonObject decr(JsonObject result, String key, String newkey) {
    	if (result.has(key)) {
    		String encr = result.get(key).getAsString();
    		if (passwordKey == null) loadPrivate();
        	if (passwordKey != null) {
        		EncrLib encrlib = new EncrLib();
        		String decoded = encrlib.decryptRSA(encr, passwordKey);
        		if (decoded !=null) {
        			result.addProperty(newkey, decoded);
        		}
        	}
    	}
        return(result);
    }
    
    private static void loadPrivate () {
    	try {
	    	Access acc = new Access();
	    	GuardKeys passwordKeys = acc.getKeys(-2, 0);
	    	if (passwordKeys != null) {
	    		passwordKey = passwordKeys.getDecodedPrivate(Config.rpass);
	    	}
    	} catch (Exception ex) {
    		logger.error("Problem loading password RSA key", ex);
    		
    	}
    	
    }
    
    public static PrivateKey getPasswordKey () {
    	if (passwordKey == null) loadPrivate();
    	return (passwordKey);
    }
    
    private static void loadPublic() {
    	Access acc = new Access();
    	GuardKeys passwordKeys;
		try {
			passwordKeys = acc.getKeys(-2, 0);
	        if (passwordKeys != null) {
	        	pubkey =  passwordKeys.getEncodedPublic();
	        }
		} catch (Exception e) {
			logger.error("Error getting public RSA key ", e);
		}

    }
    
    public static String getPublicKey() {
    	if (pubkey == null) loadPublic();
    	return (pubkey);
    }
    
    public static String getStringParameter (HttpServletRequest request, String name) throws GuardMissingParameter {
    	return (getStringParameter(request, name, false));
    }
    
    /**
     * Get String from HttpServletRequest
     * @param request
     * @param name
     * @param mandatory
     * @return
     * @throws GuardMissingParameter
     */
    public static String getStringParameter (HttpServletRequest request, String name, boolean mandatory) throws GuardMissingParameter {
    	String item = request.getParameter(name);
    	if (mandatory && (item == null)) {
    		throw new com.openexchange.guard.exceptions.GuardMissingParameter("Missing parameter " + name);
    	}
    	return (item);
    }
    
    public static int getIntParameter (HttpServletRequest request, String name) throws GuardMissingParameter {
    	return (getIntParameter (request, name, false));
    }
    
    /**
     * Get int from HttpServletRequest
     * @param request
     * @param name
     * @param mandatory
     * @return
     * @throws GuardMissingParameter
     */
    public static int getIntParameter (HttpServletRequest request, String name, boolean mandatory) throws GuardMissingParameter {
    	String item = request.getParameter(name);
    	if (item == null) {
    		if (mandatory) {
    			throw new com.openexchange.guard.exceptions.GuardMissingParameter("Missing parameter " + name);
    		} else return(0);
    	}
    	try {
    		int val = Integer.parseInt(item);
    		return (val);
    	} catch (Exception e){
    		logger.error("Problem getting integer from parameter", e);
    		throw new com.openexchange.guard.exceptions.GuardMissingParameter("Bad integer value for parameter " + name);
    	}
    }
    
    public static String getStringFromJson (JsonObject json, String name) throws GuardMissingParameter {
    	return (getStringFromJson (json, name, false));
    }
    
    /**
     * Get String from JsonObject
     * @param json
     * @param name
     * @param mandatory
     * @return
     * @throws GuardMissingParameter
     */
    public static String getStringFromJson (JsonObject json, String name, boolean mandatory) throws GuardMissingParameter {
    	if (!json.has(name)) {
    		if (mandatory) {
    			throw new com.openexchange.guard.exceptions.GuardMissingParameter("Missing JSON parameter " + name);
    		}
    		else return(null);
    	}
    	try {
    		String dat = json.get(name).getAsString();
    		return (dat);
    	} catch (Exception ex) {
    		logger.error("Problem getting string parameter " + name + "from json" , ex);
    		throw new com.openexchange.guard.exceptions.GuardMissingParameter("Problem getting string from json for parameter " + name);
    	}
    }
    
    public static int getIntFromJson (JsonObject json, String name) throws GuardMissingParameter {
    	return (getIntFromJson (json, name, false));
    }
    
    /**
     * Get int from JsonObject
     * @param json
     * @param name
     * @param mandatory
     * @return
     * @throws GuardMissingParameter
     */
    public static int getIntFromJson (JsonObject json, String name, boolean mandatory) throws GuardMissingParameter {
    	if (!json.has(name)) {
    		if (mandatory) {
    			throw new com.openexchange.guard.exceptions.GuardMissingParameter("Missing JSON parameter " + name);
    		}
    		else return(0);
    	}
    	try {
    		int dat = json.get(name).getAsInt();
    		return (dat);
    	} catch (Exception ex) {
    		logger.error("Problem getting int parameter " + name + "from json" , ex);
    		throw new com.openexchange.guard.exceptions.GuardMissingParameter("Problem getting int from json for parameter " + name);
    	}
    }
    
    public static String getErrorDescr(String error) {
    	try {
    		if (error.contains("{")) {
    			int end = error.indexOf("}")  + 1;
    			String jsonstring = error.substring(error.indexOf("{"), end);
    			JsonParser jsonParser = new JsonParser();
                JsonObject json = (JsonObject) jsonParser.parse(jsonstring);
                return (json.get("error_desc").getAsString());
    		}
    	} catch (Exception e) {
    		logger.error("Error getting error description ", e);
    	}
    	return ("");
    }
    
    public static String getIP (HttpServletRequest request) {
    	if (request.getHeader("X-Forwarded-For") != null) {
    		return (request.getHeader("X-Forwarded-For"));
    	}
    	if (request.getRemoteAddr().equals("127.0.0.1")) return(null);
    	return (request.getRemoteAddr());
    }
}
