/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2014 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.ox;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang3.StringEscapeUtils;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.DefaultHttpClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Element;
import org.xbill.DNS.Lookup;
import org.xbill.DNS.Record;
import org.xbill.DNS.TXTRecord;
import org.xbill.DNS.Type;
import org.xml.sax.InputSource;

import com.openexchange.guard.config.Config;
import com.openexchange.guard.database.DbCommand;
import com.openexchange.guard.database.DbQuery;
import com.openexchange.guard.database.GetPubKey;
import com.openexchange.guard.database.RecipKey;
import com.openexchange.guard.dbpool.ContextDBConnection;
import com.openexchange.guard.dbpool.Manager;
import com.openexchange.guard.encr.GuardKeys;
import com.openexchange.guard.exceptions.GuardMissingParameter;
import com.openexchange.guard.util.Core;


/*
 * Class for finding location of other OX servers and getting the DNS info for retrieving certs
 */
public class InterOx {
	private static Logger logger = LoggerFactory.getLogger(InterOx.class);
	
    public InterOxInfo getOXLocation(String email) {
        try {
            if (email.contains("@"))
                email = email.substring(email.indexOf("@") + 1);
            InterOxInfo info = incache(email);
            if (info != null) {
            	if (compareSqlDate(info.lastcheck)) {  // If checked today, then use, else reload
        			return (info);
        		}
            }
            String url = null;
            Record[] records = new Lookup("_oxguard." + email, Type.TXT).run();
            if (records == null) {
                return (null);
            }
            for (int i = 0; i < records.length; i++) {
                TXTRecord txt = (TXTRecord) records[i];
                if (txt.getName().toString().toLowerCase().contains("oxguard")) { // make sure this is oxguard record
                    url = txt.rdataToString().replace("\"", "").replace("\\", "");
                    if (!url.endsWith("/"))
                        url = url + "/";
                    // make sure ssl
                    if (url.startsWith("http:"))
                        url = url.replace("http:", "https:");
                    info = saveInCache(email, url);
                    return (info);
                }

            }
            return (null);
        } catch (Exception ex) {
            logger.error("Error finding system for user " + email, ex);
            return (null);
        }
    }

    private InterOxInfo incache(String domain) {
        InterOxInfo info = null;
        String check = "SELECT * FROM iox WHERE DOMAIN = ?";
        try {
            DbCommand com = new DbCommand(check);
            com.addVariables(domain);
            DbQuery db = new DbQuery();
            db.readOG(com);
            if (db.next()) {
                info = new InterOxInfo(db.rs);
            }
            db.close();
        } catch (Exception ex) {
            logger.error("Error checking cache for domain " + domain, ex);
        }
        return (info);

    }

    private InterOxInfo saveInCache(String domain, String url) {
        String command = "INSERT INTO iox (domain, url, lastcheck) VALUES (?, ?, NOW()) ON DUPLICATE KEY UPDATE url = ?, lastcheck = NOW()";
        try {
            DbCommand com = new DbCommand(command);
            com.addVariables(domain);
            com.addVariables(url);
            com.addVariables(url);
            DbQuery db = new DbQuery();
            db.writeOxGuard(com);
            db.close();
        } catch (Exception ex) {
            logger.error("Error saving url " + url + " for domain " + domain, ex);

        }
        return (new InterOxInfo(domain, url));

    }

    // ////// Receiving end of interox

    /**
     * Get public Key of email address.  Will not create key unless from and token populated
     * @param request
     * @param response
     * @throws GuardMissingParameter
     */
    public void getKey(HttpServletRequest request, HttpServletResponse response) throws GuardMissingParameter {
        String from = Core.getStringParameter(request, "from");
        String token = Core.getStringParameter(request, "token");
        String email = Core.getStringParameter(request, "email", true);
        if (com.openexchange.guard.server.CheckBad.isBad(Core.getIP(request), Config.bad_ip_count)) {
            sendFail(response, "Temporary lockout");
            return;
        }
        if (!email.contains("@")) {
            sendFail(response, "Bad email format");
            return;
        }
        try {
            boolean create = false;
            if ((from != null) && (token != null)) {
                InterOxInfo info = getOXLocation(from);
                if (info != null) {
                    if (info.token == null) {
                        create = verifyToken(info, token);
                    } else {
                        if (token.trim().equals(info.token)) {
                            create = true; // Token verifies with what we have in storage, ok to create
                        } else {
                        	logger.error("Token mismatch.  Requesting server shows token " + token + ", expecting " + info.token);
                        	if (!compareSqlDate(info.lastcheck)) {
                        		logger.info("Token not verified today, trying again");
                        		create = verifyToken(info, token);
                        		if (create) logger.info("OK Verified");
                        	}
                        }
                    }
                }
            }
            RecipKey keys = null;
            if (create) {
                GetPubKey gk = new GetPubKey();
                keys = gk.getPubKey(email, 0, 0, from, true, true);
            } else {
                keys = simipleCheck(email);
            }

            if (keys != null) {
                if (keys.pubkey == null) {
                    sendFail(response, "Unable to create keys");
                } else {
                	sendOK(response, (Base64.encodeBase64String(keys.pubkey.getEncoded())));
                }
            } else {
                com.openexchange.guard.server.CheckBad.addBad(Core.getIP(request));
                sendFail(response, "Unable to create keys");
            }
        } catch (Exception ex) {
            logger.error("Error getting key for email " + email, ex);
            sendFail(response, "Error processing request");
        }

    }

    public void saveItem(HttpServletRequest request, HttpServletResponse response) {
        logger.debug("Receiving item");
        String xml = getRequest(request);
        logger.debug("XML: " + xml);
        if (com.openexchange.guard.server.CheckBad.isBad("r-" + Core.getIP(request), Config.bad_ip_count)) {
            sendFail(response, "Temporary lockout");
            return;
        }
        DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
        try {
            DocumentBuilder builder = factory.newDocumentBuilder();
            org.w3c.dom.Document doc = builder.parse(new InputSource(new ByteArrayInputStream(xml.getBytes("utf-8"))));
            doc.getDocumentElement().normalize();
            Element item = (Element) doc.getElementsByTagName("item").item(0);
            String itemid = item.getElementsByTagName("itemid").item(0).getTextContent();
            String email = item.getElementsByTagName("email").item(0).getTextContent();
            String contentKey = item.getElementsByTagName("contentkey").item(0).getTextContent();
            long Expiration = Long.parseLong(item.getElementsByTagName("expiration").item(0).getTextContent());
            int Type = Integer.parseInt(item.getElementsByTagName("type").item(0).getTextContent());
            String XML = new String(Base64.decodeBase64(item.getElementsByTagName("xml").item(0).getTextContent()));
            String Salt = item.getElementsByTagName("salt").item(0).getTextContent();
            if ((email == null) || (contentKey == null) || (XML == null)) {
                sendFail(response, "Bad data");
                return;
            }
            if (com.openexchange.guard.server.CheckBad.isBad("r-" + email, Config.max_remote)) {  // This is checking for rate limit
                sendFail(response, "Temporary lockout");
                return;
            }
            logger.debug("Storing");
            GetPubKey gk = new GetPubKey();
            RecipKey keys = gk.getPubKey(email, 0, 0, null, false, true);
            if (keys != null) {
                if (keys.pubkey == null) {
                    sendFail(response, "Unable to create keys");
                } else {
                    com.openexchange.guard.database.Access acc = new com.openexchange.guard.database.Access();
                    acc.storeItemID(itemid, 0, keys.cid, 0, 0, Expiration, Type, XML, Salt);
                    acc.storeContentKey(itemid, keys.userid, keys.cid, contentKey);
                    logger.info("Recieved item " + itemid);
                    sendOK(response, "OK");
                    com.openexchange.guard.server.CheckBad.addBad("r-" + email);  // Here we are adding email to bad, not because fail, but for rate limit
                }
            } else {
            	com.openexchange.guard.server.CheckBad.addBad("r-" + Core.getIP(request));
                sendFail(response, "Fail");
            }

        } catch (Exception ex) {
            logger.error("Error saving item", ex);
            com.openexchange.guard.server.CheckBad.addBad("r-" + Core.getIP(request));
            sendFail(response, "Bad request");
        }

    }

    public void verifyToken(HttpServletRequest request, HttpServletResponse response) throws GuardMissingParameter {
        String token = Core.getStringParameter(request, "token", true);
        if (!com.openexchange.guard.server.CheckBad.isBad(Core.getIP(request), Config.bad_ip_count)) {
            if (token.trim().equals(Config.token.trim())) {
                com.openexchange.guard.util.Core.sendOK(response, "true");
                return;
            }
        }
        com.openexchange.guard.server.CheckBad.addBad(Core.getIP(request));
        com.openexchange.guard.util.Core.sendOK(response, "false");
    }

    public void getReadCount(HttpServletRequest request, HttpServletResponse response) throws GuardMissingParameter {
        String email = Core.getStringParameter(request, "email", true);
        String itemid = Core.getStringParameter(request, "item", true);

        if (!email.contains("@")) {
            sendFail(response, "Bad email format");
            com.openexchange.guard.server.CheckBad.addBad(Core.getIP(request));
            return;
        }
        if (com.openexchange.guard.server.CheckBad.isBad(Core.getIP(request), Config.bad_ip_count)) {
        	sendFail(response, "Lockout");
        	return;
        }
        com.openexchange.guard.database.Access acc = new com.openexchange.guard.database.Access();
        try {
            GuardKeys keys = acc.getKeysFromEmail(email);
            if (keys != null) {

                String command = "SELECT Status FROM og_content_keys WHERE ItemID = ? AND UserID = ? AND CID = ?";
                DbCommand com = new DbCommand(command);
                com.addVariables(itemid);
                com.addVariables(keys.userid);
                com.addVariables(keys.contextid);
                DbQuery db = new DbQuery();
                db.read(com, keys.userid, keys.contextid);
                if (db.next()) {
                    int data = db.rs.getInt("Status");
                    if (Config.showStatus == false) {
                    	if (data > -1) data = -9999;
                    }
                    sendOK(response, Integer.toString(data));

                }
                db.close();
            } else {
            	com.openexchange.guard.server.CheckBad.addBad(Core.getIP(request));
            }
        } catch (Exception ex) {
            logger.error("Error getting read count for item " + itemid + " for user " + email, ex);
            sendFail(response, "Problem with getting read status");
        }

    }

    public boolean deleteshare(HttpServletRequest request, HttpServletResponse response) throws GuardMissingParameter {
    	String email = Core.getStringParameter(request, "email", true);
    	String itemid = Core.getStringParameter(request, "item", true);
    	String salt = Core.getStringParameter(request, "salt", true);
        try {

            if (!email.contains("@")) {
                sendFail(response, "Bad email format");
                com.openexchange.guard.server.CheckBad.addBad(Core.getIP(request));
                return (false);
            }
            if (com.openexchange.guard.server.CheckBad.isBad(Core.getIP(request), Config.bad_ip_count)) {
            	sendFail(response, "Lockout");
            	return (false);
            }
            com.openexchange.guard.database.Access acc = new com.openexchange.guard.database.Access();
            GuardKeys keys = acc.getKeysFromEmail(email);
            if (keys != null) {
                String command = "DELETE og.* FROM og_content_keys og INNER JOIN og_encrypted_items encr ON og.ItemId = encr.ID WHERE ItemId = ? AND UserId = ? AND CID = ? AND encr.Salt = ?";
                DbCommand com = new DbCommand(command);
                com.addVariables(itemid);
                com.addVariables(keys.userid);
                com.addVariables(keys.contextid);
                com.addVariables(salt);
                DbQuery db = new DbQuery();
                db.write(com, keys.userid, keys.contextid);
                logger.debug("updated: " + db.updated);
                String resp = "OK";
                if (db.updated < 1) resp = "No";
                db.close();
                Core.sendOK(response, resp);
                return (true);
            }
        } catch (Exception ex) {
            logger.error("Error deleting share of item " + itemid + " for user " + email, ex);

        }
        com.openexchange.guard.server.CheckBad.addBad(Core.getIP(request));
        Core.sendFail(response, "Fail");
        return (false);
    }

    public boolean retract(HttpServletRequest request, HttpServletResponse response) throws GuardMissingParameter {
    	String email = Core.getStringParameter(request, "email", true);
    	String itemid = Core.getStringParameter(request, "item", true);
    	String salt = Core.getStringParameter(request, "salt", true);
    	boolean delete = Config.deleteOnRetract;
    	try {
    		if (request.getParameter("delete") != null) {
    			delete = request.getParameter("delete").equals("true");
    		}
    	} catch (Exception e) {
    		logger.error("Error parsing retract request", e);
    	}
        try {
            if (!email.contains("@")) {
                sendFail(response, "Bad email format");
                com.openexchange.guard.server.CheckBad.addBad(Core.getIP(request));
                return (false);
            }
            com.openexchange.guard.database.Access acc = new com.openexchange.guard.database.Access();
            GuardKeys keys = acc.getKeysFromEmail(email);
            if (keys != null) {
                if (acc.retract(keys.userid, keys.contextid, itemid, salt, delete)) {
                    logger.info("Retracted item " + itemid);
                    sendOK(response, "OK");
                    return (true);
                }

            }
        } catch (Exception ex) {
            logger.error("Error retracting item " + itemid + " for user " + email, ex);

        }
        com.openexchange.guard.server.CheckBad.addBad(Core.getIP(request));
        return (false);

    }

    private String getRequest(HttpServletRequest request) {
        StringBuffer sb = new StringBuffer();
        String line = null;
        try {
            BufferedReader reader = request.getReader();
            while ((line = reader.readLine()) != null) {
                sb.append(line);
                if (sb.length() > 100000) { // No reason the XML request should exceed this
                	logger.error("Exceed buffer limits in interOX query");
                	return(sb.toString());
                }
            }
        } catch (IOException e) {
            logger.error("IO exception while reading", e);
        }
        return (sb.toString());
    }

    private void sendFail(HttpServletResponse response, String answer) {
        response.setContentType("text/html");
        response.addHeader("Access-Control-Allow-Origin", "*");
        response.setStatus(HttpServletResponse.SC_NOT_ACCEPTABLE);
        try {
            response.getWriter().println(answer);
        } catch (IOException e) {
        	logger.error("IO error while writing error response", e);
        }

    }

    /**
     * Send answer to response stream
     * 
     * @param response
     * @param answer String to put in HTML
     */
    private void sendOK(HttpServletResponse response, String answer) {
        response.setContentType("text/html");
        response.addHeader("Access-Control-Allow-Origin", "*");
        response.setStatus(HttpServletResponse.SC_OK);
        try {
            response.getWriter().println(answer);
        } catch (IOException e) {
        	logger.error("IO error while writing OK response", e);
        }
    }

    // /// Sending side of Interox

    public int getRemoteReadCount(String email, String item) {
        String url = getOXLocation(email).url + "interox?action=getcount&email=" + email + "&item=" + item;
        HttpClient httpClient = new DefaultHttpClient();
        try {
            HttpGet request = new HttpGet(url);
            HttpResponse response = httpClient.execute(request);
            BufferedReader reader = new BufferedReader(new InputStreamReader(response.getEntity().getContent(), "UTF-8"));
            String answer = reader.readLine();
            int count = Integer.parseInt(answer);
            reader.close();
            return (count);
        } catch (Exception ex) {
            logger.error("Error while getting read count for item " + item + " for user " + email, ex);
            return (-1);
        }

    }

    /**
     * Send request to retract email from remote source
     * 
     * @param email
     * @param item
     * @param salt
     * @return
     */
    public String retractRemote(String email, String item, String salt, boolean delete) {
        try {
            String url = getOXLocation(email).url + "interox?action=retract&email=" + email + "&item=" + item + "&salt=" + URLEncoder.encode(
                salt ,"ISO-8859-1") + "&delete=" + (delete ? "true" : "false");
            HttpClient httpClient = new DefaultHttpClient();
            HttpGet request = new HttpGet(url);
            HttpResponse response = httpClient.execute(request);
            BufferedReader reader = new BufferedReader(new InputStreamReader(response.getEntity().getContent(), "UTF-8"));
            String answer = reader.readLine();
            reader.close();
            return (answer);
        } catch (Exception ex) {
            logger.error("Error while retracting remote item " + item + " for user " + email, ex);
            return ("Fail");
        }

    }

    /**
     * Send delete share to remote source
     * 
     * @param email
     * @param item
     * @param salt
     * @return
     */
    public boolean deleteshare(String email, String item, String salt) {
        try {
            String url = getOXLocation(email).url + "interox?action=remshare&email=" + email + "&item=" + item + "&salt=" + URLEncoder.encode(
                salt,
                "ISO-8859-1");
            HttpClient httpClient = new DefaultHttpClient();
            HttpGet request = new HttpGet(url);
            HttpResponse response = httpClient.execute(request);
            BufferedReader reader = new BufferedReader(new InputStreamReader(response.getEntity().getContent(), "UTF-8"));
            String answer = reader.readLine();
            reader.close();
            if (answer.contains("OK")) return (true);
            return(false);
        } catch (Exception ex) {
            logger.error("Error while deleting share of item " + item + " for user " + email, ex);
            return (false);
        }

    }

    public boolean verifyToken(InterOxInfo info, String token) {
        try {
            String url = info.url + "interox?action=verify&token=" + URLEncoder.encode(token, "ISO-8859-1");
            HttpClient httpClient = new DefaultHttpClient();
            HttpGet request = new HttpGet(url);
            HttpResponse response = httpClient.execute(request);
            BufferedReader reader = new BufferedReader(new InputStreamReader(response.getEntity().getContent(), "UTF-8"));
            String answer = reader.readLine();
            reader.close();
            if (answer.contains("true")) {
                updateToken(info, token);
                return (true);
            }
            return (false);
        } catch (Exception ex) {
            logger.error("Error while verifying token " + token, ex);
            return (false);
        }

    }

    public RecipKey getRemotePublicKey(String email, String from) {
        InterOxInfo info = getOXLocation(email);
        if (info == null)
            return (null);
        String url = info.url;
        HttpClient httpClient = new DefaultHttpClient();
        try {
            url += "interox?action=getkey&email=" + URLEncoder.encode(email, "ISO-8859-1") + "&from=" + URLEncoder.encode(
                from,
                "ISO-8859-1") + "&token=" + URLEncoder.encode(Config.token, "ISO-8859-1");
            HttpGet request = new HttpGet(url);
            HttpResponse response = httpClient.execute(request);
            BufferedReader reader = new BufferedReader(new InputStreamReader(response.getEntity().getContent(), "UTF-8"));
            StringBuilder sb = new StringBuilder();
            String line = "";
            while ((line = reader.readLine()) != null) {
                sb.append(line);
            }
        	reader.close();
            if (sb.length() < 30) {

                return (null);
            }
            if (sb.toString().contains("Unable to create keys")) return(null);
            RecipKey result = new RecipKey();
            GuardKeys keys = new GuardKeys();
            keys.setPublicKeyFrom64String(sb.toString());
            result.pubkey = keys.getPublic();
            result.remote = true;
            return (result);
        } catch (Exception ex) {
            logger.error("Error while getting remote public key of " + email + " for " + from, ex);  //might have misunderstood the order here
            return (null);
        }

    }

    public boolean sendItem(String email, String itemID, String contentKey, long exp, int type, String XML, String salt) throws UnsupportedEncodingException {
        StringBuilder data = new StringBuilder();
        data.append("<item>");
        data.append("<email>" + StringEscapeUtils.escapeXml11(email) + "</email>");
        data.append("<itemid>" + itemID + "</itemid>");
        data.append("<contentkey>" + contentKey + "</contentkey>");
        data.append("<expiration>" + exp + "</expiration>");
        data.append("<type>" + type + "</type>");
        data.append("<xml>" + Base64.encodeBase64String(XML.getBytes("UTF-8")) + "</xml>");
        data.append("<salt>" + salt + "</salt>");
        data.append("</item>");
        String url = getOXLocation(email).url + "interox?action=saveitem";
        HttpClient httpClient = new DefaultHttpClient();
        try {
            HttpPost postRequest = new HttpPost(url);
            StringEntity input = new StringEntity(data.toString());
            postRequest.setEntity(input);
            HttpResponse response = null;
            response = httpClient.execute(postRequest);
            if (response.getStatusLine().getStatusCode() != 200) {
                throw new RuntimeException("Failed : HTTP error code : " + response.getStatusLine().getStatusCode());
            }
            BufferedReader reader = new BufferedReader(new InputStreamReader(response.getEntity().getContent(), "UTF-8"));
            String answer = reader.readLine();
            reader.close();
            if (answer.contains("OK"))
                return (true);
            return (false);

        } catch (Exception ex) {
            logger.error("Error sending item " + itemID + " for user " + email, ex);
            return (false);
        }

    }

    private void updateToken(InterOxInfo info, String token) {
        ContextDBConnection con = null;
        try {
            String command = "UPDATE iox SET token = ?, lastcheck = NOW() WHERE domain = ?";
            DbCommand com = new DbCommand(command);
            com.addVariables(token);
            com.addVariables(info.domain);
            DbQuery db = new DbQuery();
            db.writeOxGuard(com);
            db.close();
        } catch (Exception ex) {
            logger.error("Error updating token " + token + " for domain " + info.domain, ex);
        }
        Manager.closeOG(con);

    }

    private RecipKey simipleCheck(String email) {
        try {
            GetPubKey gk = new GetPubKey();
            RecipKey result = gk.checkMaster(email);
            return (result);
        } catch (Exception ex) {
            logger.error("Error doing check for user " + email, ex);
        }
        return (null);
    }
    
    /**
     * Check sql date with just date, not time, to current date
     * @param time
     * @return
     */
    private boolean compareSqlDate (Date time) {
    	Calendar now = new GregorianCalendar();
    	now.set(Calendar.HOUR, 0);
    	now.set(Calendar.MINUTE, 0);
    	now.set(Calendar.SECOND, 0);
    	now.set(Calendar.MILLISECOND, 0);
    	return (time.compareTo(now.getTime()) == 0);

    }

}
