/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2014 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.ox;

import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import javax.servlet.http.HttpServletRequest;

import org.apache.http.HttpEntity;
import org.apache.http.NameValuePair;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.CookieStore;
import org.apache.http.client.HttpClient;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.client.protocol.ClientContext;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.entity.mime.Header;
import org.apache.http.entity.mime.HttpMultipartMode;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.entity.mime.content.ByteArrayBody;
import org.apache.http.entity.mime.content.ContentBody;
import org.apache.http.impl.client.BasicCookieStore;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.protocol.BasicHttpContext;
import org.apache.http.protocol.HttpContext;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.openexchange.guard.config.Config;
import com.openexchange.guard.encr.EncryptedObj;
import com.openexchange.guard.mailcreator.Attachment;
import com.openexchange.guard.server.OxCookie;
import com.openexchange.guard.server.OxDbConn;
import com.openexchange.guard.util.Core;

/**
 * API class for communicating with OX servers
 * 
 * @author greg
 */
public class Api {
	private static Logger logger = LoggerFactory.getLogger(Api.class);
	
    HttpContext httpContext = null;

    CloseableHttpClient httpClient = null;

    String session;

    OxCookie cookie;

    String useragent;

    /**
     * Constructor for api. Logs in, and gets httpContext and Session ID
     * 
     * @param username
     * @param password
     */
    public Api(String username, String password) {
        login(username, password);
    }

    /**
     * Constructor from cookie
     * 
     * @param ck cookie
     * @param sess sessionId
     * @param ua User-agent
     */
    public Api(OxCookie ck, String sess, String ua) {
        cookie = ck;
        session = sess;
        useragent = checkCR(ua);
    }

    public Api(OxCookie ck, HttpServletRequest request) {
        cookie = ck;
        session = checkCR(request.getParameter("session"));
        useragent = checkCR(request.getHeader("User-Agent"));
    }

    /**
     * Close the httpClient and void context;
     */
    public void dispose() {
        httpClient.getConnectionManager().shutdown();
        httpContext = null;
        session = "";
    }

    /**
     * Login and populate the httpContext and SessionID
     * 
     * @param username
     * @param password
     * @return
     */
    public String login(String username, String password) {
        // httpClient = new DefaultHttpClient();
    	CloseableHttpClient httpClient = OxDbConn.httpClient;
        CookieStore cookieStore = new BasicCookieStore();
        httpContext = new BasicHttpContext();
        httpContext.setAttribute(ClientContext.COOKIE_STORE, cookieStore);
        try {
            HttpPost request = new HttpPost("http" + (Config.backend_ssl ? "s" : "") + "://" + Config.getOxBackend() + ":" + Config.ox_backend_port + Config.OXBackendPath + "login?action=login");
            List<NameValuePair> paramsv = new ArrayList<NameValuePair>(2);
            paramsv.add(new BasicNameValuePair("name", username));
            paramsv.add(new BasicNameValuePair("password", password));
            request.setEntity(new UrlEncodedFormEntity(paramsv));
            request.addHeader("content-type", "application/x-www-form-urlencoded");
            CloseableHttpResponse response = httpClient.execute(request, httpContext);
            BufferedReader reader = new BufferedReader(new InputStreamReader(response.getEntity().getContent(), "UTF-8"));
            String json = reader.readLine();
            JsonParser parser = new JsonParser();
            JsonObject result = (JsonObject) parser.parse(json);
            session = result.get("session").getAsString();
            reader.close();
            response.close();
            request.releaseConnection();
            return (session);
            // handle response here...
        } catch (Exception ex) {
            logger.error("Login error for user " + username, ex);
            // handle exception here
        } finally {
            // httpClient.getConnectionManager().shutdown();

        }
        return ("");
    }

    public CloseableHttpResponse getResponse(String params) throws IllegalStateException, IOException {

        if (httpClient == null) {
            // httpClient = new DefaultHttpClient();
        	httpClient = OxDbConn.httpClient;
        }

        HttpGet getRequest = new HttpGet("http" + (Config.backend_ssl ? "s" : "") + "://" + Config.getOxBackend() + ":" + Config.ox_backend_port + Config.OXBackendPath + params + "&session=" + session);
        getRequest.addHeader("accept", "application/json");
        getRequest.setHeader("Connection", "close");
        CloseableHttpResponse response = null;
        if (httpContext == null) {// If login in by cookie, add header and execute
            getRequest.addHeader("Cookie", cookie.getCookieHeader());
            getRequest.setHeader("User-Agent", useragent);
            response = httpClient.execute(getRequest);
        } else { // If logging into by previous login httpcontext, then use that
            response = httpClient.execute(getRequest, httpContext);
        }
        if (response.getStatusLine().getStatusCode() != 200) {
        	try {
        		response.close();
        		getRequest.releaseConnection();
        	} catch (Exception e2) {
        		logger.error("unable to close failed connection at API getRequest", e2);
        	}
            throw new RuntimeException("Failed : HTTP error code : " + response.getStatusLine().getStatusCode() + " " + response.getStatusLine().getReasonPhrase());
        }
        return (response);
    }

    public CloseableHttpResponse putRequest(String params, String data) throws IllegalStateException, IOException {

        if (httpClient == null) {
            // httpClient = new DefaultHttpClient();
            httpClient = OxDbConn.httpClient;
        }

        HttpPut putRequest = new HttpPut("http" + (Config.backend_ssl ? "s" : "") + "://" + Config.getOxBackend() + ":" + Config.ox_backend_port + Config.OXBackendPath + params + "&session=" + session);
        StringEntity entity = new StringEntity(data);
        entity.setContentType("text/javascript;charset=UTF-8");
        putRequest.addHeader("accept", "application/json");
        putRequest.setHeader("Connection", "close");

        putRequest.setEntity(entity);
        CloseableHttpResponse response = null;
        if (httpContext == null) {// If login in by cookie, add header and execute
            putRequest.addHeader("Cookie", cookie.getCookieHeader());
            putRequest.setHeader("User-Agent", useragent);
            response = httpClient.execute(putRequest);
        } else { // If logging into by previous login httpcontext, then use that
            response = httpClient.execute(putRequest, httpContext);
        }
        if (response.getStatusLine().getStatusCode() != 200) {
        	try {
        		response.close();
        		putRequest.releaseConnection();
        	} catch (Exception e2) {
        		logger.error("unable to close failed connection at API putRequest", e2);
        	}
            throw new RuntimeException("Failed : HTTP error code : " + response.getStatusLine().getStatusCode() + " " + response.getStatusLine().getReasonPhrase());
        }
        putRequest.releaseConnection();
        return (response);

    }

    public JsonObject postRequest(String data, String params) throws ClientProtocolException, IOException {

        if (httpClient == null) {
            // httpClient = new DefaultHttpClient();
            httpClient = OxDbConn.httpClient;
        }
        HttpPost postRequest = new HttpPost("http" + (Config.backend_ssl ? "s" : "") + "://" + Config.getOxBackend() + ":" + Config.ox_backend_port + Config.OXBackendPath + params);

        StringEntity input = new StringEntity(data);
        input.setContentType("application/json");
        postRequest.setEntity(input);
        CloseableHttpResponse response = null;
        if (httpContext == null) {// If login in by cookie, add header and execute
            postRequest.addHeader("Cookie", cookie.getCookieHeader());
            postRequest.setHeader("User-Agent", useragent);
            response = httpClient.execute(postRequest);
        } else { // If logging into by previous login httpcontext, then use that
            response = httpClient.execute(postRequest, httpContext);
        }

        if (response.getStatusLine().getStatusCode() != 200) {
        	try {
        		response.close();
        		postRequest.releaseConnection();
        	} catch (Exception e2) {
        		logger.error("unable to close failed connection at API postRequest", e2);
        	}
            throw new RuntimeException("Failed : HTTP error code : " + response.getStatusLine().getStatusCode() + " " + response.getStatusLine().getReasonPhrase());
        }
        BufferedReader reader = new BufferedReader(new InputStreamReader(response.getEntity().getContent(), "UTF-8"));
        String json = reader.readLine();
        JsonParser parser = new JsonParser();
        JsonObject result = (JsonObject) parser.parse(json);
        reader.close();
        response.close();
        postRequest.releaseConnection();
        return (result);

    }

    /**
     * For uploading encrypted files to server
     * 
     * @param data The json with folder, etc
     * @param params
     * @param attachments List of files, encrypted
     * @return
     * @throws ClientProtocolException
     * @throws IOException
     */
    public String filePost(String data, String params, ArrayList<Attachment> attachments) throws ClientProtocolException, IOException {

        if (httpClient == null) {
            // httpClient = new DefaultHttpClient();
            httpClient = OxDbConn.httpClient;
        }
        MultipartEntityBuilder entity = MultipartEntityBuilder.create();
        entity.setMode(HttpMultipartMode.BROWSER_COMPATIBLE);
        entity.addTextBody("json", data);
        int attachcount = 0;
        for (Attachment attach : attachments) {
            ContentType type = ContentType.create(attach.type.replace(" ", ""));
            ContentBody file = new ByteArrayBody((attach.encrContent == null) ? attach.content : attach.encrContent, type, attach.filename);
            entity.addPart("file_" + attachcount++, file);
        }
        HttpPost postRequest = new HttpPost("http" + (Config.backend_ssl ? "s" : "") + "://" + Config.getOxBackend() + ":" + Config.ox_backend_port + Config.OXBackendPath + params);
        postRequest.setEntity(entity.build());
        CloseableHttpResponse response = null;
        if (httpContext == null) {// If login in by cookie, add header and execute
            postRequest.addHeader("cookie", cookie.getCookieHeader());
            postRequest.setHeader("User-Agent", useragent);
            response = httpClient.execute(postRequest);
        } else { // If logging into by previous login httpcontext, then use that
            response = httpClient.execute(postRequest, httpContext);
        }
        BufferedReader reader = new BufferedReader(new InputStreamReader(response.getEntity().getContent(), "UTF-8"));
        String result = reader.readLine();
        reader.close();
        response.close();
        postRequest.releaseConnection();
        return (result);

    }

    /**
     * Multipart post for email
     * 
     * @param data
     * @param params
     * @param attachments
     * @return
     * @throws ClientProtocolException
     * @throws IOException
     */
    public JsonObject multipartPost(String data, String params, ArrayList<Attachment> attachments) throws ClientProtocolException, IOException {

        long start = System.currentTimeMillis();
        if (httpClient == null) {
            // httpClient = new DefaultHttpClient();
            httpClient = OxDbConn.httpClient;
        }
        MultipartEntityBuilder entity = MultipartEntityBuilder.create();
        entity.setMode(HttpMultipartMode.BROWSER_COMPATIBLE);
        entity.addTextBody("json_0", data, ContentType.APPLICATION_JSON);
        entity.setCharset(Charset.forName("UTF-8"));
        int attachcount = 0;
        if (attachments != null) {
            for (Attachment attach : attachments) {
                ContentType type = ContentType.create(attach.type);
                ContentBody file = new ByteArrayBody(attach.encrContent, type, attach.filename);
                // ContentBody file = new ByteArrayBody(attach.encrContent, attach.type, attach.filename);
                entity.addPart("file_" + attachcount++, file);
            }
        }
        HttpPost postRequest = new HttpPost("http" + (Config.backend_ssl ? "s" : "") + "://" + Config.getOxBackend() + ":" + Config.ox_backend_port + Config.OXBackendPath + params);
        postRequest.setEntity(entity.build());
        CloseableHttpResponse response = null;
        if (httpContext == null) {// If login in by cookie, add header and execute
            postRequest.addHeader("cookie", cookie.getCookieHeader());
            postRequest.setHeader("User-Agent", useragent);
            response = httpClient.execute(postRequest);
        } else { // If logging into by previous login httpcontext, then use that
            response = httpClient.execute(postRequest, httpContext);
        }
        BufferedReader reader = new BufferedReader(new InputStreamReader(response.getEntity().getContent(), "UTF-8"));
        String json = reader.readLine();
        reader.close();
        response.close();
        postRequest.releaseConnection();
        if (json.contains("{")) {
            int i = json.indexOf("{");
            int j = json.lastIndexOf("}");
            json = json.substring(i, j + 1);
        }
        try {
            JsonParser parser = new JsonParser();
            JsonObject result = (JsonObject) parser.parse(json);
            logger.info("Sent to OX at " + start);
            return (result);
        } catch (Exception ex) {
            logger.error("Error working on multipart post", ex, json);
            return (null);
        }

    }

    public JsonObject sendMail(JsonObject mail, ArrayList<Attachment> attachments) {
        String params = "mail?action=new";
        params += "&session=" + session;
        Gson gson = new GsonBuilder().disableHtmlEscaping().create();

        try {

            String data = gson.toJson(mail.getAsJsonObject("data"));
            return (multipartPost(data, params, attachments));
        } catch (ClientProtocolException e) {
            logger.error("Error sending mail", e);

        } catch (IOException e) {
        	logger.error("Error sending mail", e);
        }
        return (null);
    }

    public JsonObject saveSentMail(JsonObject mail, ArrayList<Attachment> attachments, String sentfolder) {
	        String params = "mail?action=new";
	        try {
				params += "&folder=" + URLEncoder.encode(sentfolder, "UTF-8") + "&session=" + session;
			} catch (UnsupportedEncodingException e1) {
				logger.error("Impossible, UTF-8 encoding failed");
			}
	        Gson gson = new Gson();
	        try {
	            return (multipartPost(gson.toJson(mail.getAsJsonObject("data")), params, attachments));
	        } catch (ClientProtocolException e) {
	        	logger.error("Error saving sent mail", e);
	        } catch (IOException e) {
	        	logger.error("IO error saving sent mail", e);
	        }
        return (null);
    }
    
    public JsonObject saveDraftMail(JsonObject mail, ArrayList<Attachment> attachments) {
        String params = "mail?action=new&session=" + session;
        Gson gson = new Gson();
        try {
            return (multipartPost(gson.toJson(mail.getAsJsonObject("data")), params, attachments));
        } catch (ClientProtocolException e) {
        	logger.error("Error saving sent mail", e);
        } catch (IOException e) {
        	logger.error("IO error saving sent mail", e);
        }
    return (null);
}

    public JsonObject saveInBox(JsonObject mail, ArrayList<Attachment> attachments) {
        String params = "mail?action=new";
        params += "&folder=default0/INBOX&session=" + session;
        Gson gson = new Gson();

        try {
            return (multipartPost(gson.toJson(mail.getAsJsonObject("data")), params, attachments));
        } catch (ClientProtocolException e) {
        	logger.error("Error saving inbox", e);
        } catch (IOException e) {
        	logger.error("Error saving inbox", e);
        }
        return (null);
    }


    public EncryptedObj getAttachment(int id, int attachment, String folder) {
        long start = System.currentTimeMillis();
        String params = "mail?action=attachment" + "&folder=" + URLEncoder.encode(folder) + "&id=" + id + "&attachment=" + attachment;
        try {
            CloseableHttpResponse response = getResponse(params);
            HttpEntity ent = response.getEntity();
            InputStream in = ent.getContent();
            EncryptedObj item = new EncryptedObj(in);
            item.content_type = ent.getContentType().getValue().trim();
            in.close();
            response.close();
            logger.info("Retrieved ox item at " + start);
            return (item);

        } catch (IllegalStateException e) {
        	logger.error("Error getting attachment", e);
        } catch (IOException e) {
        	logger.error("IO error getting attachment", e);
        }
        return (null);
    }
    
    public Attachment getPlainAttachment (int id, String attachment, String folder) {
    	long start = System.currentTimeMillis();
        String params = "mail?action=attachment" + "&folder=" + URLEncoder.encode(folder) + "&id=" + id + "&attachment=" + attachment;
        try {
            CloseableHttpResponse response = getResponse(params);
            HttpEntity ent = response.getEntity();
            Attachment att = new Attachment();
            att.content = EntityUtils.toByteArray(ent);
            logger.info("Retrieved ox item at " + start);
            return (att);

        } catch (IllegalStateException e) {
        	logger.error("Error getting attachment", e);
        } catch (IOException e) {
        	logger.error("IO error getting attachment", e);
        }
        return (null);
    }

    public String getMime (String ref) {
    	int i = ref.lastIndexOf(("/"));
    	if (i < -1) return (null);
    	String folder = ref.substring(0, i);
    	String id = ref.substring(i+1);
    	int emailid;
    	try {
    		emailid = Integer.parseInt(id);
    	} catch (Exception ex) {
    		logger.error("Error parsing email id from msgref " + ref);
    		return (null);
    	}
    	return getMime(emailid, folder);
    }
    
    public String getMime(int id, String folder) {
        String params = "mail?action=get" + "&folder=" + URLEncoder.encode(folder) + "&id=" + id + "&src=1&save=1";
        try {
            CloseableHttpResponse response = getResponse(params);
            HttpEntity ent = response.getEntity();
            String data = EntityUtils.toString(ent);
            response.close();
            return (data);
        } catch (Exception ex) {
            logger.error("Error getting MIME type", ex);
            return ("");
        }
    }

    public String getMail(int id, String folder) {
        String params = "mail?action=get" + "&folder=" + URLEncoder.encode(folder) + "&id=" + id;
        try {
            CloseableHttpResponse response = getResponse(params);
            HttpEntity ent = response.getEntity();
            String data = EntityUtils.toString(ent);
            response.close();
            return (data);
        } catch (Exception ex) {
            logger.error("Error getting mail " + id + " from folder " + folder, ex);
            return ("");
        }
        
    }
    
    public int getContextId () {
    	String params = "config/context_id?session=" + session;
		try {
			CloseableHttpResponse response = getResponse(params);
			JsonObject data = Core.getJson(response);
            return (data.get("data").getAsInt());
		} catch (Exception ex) {
			logger.error("Error getting contextId", ex);
            return (0);
		}
    }

    public String getFolder(int id) {
        String params = "folders?action=get" + "&id=" + id;
        try {
            CloseableHttpResponse response = getResponse(params);
            HttpEntity ent = response.getEntity();
            String data = EntityUtils.toString(ent);
            response.close();
            return (data);
        } catch (Exception ex) {
            logger.error("Error getting folder " + id, ex);
            return ("");
        }
    }

    public String updateFolder(int id, String json) {
        String params = "folders?action=update" + "&id=" + id;
        try {
            CloseableHttpResponse response = putRequest(params, json);
            HttpEntity ent = response.getEntity();
            String data = EntityUtils.toString(ent);
            response.close();
            return (data);
        } catch (Exception ex) {
            logger.error("Error updating folder " + id, ex);
            return ("");
        }
    }

    public String getFolders() {
        String params = "folders?action=list&parent=1&tree=1&all=0&altNames=true&timezone=UTC&columns=1%2C2%2C3%2C4%2C5%2C6%2C20%2C23%2C300%2C301%2C302%2C304%2C305%2C306%2C307%2C308%2C309%2C310%2C311%2C312%2C313%2C314%2C315%2C316%2C317%2C3010%2C3020%2C3030";
        try {
            CloseableHttpResponse response = getResponse(params);
            HttpEntity ent = response.getEntity();
            String data = EntityUtils.toString(ent);
            response.close();
            return (data);
        } catch (Exception e) {
            logger.error("Error listing folders", e);
        }
        return ("");
    }

    public void getFileList(int folder) {
        String params = "files?action=all&folder=" + folder + "&session=" + session + "&columns=1,711";
        try {
            CloseableHttpResponse response = getResponse(params);
            HttpEntity ent = response.getEntity();
            String data = EntityUtils.toString(ent);
            response.close();
        } catch (Exception ex) {
            logger.error("Error listing all item in folder " + folder, ex);
        }
    }

    public void getVersions(int item) {
        String params = "files?action=versions&id=" + item + "&session=" + session + "&columns=1";
        try {
            CloseableHttpResponse response = getResponse(params);
            HttpEntity ent = response.getEntity();
            String data = EntityUtils.toString(ent);
            response.close();
        } catch (Exception ex) {
            logger.error("Error getting versions for item " + item, ex);
        }
    }

    public boolean verifyLogin() {
        String folders = getFolders();
        JsonParser parser = new JsonParser();
        JsonObject json = (JsonObject) parser.parse(folders);
        if (json.get("data") != null) {
            return (true);
        } else
            return (false);
    }

    public String updateFile(Attachment file, String data, int id, int folder) throws UnsupportedEncodingException {

        ArrayList<Attachment> attachments = new ArrayList<Attachment>();
        attachments.add(file);
        Date date = new Date();
        String params = "files?action=update&id=" + id + "&session=" + session + "&timestamp=" + date.getTime() + "&filename=" + URLEncoder.encode(
            file.filename,
            "UTF-8");
        try {
            String response = filePost(data, params, attachments);
            logger.debug(response);
            return (response);
        } catch (ClientProtocolException e) {
        	logger.error("Error updating file " + id + " in folder " + folder, e);
        } catch (IOException e) {
        	logger.error("IO error updating file " + id + " in folder " + folder, e);
        }
        return ("");
    }

    public JsonObject getInfoItemJson (int id, int folder) {
    	String json = getInfoItemJson (id, folder, -1);
    	JsonParser parser = new JsonParser();
        JsonObject result = (JsonObject) parser.parse(json);
        return(result);
    }
    
    public String getInfoItemJson(int id, int folder, int version) {
    	String params = "files?action=get&id=" + id + "&folder=" + folder + "&version=" + version + "&session=" + session;
    	if (version == -1) {
    		params = "files?action=get&id=" + id + "&folder=" + folder + "&session=" + session;
    	}
        try {
            CloseableHttpResponse response = getResponse(params);
            HttpEntity ent = response.getEntity();
            String data = EntityUtils.toString(ent);
            response.close();
            return (data);
        } catch (Exception ex) {
            logger.error("Error getting item " + id + " from folder " + folder + "in version " + version, ex);
        }
        return (null);
    }

    public byte[] getPlainTextInfoStore(int id, int folder, int version) {
        String params = "files?action=document&id=" + id + "&folder=" + folder + "&version=" + version + "&session=" + session;
        try {
            CloseableHttpResponse response = getResponse(params);
            HttpEntity ent = response.getEntity();
            byte[] info = EntityUtils.toByteArray(ent);
            response.close();
            return (info);
        } catch (Exception e) {
            logger.error("error getting item " + id + " from folder " + folder + " in version " + version + " in plaintext", e);
        }
        return (null);
    }

    /**
     * Retrive file item from infostore
     * 
     * @param id ID of the item
     * @param folder folder id
     * @param version version id
     * @return
     */
    public EncryptedObj getInfoStore(int id, int folder, int version) {
        String params = "files?action=document&id=" + id + "&folder=" + folder + "&version=" + version + "&session=" + session;
        if (version == -1) {
        	params = "files?action=document&id=" + id + "&folder=" + folder + "&session=" + session;
        }
        long start = System.currentTimeMillis();
        try {
            CloseableHttpResponse response = getResponse(params);
            HttpEntity ent = response.getEntity();
            InputStream in = ent.getContent();
            EncryptedObj obj = new EncryptedObj(in);
            obj.content_type = ent.getContentType().toString().trim();
            if (obj.content_type.contains(":")) {
            	obj.content_type = obj.content_type.substring(obj.content_type.indexOf(":") + 1).trim();
            }
            response.close();
            logger.info("Retrieved infostore at " + start);
            return (obj);
        } catch (Exception e) {
            logger.error("Error getting infostore item " + id + " in folder " + folder + "in version " + version, e);
        }
        return (null);
    }

    /**
     * Retrieve non-encrypted/encrypted infostore object as attachment
     * 
     * @param id
     * @param folder
     * @param version
     * @return
     */
    public Attachment getInfoStoreAttach(int id, int folder, int version) {
        String params = "files?action=document&id=" + id + "&folder=" + folder + ((version >= 0) ? ("&version=" + version) : "") + "&session=" + session;
        try {
            Attachment attach = new Attachment();
            CloseableHttpResponse response = getResponse(params);
            HttpEntity ent = response.getEntity();
            attach.type = ent.getContentType().toString();
            int i = attach.type.indexOf(":");
            if (i > 0) attach.type = attach.type.substring(i + 1).trim();
            attach.content = EntityUtils.toByteArray(ent);
            response.close();
            return (attach);
        } catch (Exception ex) {
            logger.error("Error getting infostore attachment " + id + " from folder " + folder + " in version " + version, ex);
        }
        return (null);
    }

    public Attachment getInlineImage (String url) {
    	try {
    		url = java.net.URLDecoder.decode(url, "UTF-8").replace("&amp;","&");
    		if (url.contains("file?action") || url.contains("image/mail")) {
    			int index = 0;
    			if (url.contains("file?action")) index = url.indexOf("file?action");
    			if (url.contains("image/mail")) index = url.indexOf("image/mail");
	        	String params = url.substring(index);
	            Attachment attach = new Attachment();
	            CloseableHttpResponse response = getResponse(params);
	            HttpEntity ent = response.getEntity();
	            attach.type = ent.getContentType().toString();
	            if (attach.type.contains(":")) {
	            	attach.type = attach.type.substring(attach.type.indexOf(":") + 1).trim();
	            }
	            org.apache.http.Header[] h = response.getAllHeaders();
	            for (int i = 0; i < h.length; i++) {
	         //       System.out.println(h[i].getName() + "  " + h[i].getValue());
	                if (h[i].getValue().contains("filename")) {
	                	String filename = h[i].getValue();
	                	filename = filename.substring(filename.indexOf("filename="));
	                	attach.filename = filename.replace("filename=", "").replace("\"", "");
	                }
	            }
	            attach.content = EntityUtils.toByteArray(ent);
	            EntityUtils.consume(ent);
	            response.close();
	            return (attach);
    		}
        } catch (Exception ex) {
            logger.error("Error getting inline image from " + url, ex);
        }
        return (null);
    }
    public String getCapabilities() {
        String params = "capabilities?action=all";
        String dat = null;
        try {
            CloseableHttpResponse response = getResponse(params);
            HttpEntity ent = response.getEntity();
            dat = EntityUtils.toString(ent);
            response.close();
        } catch (Exception e) {
            logger.error("Error retrieving all capabilities", e);
        }
        return (dat);
    }
    
    public String getConfig (String module) {
    	String params = "config/" + module + "?session=" + session;
        String dat = null;
        try {
            CloseableHttpResponse response = getResponse(params);
            HttpEntity ent = response.getEntity();
            dat = EntityUtils.toString(ent);
            response.close();
        } catch (Exception e) {
            logger.error("Error retrieving configuration", e);
        }
        return (dat);
    }
    
    public void deleteItem(int id, int folder) {
        String params = "files?action=delete&hardDelete=true&timestamp=2116800000000";
        try {
            String data = "[{\"id\":\"" + id + "\",\"folder\":\"" + folder + "\"}]";
            CloseableHttpResponse response = putRequest(params, data);
            HttpEntity ent = response.getEntity();
            String dat = EntityUtils.toString(ent);
            response.close();
        } catch (Exception e) {
            logger.error("Error deleting item " + id + " in folder " + folder, e);
        }
    }
    
    public void deleteDraft(String msgref) {
    	int i = msgref.lastIndexOf("/");
    	if (i>0) {
    		try {
	    		String defaultDraft = getConfig("mail/folder/drafts");
	            JsonParser parser = new JsonParser();
	            JsonObject draft = (JsonObject) parser.parse(defaultDraft);
	    		if (!msgref.startsWith(draft.get("data").getAsString())) return;  // If not from default draft folder, then exit (reply or forward)
	    		String id = msgref.substring(i + 1);
	    		String folder = msgref.substring(0, i);
	    		
	    		String params = "mail?action=delete&timestamp=2116800000000";
	    		JsonObject data = new JsonObject();
	    		data.addProperty("folder", folder);
	    		data.addProperty("id", id);
	    		try {
					putRequest(params, "[" + data.toString() + "]");
				} catch (IllegalStateException | IOException e) {
					logger.error("Problem deleting from draft", e);
				}
    		} catch (Exception ex) {
    			logger.error("Problem getting draft folder ", ex);
    			return;
    		}
    	}
    }
    
    private String checkCR (String data) {
    	if (data == null) return (data);
    	try {
    		return (data.replace("\r", "").replace("\n", ""));
    	} catch (Exception ex) {
    		logger.error("Error checking header", ex);
    		return(data);
    	}
    }

}
