/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2014 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.encr;

import org.apache.commons.codec.binary.Base64;
import org.bouncycastle.bcpg.ArmoredOutputStream;
import org.bouncycastle.openpgp.PGPKeyRingGenerator;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.security.KeyFactory;
import java.security.NoSuchAlgorithmException;
import java.security.PublicKey;
import java.security.PrivateKey;
import java.security.interfaces.RSAPublicKey;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.sql.Date;

import org.bouncycastle.openpgp.PGPException;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.PGPSecretKey;
import org.bouncycastle.openpgp.PGPSecretKeyRing;
import org.bouncycastle.openpgp.operator.bc.BcKeyFingerprintCalculator;
import org.bouncycastle.openpgp.PGPUtil;
import org.bouncycastle.openpgp.PGPObjectFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.openexchange.guard.config.Config;
import com.openexchange.guard.database.RecipKey;
import com.openexchange.guard.exceptions.BadPasswordException;

/**
 * Manage PGP and RSA keys. Call create and encrypt of keys
 * 
 * @author greg
 */
public class GuardKeys {
	private static Logger logger = LoggerFactory.getLogger(GuardKeys.class);
	
    PGPSecretKeyRing keyring;

    public PGPPublicKeyRing pubring;

    PublicKey pubkey;

    String privkey;

    String salt;

    public boolean passwordNeeded = false;

    public int userid;

    public int contextid;

    public Date lastup;

    public String misc;

    public String settings;

    public String email;
    
    public String question;
    
    public String answer;

    public GuardKeys() {

    }

    public GuardKeys(RecipKey key) {
        userid = key.userid;
        contextid = key.cid;
        pubkey = key.pubkey;
        email = key.email;
    }

    /**
     * @param gen PGPKeyRingGenerator
     * @param pub PublicKey
     * @param priv PrivateKey
     * @param Password Password for the private keys
     * @param Salt Salt used for the passwords.
     */
    public GuardKeys(PGPKeyRingGenerator gen, PrivateKey priv, PublicKey pub, String password, String salted) {
        keyring = gen.generateSecretKeyRing();
        pubring = gen.generatePublicKeyRing();
        pubkey = pub;
        salt = salted;
        setPrivate(priv, password);
    }

    public void setPGPSecretKeyRing(PGPSecretKeyRing ring) {
        keyring = ring;
    }

    /**
     * Set the private RSA key, need password and salt to encrypt
     * 
     * @param priv
     * @param password
     * @param salt
     * @return
     */
    public Boolean setPrivate(PrivateKey priv, String password) {
        EncrLib crypto = new EncrLib();
        privkey = crypto.encryptAES(Base64.encodeBase64String(priv.getEncoded()), EncrLib.getSHA(password, salt), salt, Config.AESKeyLen);
        if (privkey == "") { // Return false if failed to save
            return (false);
        } else
            return (true);

    }

    /**
     * Set the already encoded private RSA
     * 
     * @param priv
     */
    public void setEncodedPrivate(String priv) {
        privkey = priv;
    }

    public PrivateKey getDecodedPrivate(String password) throws BadPasswordException, UnsupportedEncodingException {
        return (getDecodedPrivate(password, true));
    }

    /**
     * Retrieve the Private RSA key using password and salt
     * 
     * @param password
     * @param logbad Log Bad passwords, false if just testing
     * @return
     * @throws BadPasswordException 
     * @throws UnsupportedEncodingException 
     */
    public PrivateKey getDecodedPrivate(String password, boolean logbad) throws BadPasswordException, UnsupportedEncodingException {
    	
            EncrLib crypto = new EncrLib();
            byte[] decoded = crypto.decryptAES(privkey, EncrLib.getSHA(password, salt), salt, Config.AESKeyLen, logbad).getBytes("UTF-8");
            if (decoded.length == 0)
            	throw new BadPasswordException("Bad Password");
            // create a keyfactory - use whichever algorithm and provider
            KeyFactory kf;
			try {
				kf = KeyFactory.getInstance("RSA");
			} catch (NoSuchAlgorithmException e) {
				logger.error("Unable decode private", e);
	            return (null);
			}
            // for private keys use PKCS8EncodedKeySpec; for public keys use X509EncodedKeySpec
            PKCS8EncodedKeySpec ks = new PKCS8EncodedKeySpec(Base64.decodeBase64(decoded));
            PrivateKey priv;
			try {
				priv = kf.generatePrivate(ks);
			} catch (InvalidKeySpecException e) {
				logger.error("Unable decode private", e);
	            return (null);
			}
            return (priv);
    }

    /**
     * Retrieve the Private RSA key using password and salt
     * 
     * @param password
     * @param salt
     * @return
     */
    public PrivateKey getDecodedPrivateWithRecovery(String hash) {
        try {
            EncrLib crypto = new EncrLib();
            byte[] decoded = crypto.decryptAES(privkey, hash, salt, Config.AESKeyLen).getBytes("UTF-8");
            // create a keyfactory - use whichever algorithm and provider
            KeyFactory kf = KeyFactory.getInstance("RSA");
            // for private keys use PKCS8EncodedKeySpec; for public keys use X509EncodedKeySpec
            PKCS8EncodedKeySpec ks = new PKCS8EncodedKeySpec(Base64.decodeBase64(decoded));
            PrivateKey priv = kf.generatePrivate(ks);
            return (priv);
        } catch (Exception ex) {
            logger.info("Unable decode private");
            return (null);
        }
    }

    /**
     * Set the RSA public key from stored string
     * 
     * @param data
     * @throws InvalidKeySpecException
     * @throws NoSuchAlgorithmException
     */
    public void setPublicKeyFrom64String(String data) throws InvalidKeySpecException, NoSuchAlgorithmException {
        KeyFactory kf = KeyFactory.getInstance("RSA");
        // for private keys use PKCS8EncodedKeySpec; for public keys use X509EncodedKeySpec
        X509EncodedKeySpec ks = new X509EncodedKeySpec(Base64.decodeBase64(data));
        pubkey = kf.generatePublic(ks);

    }

    /**
     * Return public RSA key
     * 
     * @return
     */
    public PublicKey getPublic() {
        return (pubkey);
    }

    /**
     * Get Base64 representation of Public key
     * 
     * @return
     */
    public String getEncodedPublic() {
        return (Base64.encodeBase64String(pubkey.getEncoded()));
    }

    /**
     * Get the Base64 Encoded and encrypted Private key
     * 
     * @return
     */
    public String getEncodedPrivate() {
        return (privkey);
    }

    /**
     * Set RSA Public Key
     * 
     * @param pub
     */
    public void setPublic(PublicKey pub) {
        pubkey = pub;
    }

    /**
     * Return the Public Key
     * 
     * @return
     */
    public PGPPublicKey getPGPPublicKey() {
        Iterator keys = keyring.getPublicKeys();
        PGPPublicKey key = null;
        while (keys.hasNext()) {
            key = (PGPPublicKey) keys.next();
            if (key.isEncryptionKey() && !(key.isMasterKey())) {
                return key;
            }
        }
        return (keyring.getPublicKey());

    }

    /**
     * Get Base64 Encoded String
     * 
     * @return
     * @throws IOException
     */
    public String getEncodedPGPPublic() throws IOException {
        return (Base64.encodeBase64String(getPGPPublicKey().getEncoded()));
    }

    /**
     * Return the secret Key
     * 
     * @return
     */
    public PGPSecretKey getPGPSecretKey() {
        return (keyring.getSecretKey());
    }

    public PGPSecretKeyRing getPGPSecretKeyRing() {
        return (keyring);
    }

    /**
     * Get Base64 Encoded Secret Keys
     * 
     * @return
     * @throws IOException
     */
    public String getEncodedPGPSecret() throws IOException {
        return (Base64.encodeBase64String(keyring.getEncoded()));
    }

    /**
     * Set PGPSecret info from base64 string
     * 
     * @param keystring
     * @return
     * @throws IOException
     * @throws PGPException
     */
    public void setPGPSecretFromString(String keystring) throws IOException, PGPException {

        keyring = new PGPSecretKeyRing(Base64.decodeBase64(keystring), new BcKeyFingerprintCalculator());

    }

    /**
     * Get public key from base64 String
     * 
     * @param keystring Base64String
     * @return
     * @throws IOException
     */
    public PGPPublicKey getPGPPublicKeyFromString(String keystring) throws IOException {
        PGPPublicKeyRing pkr = new PGPPublicKeyRing(Base64.decodeBase64(keystring), new BcKeyFingerprintCalculator());
        return (pkr.getPublicKey());
    }

    public void setSalt(String salted) {
        salt = salted;
    }

    public String getSalt() {
        return (salt);

    }

    /**
     * Get encryption public key from ASC data
     * 
     * @param ascdata
     * @return
     * @throws IOException
     */
    public PGPPublicKey getPGPKeyfromASC(String ascdata) throws IOException {
        return (getEncryptionKey(getKeyring(ascdata)));
    }

    private PGPPublicKeyRing getKeyring(String AscData) throws IOException {
        // PGPUtil.getDecoderStream() will detect ASCII-armor automatically and decode it,
        // the PGPObject factory then knows how to read all the data in the encoded stream
        PGPObjectFactory factory = new PGPObjectFactory(PGPUtil.getDecoderStream(new java.io.ByteArrayInputStream(AscData.getBytes("UTF-8"))));
        // these files should really just have one object in them,
        // and that object should be a PGPPublicKeyRing.
        Object o = factory.nextObject();
        if (o instanceof PGPPublicKeyRing) {
            return (PGPPublicKeyRing) o;
        }
        throw new IllegalArgumentException("Input text does not contain a PGP Public Key");
    }

    /**
     * Get the first encyption key off the given keyring.
     */
    private PGPPublicKey getEncryptionKey(PGPPublicKeyRing keyRing) {
        if (keyRing == null)
            return null;

        // iterate over the keys on the ring, look for one
        // which is suitable for encryption.
        Iterator keys = pubring.getPublicKeys();
        PGPPublicKey key = null;
        while (keys.hasNext()) {
            key = (PGPPublicKey) keys.next();
            if (key.isEncryptionKey()) {
                if (!key.isMasterKey()) {
                    return key;
                }
            }
        }
        return null;
    }

    /**
     * Export PGP Public keys in standard armored format
     * 
     * @return
     * @throws IOException
     */
    public String exportPgpPublicKey() throws IOException {
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        ArmoredOutputStream arm = new ArmoredOutputStream(out);
        // PGPPublicKeyRing pbkeyring = keyring.get
        // arm.write(pbkeyring.getEncoded());
        Iterator<PGPPublicKey> keys = pubring.getPublicKeys();
        ByteArrayOutputStream bout = new ByteArrayOutputStream();
        List<PGPPublicKey> pkeys = new ArrayList<PGPPublicKey>();
        while (keys.hasNext()) {
            PGPPublicKey k = keys.next();
            k.encode(bout);
        }
        arm.write(bout.toByteArray());
        arm.close();
        return (new String(out.toByteArray()));
    }

    public String exportPGPPublicKeyRing() throws IOException {
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        ArmoredOutputStream arm = new ArmoredOutputStream(out);
        arm.write(pubring.getEncoded());
        arm.close();
        return (new String(out.toByteArray()));
    }

    public void setPGPPublicKeyRing(String data) {
        try {
            pubring = getKeyring(data);
        } catch (IOException e) {
            logger.error("Failed to store in public key ring", e);
        }
    }

    public int getKeyLen() {
        RSAPublicKey rsapub = (RSAPublicKey) pubkey;
        return (rsapub.getModulus().bitLength());
    }

}
