/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2014 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.database;

import java.sql.Connection;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.openexchange.guard.cacheitems.Cache;
import com.openexchange.guard.config.Config;
import com.openexchange.guard.dbpool.ContextDBConnection;
import com.openexchange.guard.encr.EncrLib;
import com.openexchange.guard.encr.PgpKeys;
import com.openexchange.guard.encr.RSAGen;
import com.openexchange.guard.encr.GuardKeys;

public class Keymanager {
	private static Logger logger = LoggerFactory.getLogger(Keymanager.class);

//    public GuardKeys createMasterKeys(int id, int cid, String email, String name, String language) {
 //       return (createMasterKeys(id, cid, email, name, "", language));
 //   }
	public GuardKeys createMasterKeys(int id, int cid, String email, String name, String password, String language) {
		return(createMasterKeys (id, cid, email, name, password, language, false));
	}
    public GuardKeys createMasterKeys(int id, int cid, String email, String name, String password, String language, boolean userCreated) {
        // PgpKeys pgp = new PgpKeys();
        RSAGen gen = new RSAGen();
        try {
            // oxGuardKeys keys = pgp.generateKeys(name, email, password);
            GuardKeys keys = gen.generateKeyPair(name, email, password);
            if (keys != null) {
                Access acc = new Access();
                keys.email = email;
                keys.userid = acc.storeKeys(id, cid, email, keys, createRecovery(keys, password), language, userCreated);
                if (keys.userid == 0) return(null); //Failed to create keys
            } else {
            	return(createMasterKeysNC(id, cid, email, name, password, language, userCreated));// Try failover to generated realtime
            }
            return (keys);
        } catch (Exception e) {
            logger.error("Error create keys", e);
            if (e.getMessage().contains("Duplicate")) {
        		logger.error("Corrupt key lookup?  Check table oxguard.og_email");
        	}
            return (null);
        }
    }
    
    public GuardKeys createMasterKeysNC(int id, int cid, String email, String name, String password, String language) {
    	return (createMasterKeysNC (id, cid, email, name, password, language, false));
    }

    public GuardKeys createMasterKeysNC(int id, int cid, String email, String name, String password, String language, boolean userCreated) {
        logger.debug("Not using cache, Creating master keys for " + email);
    	PgpKeys pgp = new PgpKeys();
    	Access acc = new Access();
        try {
            GuardKeys keys = pgp.generateKeys(name, email, password);
            if (keys != null) {
                
                keys.userid = acc.storeKeys(id, cid, email, keys, createRecovery(keys, password), language, userCreated);
                if (keys.userid == 0) return(null); //Failed to create keys
            }
            return (keys);
        } catch (Exception e) {
        	if (e.getMessage().contains("Duplicate entry")) {
        		try {
        			logger.debug("Key already exists, using existing");
					return(acc.getKeys(id, cid));
				} catch (Exception e1) {
					logger.error("Unable to get existing", e1);
					return(null);
				}
        	}
            logger.error("Error creating master keys for " + email, e);
            return (null);
        }
    }


    public String createRecovery(GuardKeys keys, String password) throws Exception {
        if (password.equals(""))
            return ("-1");
        if (Config.no_recovery)
            return (""); // If selected for no recovery, then just return blank.
        password = EncrLib.getSHA(password, keys.getSalt());
        String recovery = "";
        EncrLib encr = new EncrLib();
        String command = "SELECT RSAPublic FROM og_KeyTable WHERE id = -1 AND cid = 0";
        DbCommand com = new DbCommand(command);
        DbQuery db = new DbQuery();
        db.readOG(com);
        if (db.next()) {
            String pub = db.rs.getString("RSAPublic");
            GuardKeys masterKeys = new GuardKeys();
            masterKeys.setPublicKeyFrom64String(pub);
            String encrPass = encr.encryptAES(
                password,
                Config.rpass + Integer.toString(keys.getEncodedPublic().hashCode()),
                keys.getSalt(),
                Config.AESKeyLen);
            recovery = encr.encryptRSA(encrPass, masterKeys.getPublic());

        }
        db.close();
        if (recovery.equals("")) {
        	if (keys.userid > 0) {  // make sure not for system email
        		logger.error("Unable to create recovery for " + keys.email);
        		throw (new Exception ("Unable to create recovery"));
        	}
        }
        return (recovery);
    }

    public String recoverPassHash(int id, int cid) {
        String hash = "";
        ContextDBConnection conn = null;
        try {
            Access acc = new Access();
            String command = "SELECT Recovery, Salt, RSAPublic FROM og_KeyTable WHERE id = ? AND cid = ?";
            DbCommand com = new DbCommand(command);
            com.addVariables(id);
            com.addVariables(cid);
            DbQuery db = new DbQuery();
            db.read(com, id, cid);
            if (db.next()) {
                String recovery = db.rs.getString("Recovery");
                if (!recovery.equals("")) {
                	String salt = db.rs.getString("Salt");
                    String rsapub = db.rs.getString("RSAPublic");
                    GuardKeys masterKeys = acc.getKeys(-1, 0);
                    EncrLib encr = new EncrLib();
                    String encrhash = encr.decryptRSA(recovery, masterKeys.getDecodedPrivate(Config.mpass));
                    if (encrhash == null) {
                    	logger.error("Unable to decode master key.  Possible corrupted data");
                    	throw(new Exception("Bad master key"));
                    }
                    hash = encr.decryptAES(
                        encrhash,
                        Config.rpass + Integer.toString(rsapub.hashCode()),
                        salt,
                        Config.AESKeyLen);
                }
            }
            db.close();
        } catch (Exception ex) {
            logger.error("Error recovering the password hash for " + cid + "/" + id, ex);
        }
        return (hash);
    }

    public String resetPassword(String email) {
        String newpass = EncrLib.randomPass(Config.newPassLength);
        return (resetPassword(email, newpass, true));
    }

    public String resetPassword(String email, String newpass) {
    	return (resetPassword (email, newpass, false));
    }
    /**
     * Resets password for an account with new password, stores new keys, and stores new recovery
     * 
     * @param email
     * @param newpass
     * @return
     */
    public String resetPassword(String email, String newpass, boolean reset) {
        try {
            Access acc = new Access();
            GuardKeys oldkeys = acc.getKeysFromEmail(email);
            String hash = recoverPassHash(oldkeys.userid, oldkeys.contextid);
            if (hash == "") {
                return ("NR");  // Return no recovery
            }

            PgpKeys keygen = new PgpKeys();
            GuardKeys newkeys = keygen.changePasswordWithRecovery(hash, newpass, oldkeys);
            if (newkeys == null) {
                logger.error("Failed to change password");
                return (null);
            }
            String newrecovery = createRecovery(newkeys, newpass);
            acc.updateKeys(oldkeys.userid, oldkeys.contextid, newkeys, newrecovery, reset);
            return (newpass);
        } catch (Exception ex) {
            logger.error("Error resetting password for " + email, ex);
            return (null);
        }

    }
}
