/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2014 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

/*
 * Class for query database through OX HTTP Api
 * 
 */

package com.openexchange.guard.database;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;

import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.client.ClientProtocolException;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.protocol.HTTP;
import org.apache.http.protocol.HttpContext;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.openexchange.guard.config.Config;
import com.openexchange.guard.encr.EncrLib;
import com.openexchange.guard.server.OxDbConn;

public class DbQuery {
	private static Logger logger = LoggerFactory.getLogger(DbQuery.class);

    public JsonObject result = null;

    public OxResultSet rs = null;

    public boolean keepOpen = false;
    
    public int updated = -1;

    String transaction = null;

    int index = -1;
    

    String query_id = "";

    /**
     * Actual send of data to OX Backend
     * 
     * @param data String of JSON data to send
     * @param database The ID of the database
     * @param cid ContextId if pertinent. 0 if not
     * @param write Write or read ony
     * @return
     * @throws Exception
     */
    private JsonObject putRequest(String data, String database, int cid, boolean write) throws Exception {

        String url = "";
        if (database.equals("oxdb") || database.equals("configdb")) {
            url = "http" + (Config.backend_ssl ? "s" : "") + "://" + Config.getOxBackend() + ":" + Config.ox_backend_port + "/preliminary/database/v1/" + database + "/" + ((cid == 0) ? "" : cid + "/") + (write ? "writable" : "readOnly");
        } else {
            url = "http" + (Config.backend_ssl ? "s" : "") + "://" + Config.getOxBackend() + ":" + Config.ox_backend_port + "/preliminary/database/v1/pool/r/" + OGInit.read_map.get(database) + "/w/" + OGInit.write_map.get(database) + "/" + database + "/" + (write ? "writable" : "readOnly");
        }
        if (keepOpen)
            url += "?keepOpen=true";
        if (transaction != null) {
            url = "http" + (Config.backend_ssl ? "s" : "") + "://" + Config.getOxBackend() + ":" + Config.ox_backend_port + "/preliminary/database/v1/transaction/" + transaction;
        }
        return (putRequest(data, url));

    }

    /**
     * Actual send of data to OX Backend
     * 
     * @param data String of JSON data to send
     * @param database The ID of the database
     * @param cid ContextId if pertinent. 0 if not
     * @param write Write or read ony
     * @return
     * @throws Exception
     */
    private JsonObject putOgRequest(String data, boolean write) throws Exception {


        String url = "http" + (Config.backend_ssl ? "s" : "") + "://" + Config.getOxBackend() + ":" + Config.ox_backend_port + "/preliminary/database/v1/pool/r/" + OGInit.read_id + "/w/" + OGInit.write_id + "/oxguard/" + (write ? "writable" : "readOnly");
        if (keepOpen)
            url += "?keepOpen=true";
        if (transaction != null) {
            url = "http" + (Config.backend_ssl ? "s" : "") + "://" + Config.getOxBackend() + ":" + Config.ox_backend_port + "/preliminary/database/v1/transaction/" + transaction;
        }
        return (putRequest (data, url));

    }
    
    private JsonObject putRequest (String data, String url) throws Exception {
    	CloseableHttpClient httpClient = OxDbConn.httpClient;
        HttpContext context = HttpClientContext.create();
        HttpPut putRequest = new HttpPut(url);
        StringEntity entity = new StringEntity(data, "UTF-8");
        entity.setContentType("text/javascript;charset=UTF-8");
        putRequest.addHeader("accept", "application/json");
        putRequest.addHeader("Authorization", "Basic " + Config.getBasicAuthenticationEncoding());
        putRequest.setEntity(entity);
        CloseableHttpResponse response = null;
        try {
        	response = httpClient.execute(putRequest, context);
        } catch (IllegalStateException | org.apache.http.conn.ConnectTimeoutException e) {
        	logger.error("Connection issue with backend.  Trying reset pool" , e);
        	OxDbConn.reset();
        	response = OxDbConn.httpClient.execute(putRequest, context);
        } catch (org.apache.http.conn.HttpHostConnectException ex) {
        	logger.error("Unable to connect to backend.  Connection refused.  ?? Down ??");
        	result = null;
        	return(null);
        }
        if (response.getStatusLine().getStatusCode() != 200) {
            result = null;
            try {
                result = getJson(response);
            } catch (Exception ex) {
            	try {
            		EntityUtils.consume(response.getEntity());
            		putRequest.releaseConnection();
            		response.close();
            	} catch (Exception e2) {
            		logger.error("unable to close http stream after error", e2);
            	}
                throw new Exception("Failed : HTTP error code : " + response.getStatusLine().getStatusCode() + " url: " + url);
            }
            if (hasError()) {
            	logger.error("Failed with url:" + url);
            	logger.error("Status code:" + response.getStatusLine().getStatusCode());
            	logger.error("Reason: " + response.getStatusLine().getReasonPhrase());
            	logger.error("Failed with query:" + data);
            	EntityUtils.consume(response.getEntity());
            	response.close();
            	putRequest.releaseConnection();
                throw new Exception(getError());
            }
            response.close();
            putRequest.releaseConnection();
            return (result);

        }
        result = getJson(response);
        checkUpdated();
        EntityUtils.consume(response.getEntity());
        response.close();
        putRequest.releaseConnection();
        index = -1;
        if (hasError()) {
            logger.error(getError());
            throw new RuntimeException(getError());
        }
        return (result);
    }

    private void checkUpdated () {
    	if (result == null) return;
    	if (result.has("results")) {
    		JsonObject sub = result.get("results").getAsJsonObject();
    		if (sub.has(query_id)) {
    			JsonObject qr = sub.get(query_id).getAsJsonObject();
    			if (qr.has("updated")) {
    				updated = qr.get("updated").getAsInt();
    			}
    		}
    	}
    }
    /**
     * Format the query into proper JSON for backend
     * 
     * @param com
     * @return
     */
    private String createQuery(DbCommand com) {
        query_id = EncrLib.getUUID();
        StringBuilder sb = new StringBuilder();
        sb.append("{");
        sb.append("\"" + query_id + "\": {");
        sb.append("\"query\": \"" + com.command + "\",");
        sb.append("\"params\" : " + com.getVariables());
        sb.append("} }");
        return (sb.toString());
    }

    private String createComplexQuery(MultiCommand com) {
        StringBuilder sb = new StringBuilder();
        sb.append("{");
        for (int i = 0; i < com.count(); i++) {
            DbCommand command = com.getCommand(i);
            String id = EncrLib.getUUID();
            sb.append("\"" + id + "\": {");
            sb.append("\"query\": \"" + command.command + "\",");
            sb.append("\"params\" : " + command.getVariables());
            if (command.resultSet) {
                sb.append(", \"resultSet\" : true");
                query_id = id;
            }
            sb.append("}");
            if (i < (com.count() - 1))
                sb.append(',');
        }

        sb.append("}");
        return (sb.toString());
    }

    /**
     * Format response into a JSON object
     * 
     * @param response
     * @return
     * @throws UnsupportedEncodingException
     * @throws IllegalStateException
     * @throws IOException
     */
    private JsonObject getJson(HttpResponse response) throws UnsupportedEncodingException, IllegalStateException, IOException {
        BufferedReader reader = new BufferedReader(new InputStreamReader(response.getEntity().getContent(), "UTF-8"));
        JsonParser parser = new JsonParser();
        HttpEntity entity = response.getEntity();
        String resp = EntityUtils.toString(entity);
        if (resp.equals("")) {
        	JsonObject result = new JsonObject();
        	result.addProperty("error", "Blank response from OX Backend");
        	return(result);
        }
        JsonObject result = (JsonObject) parser.parse(resp);
        if (result.has("tx")) {
            transaction = result.get("tx").getAsString();
        } else {
            transaction = null;
        }
        reader.close();
        return (result);
    }

    /**
     * Send command to ConfigDb
     * 
     * @param command
     * @return
     * @throws Exception
     */
    public JsonObject readFromConfigDb(DbCommand command) throws Exception {
        try {
            return (putRequest(createQuery(command), "configdb", 0, false));
        } catch (Exception e) {
            throw (e);
        }
    }

    public JsonObject writeConfigDb(DbCommand command) throws Exception {
        try {
            return (putRequest(createQuery(command), "configdb", 0, true));
        } catch (Exception e) {
            throw (e);
        }
    }

    // Read from specifiend non-OX database
    public JsonObject readFromDB(DbCommand command, String database) throws Exception {
        try {
            return (putRequest(createQuery(command), database, 0, false));
        } catch (Exception e) {
            throw (e);
        }
    }

    // Write to specified non-OX database
    public JsonObject writeToDb(DbCommand command, String database) throws Exception {
        try {
            return (putRequest(createQuery(command), database, 0, true));
        } catch (Exception e) {
            throw (e);
        }
    }

    /**
     * Get request from Database
     * 
     * @param command
     * @param cid Context ID for the database
     * @return
     * @throws Exception
     */
    public JsonObject readFromDB(DbCommand command, int cid) throws Exception {
        try {
            return (putRequest(createQuery(command), "oxdb", cid, false));
        } catch (Exception e) {
            throw (e);
        }
    }

    /**
     * Get request from Database
     * 
     * @param command
     * @param cid Context ID for the database
     * @return
     * @throws Exception
     */
    public JsonObject readFromShard(DbCommand command, int db) throws Exception {
        try {
            if (db < 0)
                return (null);
            return (putRequest(createQuery(command), Sharding.database_name(db), 0, false));
        } catch (Exception e) {
            throw (e);
        }
    }

    public JsonObject read(DbCommand command, int id, int cid) throws Exception {
        if (cid > 0) {
            return (readFromDB(command, cid));
        } else {
            return (readFromShard(command, Sharding.getShard(id, cid)));
        }
    }

    /**
     * Get request from OxGuard
     * 
     * @param command
     * @param cid Context ID for the database
     * @return
     * @throws Exception
     */
    public JsonObject readOG(DbCommand command) throws Exception {
        try {
            return (putOgRequest(createQuery(command), false));
        } catch (Exception e) {
            throw (e);
        }
    }

    /**
     * Write to database
     * 
     * @param command
     * @param cid Context ID for the database
     * @return
     * @throws Exception
     */
    public JsonObject writeToDB(DbCommand command, int cid) throws Exception {
        try {
            return (putRequest(createQuery(command), "oxdb", cid, true));
        } catch (Exception e) {
            throw (e);
        }
    }

    /**
     * Write to database
     * 
     * @param command
     * @param cid Context ID for the database
     * @return
     * @throws Exception
     */
    public JsonObject writeToShard(DbCommand command, int db) throws Exception {
        try {
            if (db < 0)
                return (null);
            return (putRequest(createQuery(command), Sharding.database_name(db), 0, true));
        } catch (Exception e) {
            throw (e);
        }
    }

    public JsonObject write(DbCommand command, int id, int cid) throws Exception {
        if (cid > 0) {
            return (writeToDB(command, cid));
        } else {
            return (writeToShard(command, Sharding.getShard(id, cid)));
        }
    }

    /**
     * Write to oxguard
     * 
     * @param command
     * @param cid Context ID for the database
     * @return
     * @throws Exception
     */
    public JsonObject writeOxGuard(Object command) throws Exception {
        try {
            if (command.getClass() == DbCommand.class) {
                return (putOgRequest(createQuery((DbCommand) command), true));
            } else {
                return (putOgRequest(createComplexQuery((MultiCommand) command), true));
            }
        } catch (Exception e) {
            throw (e);
        }
    }

    /**
     * Return if has error
     * 
     * @return
     */
    public boolean hasError() {
        if (result == null)
            return (true);
        return (result.get("error") != null);
    }

    /**
     * Get the error string
     * 
     * @return
     */
    public String getError() {
        if (result == null)
            return ("Null result");
        if (result.get("error") != null) {
            return (result.get("error").getAsString());
        }
        return (null);
    }

    /**
     * Has results?
     * 
     * @return
     */
    public boolean hasResults() {
        return (getRowCount() > 0);
    }

    /**
     * Get next row of data
     * 
     * @return
     */
    public boolean next() {
    	if (result == null) return(false);
    	try {
    		JsonArray rows = result.get("results").getAsJsonObject().get(query_id).getAsJsonObject().get("rows").getAsJsonArray();
    		index++;
    		if (index < rows.size()) {
    			rs = new OxResultSet(rows.get(index));
    			return (true);
    		}
    		return (false);
    	} catch (Exception ex) {
    		return(false);
    	}
    }

    /**
     * Get the row of data at index position
     * 
     * @param i index of row
     * @return
     */
    private OxResultSet getRow(int i) {
        return (new OxResultSet(result.get("results").getAsJsonObject().get(query_id).getAsJsonObject().get("rows").getAsJsonArray().get(i)));
    }

    /**
     * Get count of rows
     * 
     * @return
     */
    public int getRowCount() {
        return (result.get("results").getAsJsonObject().get(query_id).getAsJsonObject().get("rows").getAsJsonArray().size());
    }

    /**
     * Get result at an index
     * 
     * @param name
     * @param index
     * @return
     */
    public JsonElement getResult(String name, int index) {
        try {
            JsonArray rows = result.get("results").getAsJsonObject().get(query_id).getAsJsonObject().get("rows").getAsJsonArray();
            JsonObject row = (JsonObject) rows.get(index);
            return (row.get(name));
        } catch (Exception ex) {
            return (null);
        }
    }

    // Not really necessary, but used for compat if database used
    public void close() {
        result = null;
        rs = null;
        query_id = null;
        index = -1;
        updated = -1;
    }
}
