/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2014 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.storage;

import java.io.IOException;

import com.openexchange.guard.config.Config;
import com.openexchange.guard.database.CacheManager;
import com.openexchange.guard.database.Sharding;
import com.openexchange.guard.encr.EncryptedObj;
import com.openexchange.guard.storage.file.FileStorage;
import com.openexchange.guard.storage.s3.S3Storage;

/**
 * Interface handler for the file storage.
 * 
 * @author benjamin.otterbach@open-xchange.com
 */
public class Storage {

    /**
     * The delimiter character to separate the prefix from the keys
     */
    public static final String DELIMITER = "/";
	
    /**
     * Read encrypted object from file
     * 
     * @param userid User Id
     * @param contextid Users context id
     * @param ObjId Unique object id
     * @return
     * @throws Exception
     */
    public static EncryptedObj readEncrObj(int userid, int contextid, String ObjId) throws Exception {
    	filterObjId(ObjId);
    	switch (Config.getStorageType()) {
	    	case "file":
	    		return (FileStorage.getEncrObj(getDirectoryPrefix(userid, contextid, Sharding.getShard(userid, contextid)), ObjId));
	    	case "s3":
	    		return (S3Storage.getEncrObj(getDirectoryPrefix(userid, contextid, Sharding.getShard(userid, contextid)), ObjId));
	    	default:
	    		throw new Exception("Unknown value \"" + Config.getStorageType() + "\" for configuration com.openexchange.guard.storage.type while fetching encrypted file");
    	}
    }

    /**
     * Write encrypted object to file
     * 
     * @param userid User Id
     * @param contextid Users context id
     * @param ObjId Unique object id
     * @param data byte array of data to write
     * @return
     * @throws Exception
     */
    public static Boolean saveEncrObj(int userid, int contextid, String ObjId, byte[] data) throws Exception {
    	boolean saveSuccess = false;
    	filterObjId(ObjId);
    	String directoryPrefix = getDirectoryPrefix(userid, contextid, Sharding.getShard(userid, contextid));
    	
    	switch (Config.getStorageType()) {
	    	case "file":
	    		saveSuccess = FileStorage.writeObj(directoryPrefix, ObjId, data);
	    		break;
	    	case "s3":
	    		saveSuccess = S3Storage.writeObj(directoryPrefix, ObjId, data);
	    		break;
	    	default:
	    		throw new Exception("Unknown value \"" + Config.getStorageType() + "\" for configuration com.openexchange.guard.storage.type while storing encrypted file");
    	}
    	
    	if (saveSuccess)
    		CacheManager.addToCache(ObjId, userid, directoryPrefix + DELIMITER + ObjId);
    	
    	return saveSuccess;
    }
    
    public static void deleteEncrObj(String location) throws IOException, Exception {
    	switch (Config.getStorageType()) {
	    	case "file":
	    		FileStorage.deleteObj(location);
	    		break;
	    	case "s3":
	    		S3Storage.deleteObj(location);
	    		break;
	    	default:
	    		throw new Exception("Unknown value \"" + Config.getStorageType() + "\" for configuration com.openexchange.guard.storage.type while deleting encrypted file");
    	}
    }
    
    /**
     * Get directory prefix for the user account
     * 
     * @param userid
     * @param contextid
     * @param shardid
     * @return
     */
    private static String getDirectoryPrefix(int userid, int contextid, int shardid) {
        String directory = "";
        if (contextid < 0) {
        	directory = "ctx_" + contextid;
        } else {
        	directory = "ext_" + shardid + "_" + userid;
        }
        return (directory);
    }
    
    // Filter for .. attack
    private static void filterObjId (String ObjId) throws Exception {
    	if (ObjId.contains("..")) {
    		throw new Exception("Back Object Id " + ObjId);
    	}
    }
	
}
