/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2014 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.server;


import java.lang.management.ManagementFactory;
import java.security.Security;

import javax.management.MBeanServer;
import javax.management.ObjectName;

import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.eclipse.jetty.server.Connector;
import org.eclipse.jetty.server.Handler;
import org.eclipse.jetty.server.HttpConfiguration;
import org.eclipse.jetty.server.HttpConnectionFactory;
import org.eclipse.jetty.server.SecureRequestCustomizer;
import org.eclipse.jetty.server.Server;
import org.eclipse.jetty.server.ServerConnector;
import org.eclipse.jetty.server.SslConnectionFactory;
import org.eclipse.jetty.server.handler.ContextHandler;
import org.eclipse.jetty.server.handler.ContextHandlerCollection;
import org.eclipse.jetty.server.handler.DefaultHandler;
import org.eclipse.jetty.server.handler.HandlerCollection;
import org.eclipse.jetty.server.handler.RequestLogHandler;
import org.eclipse.jetty.server.handler.ResourceHandler;
import org.eclipse.jetty.servlet.ServletContextHandler;
import org.eclipse.jetty.servlet.ServletHolder;
import org.eclipse.jetty.util.resource.Resource;
import org.eclipse.jetty.util.ssl.SslContextFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ch.qos.logback.classic.LoggerContext;

import com.amazonaws.AmazonClientException;
import com.amazonaws.AmazonServiceException;
import com.amazonaws.services.s3.model.ListObjectsRequest;
import com.amazonaws.services.s3.model.ObjectListing;
import com.amazonaws.services.s3.model.S3ObjectSummary;
import com.openexchange.guard.cacheitems.Cache;
import com.openexchange.guard.config.Config;
import com.openexchange.guard.cron.Jobs;
import com.openexchange.guard.database.OGInit;
import com.openexchange.guard.dbpool.Manager;
import com.openexchange.guard.encr.PopulateRSA;
import com.openexchange.guard.ox.Languages;
import com.openexchange.guard.storage.s3.S3ClientFactory;

public class Startup {

	private static Logger logger = LoggerFactory.getLogger(Startup.class);
	
    public static void main(String[] args) throws Exception {
        if (com.openexchange.guard.setup.Check.confirm(args)) {
            try {
                final Server server = new Server();
                Security.addProvider(new BouncyCastleProvider());

                setLogLevel();
                
                Manager.init(); // Initialize the connection pool manager
                OxDbConn.init(); // Initialize ox backend http pool
                GuardServerStats stats = new GuardServerStats();
                MBeanServer mbs = ManagementFactory.getPlatformMBeanServer();
                ObjectName name = new ObjectName("com.open-xchange.guard:name=GuardServerStats");		
                mbs.registerMBean(stats, name);
                if (Config.getStorageType().equals("s3")) {
                	try {
                		S3ClientFactory.init();
                		logger.info("Using S3 store as storage for cached objects");
                    } catch (AmazonServiceException ase) {
                    	logger.error("Error S3ClientFactory init", ase);
                    	Runtime.getRuntime().halt(1);
                    } catch (AmazonClientException ace) {
                    	logger.error("Amazon client exception at S3ClientFactory init", ace);
                    	Runtime.getRuntime().halt(1);
                    }
                } else if (Config.getStorageType().equals("file")) {
                	try {
                        java.io.File file = new java.io.File(Config.getRootFileDirectory());
                        if (!file.exists()) {
                            file.mkdir();
                        }
                        logger.info("Using file store as storage for cached objects");
                    } catch (Exception e) {
                        logger.error("Problem with filestore", e);
                        Runtime.getRuntime().halt(1);
                    }
                } else {
                    logger.error("Unknown value \"" + Config.getStorageType() + "\" for configuration com.openexchange.guard.storage.type. Exiting.");
                	Runtime.getRuntime().halt(1);
                }
                                

                if (!com.openexchange.guard.util.CheckOx.oxAlive()) {
                    logger.error("OX Not responding.  Will wait 15 seconds and try again");
                    Thread.sleep(15000);
                    if (!com.openexchange.guard.util.CheckOx.oxAlive()) {
                        logger.error("OX Not responding.  Will wait 15 seconds and try again");
                        Thread.sleep(15000);
                        if (!com.openexchange.guard.util.CheckOx.oxAlive()) {
                            logger.error("FAILED TO CONNECT TO OX BACKEND.  EXITING");
                            System.out.println("Unable to connect to OX backend.  Verify open-xchange-rest is installed and this server has access to port " + Config.ox_backend_port);
                            Runtime.getRuntime().halt(1);
                        }
                    }
                }
                
                OGInit.initGuardDatabase(); // Initialize OxGuard databases
                
                Config.loadMysqlConfigData();
                final Thread rsathread = new Thread(new PopulateRSA());
                server.setStopTimeout(10000L);
                Runtime.getRuntime().addShutdownHook(new Thread() {

                    @Override
                    public void run() {
                        logger.info("Shutting down....Up to 10 second delay to finish");
                        try {
                            OxDbConn.close();
                            server.stop();
                            server.destroy();
                            try {
                                rsathread.interrupt();
                            } catch (Exception e) {
                            }
                            logger.info("Closing MySql Connections");
                            Manager.shutdown();
                            logger.info("OxGuard Shutdown complete");
                            Runtime.getRuntime().halt(0);
                        } catch (Exception e) {
                            logger.info("Failed to properly shut jetty down.");
                        }

                    }
                });
                // Handle arguments, and if done, exit
                Languages lang = new Languages(); // Initialize translation engine
                
                if (HandleArgs.handle(args)) Runtime.getRuntime().halt(0);
                
                Cache cache = new Cache(); // Initialize the cache
                Jobs cronJobs = new Jobs(); // Initialize the cron jobs
                

                if (Config.rsa_cache) { // If rsa caching enabled, start
                    rsathread.setPriority(Thread.MIN_PRIORITY);// Low priority background process
                    rsathread.start();
                }

                HttpConfiguration http_config = new HttpConfiguration();
                http_config.setSecureScheme("https");
                http_config.setSecurePort(Config.sslport);
                http_config.setOutputBufferSize(32768);
                ServerConnector http = new ServerConnector(server, new HttpConnectionFactory(http_config));
                http.setPort(Config.jettyport);
                http.setIdleTimeout(30000);
                if (Config.usessl) {
                    SslContextFactory sslContextFactory = new SslContextFactory();
                    sslContextFactory.setKeyStorePath(Config.sslkeystore);
                    sslContextFactory.setKeyStorePassword(Config.sslkeypass);
                    sslContextFactory.setCertAlias(Config.sslkeyname);
                    sslContextFactory.setKeyManagerPassword(Config.sslkeypass);
                    HttpConfiguration https_config = new HttpConfiguration(http_config);
                    https_config.addCustomizer(new SecureRequestCustomizer());
                    ServerConnector https = new ServerConnector(
                        server,
                        new SslConnectionFactory(sslContextFactory, "http/1.1"),
                        new HttpConnectionFactory(https_config));
                    https.setPort(Config.sslport);
                    https.setIdleTimeout(500000);
                    server.setConnectors(new Connector[] { http, https });
                } else {
                    server.setConnectors(new Connector[] { http });
                }
                ServletContextHandler context0 = new ServletContextHandler(ServletContextHandler.SESSIONS);
                context0.setContextPath("/oxguard");
                context0.addServlet(new ServletHolder(new MainServlet()), "/*");

                ResourceHandler rh0 = new ResourceHandler();
                logger.debug("Code Path: " + Config.codePath);
                rh0.setBaseResource(Resource.newResource(Config.codePath+ "reader"));
                ContextHandler context1 = new ContextHandler();
                context1.setContextPath("/oxguard/reader/*");
                context1.setHandler(rh0);

                ContextHandlerCollection contexts = new ContextHandlerCollection();
                contexts.setHandlers(new Handler[] { context0, context1 });

                HandlerCollection handlers = new HandlerCollection();
                RequestLogHandler requestLogHandler = new RequestLogHandler();
                handlers.setHandlers(new Handler[] { contexts, new DefaultHandler(), requestLogHandler });
                server.setHandler(handlers);
                
                logger.info("Guard server started");
                
                server.start();
                server.join();
            } catch (java.net.BindException e) {
                logger.error("Port already in use", e);
                Runtime.getRuntime().halt(1);
            }
        }
    }
    
    private static void setLogLevel() {
        ch.qos.logback.classic.Logger root = (ch.qos.logback.classic.Logger) LoggerFactory.getLogger(org.slf4j.Logger.ROOT_LOGGER_NAME);
        root.setLevel(Config.logLevel);
        LoggerContext lc = (LoggerContext) LoggerFactory.getILoggerFactory();
        for (ch.qos.logback.classic.Logger log : lc.getLoggerList()) {
        	if (log.getName().startsWith("org.eclipse.jetty")) {
        		log.setLevel(Config.jettyLogLevel);
        	}
        }
    }

}
