/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2014 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.server;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.security.NoSuchAlgorithmException;
import java.security.spec.InvalidKeySpecException;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.mail.internet.InternetAddress;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.apache.commons.fileupload.FileItem;
import org.apache.commons.fileupload.FileItemFactory;
import org.apache.commons.fileupload.disk.DiskFileItemFactory;
import org.apache.commons.fileupload.servlet.ServletFileUpload;
import org.apache.commons.io.IOUtils;
import org.bouncycastle.openpgp.PGPException;
import org.bouncycastle.util.encoders.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;

import com.google.gson.Gson;
import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.google.gson.JsonPrimitive;
import com.openexchange.guard.config.Config;
import com.openexchange.guard.config.OxConfiguration;
import com.openexchange.guard.database.Access;
import com.openexchange.guard.database.EmailInfo;
import com.openexchange.guard.database.GetPubKey;
import com.openexchange.guard.database.RecipKey;
import com.openexchange.guard.database.RetrievedItem;
import com.openexchange.guard.encr.Crypto;
import com.openexchange.guard.encr.EncrLib;
import com.openexchange.guard.encr.EncryptedObj;
import com.openexchange.guard.mailcreator.Attachment;
import com.openexchange.guard.mailcreator.Mail;
import com.openexchange.guard.ox.Api;
import com.openexchange.guard.ox.InterOx;
import com.openexchange.guard.ox.Languages;
import com.openexchange.guard.server.Settings.permissions;
import com.openexchange.guard.translate.GetText;
import com.openexchange.guard.util.Core;
import com.openexchange.guard.validator.Normalizer;

public class MailHandler {
	
	private static Logger logger = LoggerFactory.getLogger(MailHandler.class);
	
    OxCookie cookie;
    HttpServletRequest req;
    String sessionId;

    public MailHandler(OxCookie ck, HttpServletRequest rq) {
        cookie = ck;
        req = rq;
    }

    /**
     * Handle email sent from UI
     * 
     * @param request
     * @param response
     * @throws IOException
     */
    public void incomingEmailForm(HttpServletRequest request, HttpServletResponse response) throws IOException {
        FileItemFactory fif = new DiskFileItemFactory();
        ServletFileUpload sfu = new ServletFileUpload(fif);
        String itemID = EncrLib.getUUID(); // One itemID for email and it's attachments
        String salt = EncrLib.getsalt();// Create salt
        int userid = Integer.parseInt(request.getParameter("userid"));
        int cid = Integer.parseInt(request.getParameter("cid"));
        String sentfolder = request.getParameter("sentfolder");
        if (sentfolder == null) sentfolder = "default0/Sent Items";
        String guest = request.getParameter("guest");
        if (guest != null) {
        	if (!guest.toLowerCase().trim().equals("true")) guest = null;
        }
        String language = request.getParameter("lang");
        String guestlanguage = request.getParameter("guestlanguage");
        String attach_password = request.getParameter("password"); // encrypted attachment password
        String attach_extrapass = request.getParameter("extrapass"); // encrypted attachment extra password
        String auth = request.getParameter("auth");
        int templId = request.getParameter("templid") == null ? 0 : Integer.parseInt(request.getParameter("templid"));
        if (language == null)
            language = "en-US";
        EmailInfo info = null;
        String sr = request.getParameter("sr"); 
        // If not guest, verify logged in
        if (guest == null) {
        	Api ap = new Api(cookie, request);
        	if (!ap.verifyLogin()) {
        		logger.error("not logged into ui");
                Core.sendFail(response, "not authorized");
                return;
        	}
        	   // Load custom template Id for this user
        } else {
        	if (sr == null) {  // Reply ID must be sent if not part of UI
        		logger.error("Reply ID not sent");
        		Core.sendFail(response, "not authorized");
                return;
        	}	
        	cookie.JSESSIONID = cookie.OxReaderID;
        	UserData userdata = new UserData(auth, cookie);
        	cid = userdata.cid;
        	if (userid < 0) {
        		logger.debug("Bad authorization for guest login");
        		Core.sendFail(response, "not authorized");
                return;
        	}
        	userid = userdata.userid;
        	
        }

        boolean sendPermission = false;
        // Check for permission for sending email
        if (guest == null) {
	        if (!Auth.checkPermission(cookie, request, permissions.MAIL)) {
	            // If not permission, check if secure reply
	            if (sr == null) {
	                logger.error("not authorized");
	                Core.sendFail(response, "not authorized");
	                return;
	            }
	        } else sendPermission = true;
        }
        if ((sr != null) && !sendPermission) info = new EmailInfo(sr, userid, cid);
        if (cid < 0) {  // If negative context ID (non-ox user), use the orig senders templates for sending
        	templId = OxConfiguration.getTemplateId(info.owner_cid, info.owner_id);
        } else {
            templId = OxConfiguration.getTemplateId(cid, userid);
        }
        
        Crypto crypt = new Crypto();
        String contentKey = crypt.createContentKey();
        try {
            ArrayList<Attachment> attachments = new ArrayList<Attachment>();// Get list of all attachments
            String emaildata = "";
            String guestmessage = "";
            List files = sfu.parseRequest(request);
            sessionId = request.getParameter("session");
            String ua = request.getHeader("User-Agent");
            Iterator iterator = files.iterator(); // Iterate through attachments
            while (iterator.hasNext()) {
                FileItem fi = (FileItem) iterator.next();
                String name = fi.getFieldName();
                if (name.startsWith("guest_message")) {
                    guestmessage = fi.getString("UTF-8");
                    continue;
                }
                if (name.startsWith("json")) { // Field name json is the email data (or should be). So get email data
                    emaildata = fi.getString("UTF-8");
                } else {
                    name = fi.getName(); // Otherwise, assume it is an attachment
                    if (name != null) {// Make sure the name of the file exists, may just be blank field
                        if (name.trim() != "") {
                            if (name.lastIndexOf("\\") > -1) {
                                name = name.substring(name.lastIndexOf("\\"));
                            }
                            if (name.lastIndexOf("/") > -1) {
                                name = name.substring(name.lastIndexOf("/"));
                            }
                            ByteArrayOutputStream out = new ByteArrayOutputStream();
                            InputStream input = fi.getInputStream();
                            IOUtils.copy(input, out); // Copy stream to byte array
                            Attachment attach = new Attachment();
                            attach.content = out.toByteArray(); // Populate attachment data
                            attach.filename = fi.getName();

                            attach.type = fi.getContentType();
                            attachments.add(attach); // Add to our array of attachments
                            out.close();
                            input.close();
                        }
                    }
                }
            }
            if (emaildata != "") { // We must have the basic email data
                JsonParser jsonParser = new JsonParser(); // Put it into json
                JsonObject json = (JsonObject) jsonParser.parse(emaildata);
                JsonObject data = json.getAsJsonObject("data");
                ArrayList<RecipKey> recipients = getUsers(data, cid, guestlanguage); // get the users and populate the recipient keys
                if (recipients == null) {
                    logger.error("Unable to get keys for all users");
                    Core.sendFail(response, "keyproblem");
                    return;
                }
                
                if (!sendPermission) {
                	JsonArray from = data.get("from").getAsJsonArray().get(0).getAsJsonArray();
                	if (!checkReply(from, recipients, info)) {
                		logger.error("Attempt to reply to users not in original email list");
                		Core.sendFail(response, "unauth recip");
                		return;
                	}
                }
                
                // Encrypt email
                JsonObject header = data.getAsJsonObject("headers");
                String extrapass = "";
                try {
                    extrapass = header.get("X-OxGuard-ExtraPass").getAsString();
                } catch (Exception e) {
                    logger.error("Problem getting extra pass " + e.toString());
                }

                // Get infostore attachments
                try {
                    if (!data.get("infostore_ids").isJsonNull()) {
                        JsonArray infostores = data.get("infostore_ids").getAsJsonArray();
                        if (infostores != null) {
                            for (int i = 0; i < infostores.size(); i++) {
                                Attachment attach = getInfoAttach(infostores.get(i).getAsJsonObject(), request);
                                if (attach != null) 
                                	attachments.add(attach);
                            }
                        }
                    }
                } catch (Exception ex2) {
                    logger.error("Problem extracting infostore items.", ex2);

                }
                getInlineImages (data, attachments, request);
                Gson gson = new Gson();
                EncrLib encr = new EncrLib();
                String mailId = "<OxGuard-" + EncrLib.getUUID() + ">";
                try {
                    // Let's encrypt the email
                    long exp = getExpDate(data);
                    String oxemail = Mail.PrepareOXEmail(data, attachments, exp, mailId);
                    if (!extrapass.equals("")) {// if extra password specified, then encrypt first with aes
                        logger.debug("encrypting with extra password");
                        // oxemail = encr.encryptAES(oxemail, extrapass, itemID, Config.AESKeyLen);
                        oxemail = crypt.addExtraPass(oxemail, extrapass, itemID);
                    }
                    String encrypted = crypt.EncryptEmail(userid, cid, recipients, oxemail, exp, 1, "", itemID, contentKey, salt);
                    if (encrypted == null) {
                        com.openexchange.guard.util.Core.sendFail(response, "Failed to send");
                        return;
                    }
                    Attachment main = new Attachment();
                    main.encrContent = crypt.addHeader(encrypted, itemID); // Add the header
                    main.filename = itemID + ".grd"; // Our email filename is the itemID + .loc
                    main.type = "text/plain";
                    attachments.add(main); // Add it to our list of attachments for the email

                } catch (SQLException e) {
                    logger.error("Error while trying to encrypt email for item " + itemID, e);
                    com.openexchange.guard.util.Core.sendFail(response, "Failed to send");
                    return;

                }

                // Encrypt the attachments, using the same content key as for the email.
                for (int i = 0; i < attachments.size(); i++) {
                    Attachment attach = attachments.get(i);
                    if (attach.encrContent == null) {// Make sure this isn't the email
                        // Check if attachments are encrytped objects

                        if (com.openexchange.guard.util.Core.isEncrItem(attach.content)) {
                            try {
                                logger.debug("already encrypted");
                                EncryptedObj encrAttach = new EncryptedObj(attach.content);
                                String decoded = crypt.decodeEncrObj(userid, cid, encrAttach, attach_password);
                                if (decoded.startsWith("EXTRA")) {
                                    if (attach_extrapass != null) {
                                        decoded = crypt.decodeExtraPass(decoded, attach_extrapass, encrAttach.ItemID, encrAttach.keyLength);
                                    }
                                }
                                attach.content = Base64.decode(decoded);
                            } catch (Exception ex) {
                                logger.debug("Failed to decode attachments");
                                com.openexchange.guard.util.Core.sendFail(response, "FailDecodeAttach");
                                return;
                            }

                        }

                        attach.filename += ".grd";

                        if (!extrapass.equals("")) {
                            attach.encrContent = crypt.encrItem(
                                userid,
                                cid,
                                recipients,
                                crypt.addExtraPass(Base64.toBase64String(attach.content), extrapass, itemID),
                                getExpDate(data),
                                2,
                                "",
                                itemID,
                                contentKey,
                                true,
                                attach.filename);
                            // crypt.encrItem(userid, cid, recipients,encr.encryptAES(
                            // Base64.toBase64String(attach.content),extrapass,itemID, Config.AESKeyLen) , getExpDate(data), 2, "", itemID,
                            // contentKey, true, attach.filename);
                            attach.filename = attach.filename.replace(".grd", ".grd2"); // mark the attachment as extra password
                        } else {
                            attach.encrContent = crypt.encrItem(
                                userid,
                                cid,
                                recipients,
                                Base64.toBase64String(attach.content),
                                getExpDate(data),
                                2,
                                "",
                                itemID,
                                contentKey,
                                true,
                                attach.filename);
                        }

                        attachments.set(i, attach);
                    }
                }
                // Send the email link message
                if (sendEmailNotice(
                    recipients,
                    data,
                    itemID,
                    ua,
                    attachments,
                    !(extrapass.equals("")),
                    guest,
                    info,
                    language,
                    cid,
                    guestmessage,
                    header,
                    sentfolder,
                    mailId,
                    templId)) {
                    logger.info("Success email send");
                    // If was draft email, delete it from the draft folder
                    if (data.has("msgref")) {
                    	Api ap = new Api(cookie, request);
                    	ap.deleteDraft(data.get("msgref").getAsString());
                    }
                    com.openexchange.guard.util.Core.sendOK(response, "OK");// If OK, send response
                } else {
                    com.openexchange.guard.util.Core.sendFail(response, "Failed to send");
                    logger.debug("Failed to send email");
                }

            } else {// Our reply that the email was blank
                com.openexchange.guard.util.Core.sendOK(response, "No Email Data");
            }

        } catch (Exception ex) {// General failure response
            com.openexchange.guard.util.Core.sendFail(response, "Fail");
            logger.error("Unspecified error", ex);
        }
    }

    private Attachment getInfoAttach(JsonObject jsonElement, HttpServletRequest request) {
    	try {
	        int id = jsonElement.get("id").getAsInt();
	        int version = jsonElement.has("version") ? jsonElement.get("version").getAsInt() : -1;
	        int folder = jsonElement.get("folder_id").getAsInt();
	        Api ap = new Api(cookie, request);
	        Attachment attach = ap.getInfoStoreAttach(id, folder, version);
	        attach.filename = jsonElement.get("filename").getAsString();
	        return (attach);
    	} catch (Exception ex) {
    		logger.error("Unable to get infostore attachment ", ex);
    		return (null);
    	}
    }

    /**
     * Returns decrypted email
     * 
     * @param request
     * @param response
     * @param cookie
     * @throws Exception
     */
    public void getEncrEmail(HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie cookie) throws Exception {
    	
    	JsonObject jdata = Core.getJSON(request);
    	
    	
        int emailid = jdata.get("emailid").getAsInt();
        int userid = 0;
        if (!jdata.get("userid").isJsonNull()) {
            userid = jdata.get("userid").getAsInt();
        }
        int cid = jdata.get("cid").getAsInt();
        int attachid = jdata.get("attach").getAsInt();
        String auth = request.getParameter("auth");
        String sessionID = request.getParameter("session");
        String folder = jdata.get("folder").getAsString();
        String extraPass = jdata.get("extrapass").getAsString();
        
        // cc is contentKey.  not currently implemented.  For possible client side decode of contentKey
        // if implemented, should be encrypted between client and server
        String cc = jdata.get("cc").getAsString();
        if (cc.equals("")) cc = null;
        //
        //
        if (CheckBad.isBad(Integer.toString(emailid), Config.bad_password_count)) {
            com.openexchange.guard.util.Core.sendOK(response, "{\"Exceeded\" : " + Config.bad_minute_lock + "}");
            return;
        }
        UserData data;
        try {
            if (auth.length() < 20) {

                String password = jdata.get("password").getAsString();
                if ((password == null) && (cc != null))
                    password = "";// if content key sent decoded, we don't need password. Remove null
                data = new UserData();
                data.encr_password = password;
                data.userid = userid;

            } else {
                long start = System.currentTimeMillis();
                data = new UserData(auth, cookie);
                logger.info("Auth decoded at " + start);
            }

            if ((data.userid == -1) || (data.encr_password == null)) {// Make sure our authorization is OK.
                com.openexchange.guard.util.Core.sendFail(response, "Bad authorization");
                CheckBad.addBad(sessionID);
                return;
            }
            Crypto crypto = new Crypto();

            Api ap = new Api(cookie, sessionID, request.getHeader("User-Agent"));
            EncryptedObj emaildata = ap.getAttachment(emailid, attachid, folder);// Get the attachment from our orig email that contains
                                                                                 // encrypted email
            String email = crypto.decodeEncrObj(userid, cid, emaildata, data.encr_password, cc); // Decode it
            if (email != null) {
            	if (!email.equals("")) {
	                if ((auth.length() < 20) || extraPass != null) { // If we provided plaintext password or extrapass, we need to encrypt to
	                                                                 // return for attachments
	                    String extrap = "";
	                    String epassword = "";
	                    EncrLib encr = new EncrLib();
	                    Access acc = new Access();
	                    String token = acc.getToken(cookie.JSESSIONID);
	                    if (token == null)
	                        token = acc.newToken(cookie.JSESSIONID, userid, cid);
	                    if ((extraPass != null) && (email.length() > 10)) { // Extra pass, and not retracted/expired
	                        if (extraPass.length() > 2) {
	                            email = crypto.decodeExtraPass(email, extraPass, emaildata.ItemID, emaildata.keyLength);
	                            if (email.equals("")) {// If bad extra password
	                                com.openexchange.guard.util.Core.sendFail(response, "Bad authorization");
	                                CheckBad.addBad(emaildata.ItemID);
	                                return;
	                            }
	                            extrap = encr.encryptAES(extraPass, token);
	                        }
	                    }
	                    epassword = encr.encryptAES(data.encr_password, token);
	                    email = "{" + "\"extrapass\":\"" + extrap + "\", \"epass\":\"" + epassword + "\"," + (email.startsWith("{") ? email.substring(1) : email); // add
	                                                                                                                                                               // extrapass
	                                                                                                                                                               // info
	                                                                                                                                                               // to
	                                                                                                                                                               // email
	                                                                                                                                                               // json
	                }
	                email = Normalizer.normalize(email);
	                logger.info("Success decode email");
	                response.addHeader("content-type", "text/html; charset=utf-8");
	                com.openexchange.guard.util.Core.sendOK(response, email);
	                return;
            	}
            }
        } catch (InvalidKeySpecException | NoSuchAlgorithmException | SQLException | PGPException e) {
        	logger.error("Error while encrypting email " + emailid + " for user " + cid + "/" + userid, e);
        }
        com.openexchange.guard.util.Core.sendFail(response, "Unable to decrypt");

    }

    /*
     * Decodes an encrypted attachment and returns to decoded
     */
    private com.openexchange.guard.encr.DecodedObj decodeAttach(HttpServletRequest request, HttpServletResponse response, int emailid, int userid, int attachid, int cid, String auth, String filename, String sessionID, String extrapass, String encrextrapass, String folder, String cc, UserData data) {

        if (data.userid == -1) {
            com.openexchange.guard.util.Core.sendFail(response, "Bad authorization");
            return (null);
        }
        Crypto crypto = new Crypto();
        try {
            Api ap = new Api(cookie, sessionID, request.getHeader("User-Agent"));
            EncryptedObj attdata = ap.getAttachment(emailid, attachid, folder);
            com.openexchange.guard.encr.DecodedObj decoded = new com.openexchange.guard.encr.DecodedObj();
            decoded.type = attdata.content_type;// save content type
            decoded.decoded64 = crypto.decodeEncrObj(userid, cid, attdata, data.encr_password, cc);
            if (decoded.decoded64 != null) {// Data was stored base64, so we need to decode when returned
                EncrLib encr = new EncrLib();
                if (extrapass == null) {// If plaintext extrapass is null, then look at encryptedpass
                    if (encrextrapass != null) {
                        if (!encrextrapass.equals("")) {
                            Access acc = new Access();
                            String token = acc.getToken(cookie.JSESSIONID);
                            extrapass = encr.decryptAES(encrextrapass, token);
                        }
                    }
                }
                if (extrapass != null) {
                    if (!extrapass.equals("")) {
                        decoded.decoded64 = encr.decryptAES(decoded.decoded64.substring(5), extrapass, attdata.ItemID, attdata.keyLength);
                    }
                }
                
                return (decoded);
            }
        } catch (Exception e) {
            logger.error("Error decoding attachment " + attachid + " of email " + emailid, e);
        }
        com.openexchange.guard.util.Core.sendFail(response, "Unable to decrypt");
        return (null);
    }

    /**
     * Function to return encrypted attachment (non email)
     * 
     * @param request
     * @param response
     * @param cookie
     * @throws IOException
     */
    public void getEncrAttach(HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie cookie) throws IOException {
    	String email = request.getParameter("emailid");
    	int emailid = 0;
    	String folder = request.getParameter("folder");
    	try { // Unified inbox combines folder id and item id.  Try separating 
	    	if (email.contains("/")) {
	    		emailid = Integer.parseInt(email.substring(email.lastIndexOf("/") + 1));
	    		folder = email.substring(0, email.lastIndexOf("/"));
	    	} else {
	    		emailid = Integer.parseInt(email);
	    	}
    	} catch (Exception ex) {
    		logger.error("Error getting emailid and folder for attachment", ex);
    		Core.sendFail(response, "Error parameters");
    	}
        int userid = Integer.parseInt(request.getParameter("userid"));
        int attachid = Integer.parseInt(request.getParameter("attach"));
        int cid = Integer.parseInt(request.getParameter("cid"));
        String auth = request.getParameter("auth");
        String filename = request.getParameter("attname");
        String sessionID = request.getParameter("session");
        String extrapass = request.getParameter("extrapass");
        String encrextrapass = request.getParameter("encrextrapass");
        String download = request.getParameter("download");
        String cc = request.getParameter("cc");// Content key if decoded offsite
        UserData data;
        if (auth.length() < 20) {
            String password = request.getParameter("password");
            if ((password == null) || password.equals("undefined")) {
                String epassword = request.getParameter("epassword");
                if (epassword != null) {
                    Access acc = new Access();
                    String token = acc.getToken(cookie.JSESSIONID);
                    EncrLib enc = new EncrLib();
                    password = enc.decryptAES(epassword, token);
                }
            }
            data = new UserData();
            data.encr_password = password;
            data.userid = userid;
        } else {
            data = new UserData(auth, cookie);
        }
        com.openexchange.guard.encr.DecodedObj decoded = decodeAttach(
            request,
            response,
            emailid,
            userid,
            attachid,
            cid,
            auth,
            filename,
            sessionID,
            extrapass,
            encrextrapass,
            folder,
            cc,
            data);
        if (decoded == null) {
        	logger.debug("Failed to decode attachment");
            com.openexchange.guard.util.Core.sendFail(response, "Failed to decode");
            return;
        }
        if (decoded.decoded64 == "exp") { // If our email has expired, then just send that
        	logger.info("Expired attachment");
            com.openexchange.guard.util.Core.sendOK(response, "Attachment Expired");
            return;
        }
        logger.info("Success decode attachment");
        if ((download != null) && download.contains("true")) { // Send the attachment to broswer to save
            sendToSaveAttach(response, decoded.type, Base64.decode(decoded.decoded64), filename);
        } else {
            try {
                com.openexchange.guard.util.Core.sendOK(response, decoded.type, Base64.decode(decoded.decoded64));
            } catch (Exception ex) {
                logger.error("Error sending OK response", ex);
                com.openexchange.guard.util.Core.sendFail(response, "Error");
            }
        }

    }

    public void saveDecodedAttach(HttpServletRequest request, HttpServletResponse response, com.openexchange.guard.server.OxCookie cookie) throws IOException {
        JsonObject json = com.openexchange.guard.util.Core.getJSON(request);
        String folder = json.get("folder").getAsString();
        int emailid = 0;
        try {
        	emailid = json.get("id").getAsInt();
        } catch (Exception ex) {  // Unified inbox combines folder id and item id.  Try separating 
        	String email = java.net.URLDecoder.decode(json.get("id").getAsString(), "UTF-8");
        	if (email.contains("/")) {
	    		emailid = Integer.parseInt(email.substring(email.lastIndexOf("/") + 1));
	    		folder = email.substring(0, email.lastIndexOf("/"));
	    	} else {
	    		emailid = Integer.parseInt(email);
	    	}
        }
        int userid = json.get("userid").getAsInt();
        int attachid = json.get("attachment").getAsInt();
        int cid = json.get("cid").getAsInt();
        String auth = json.get("auth").isJsonNull() ? "" : json.get("auth").getAsString();
        String filename = json.get("filename").getAsString();
        String sessionID = json.get("session_id").getAsString();
        String description = json.get("description").getAsString();
        String dest_folder = json.get("dest_folder").getAsString();
        String cc = null;
        String extrapass = null;
        String encrextrapass = "";
        if (json.get("cc") != null) {
            cc = json.get("cc").getAsString();
        }
        if (json.get("extrapass") != null) {
            extrapass = json.get("extrapass").getAsString();
        }
        if (json.get("encrextrapass") != null) {
            encrextrapass = json.get("encrextrapass").getAsString();
        }
        UserData data;
        if (auth.length() < 20) {
            String password = request.getParameter("password");
            if ((password == null) || password.equals("undefined")) {
                String epassword = json.get("epassword").getAsString();
                if (epassword != null) {
                    Access acc = new Access();
                    String token = acc.getToken(cookie.JSESSIONID);
                    EncrLib enc = new EncrLib();
                    password = enc.decryptAES(epassword, token);
                } else password = null; // null password if was undefined and no epassword
            }
            data = new UserData();
            data.encr_password = password;
            data.userid = userid;
        } else {
            data = new UserData(URLDecoder.decode(auth), cookie);
        }
        if (data.encr_password == null) {
            com.openexchange.guard.util.Core.sendFail(response, "Failed to decode");
            return;
        }
        com.openexchange.guard.encr.DecodedObj decoded = decodeAttach(
            request,
            response,
            emailid,
            userid,
            attachid,
            cid,
            auth,
            filename,
            sessionID,
            extrapass,
            encrextrapass,
            folder,
            cc,
            data);
        if (decoded == null) {
            com.openexchange.guard.util.Core.sendFail(response, "Failed to decode");
            return;
        }
        ArrayList<Attachment> attachments = new ArrayList<Attachment>();
        Attachment file = new Attachment();
        file.content = Base64.decode(decoded.decoded64);
        file.filename = filename;
        file.type = decoded.type;
        attachments.add(file);

        // Save to ox backend
        String params = "files?action=new&filename=" + URLEncoder.encode(filename) + "&session=" + sessionID;
        com.openexchange.guard.ox.Api ap = new com.openexchange.guard.ox.Api(cookie, request);
        String newjson = "{\"folder_id\" : \"" + dest_folder + "\",\"description\":\"" + description + "\"}";
        String resp = ap.filePost(newjson, params, attachments);
        if (com.openexchange.guard.util.Core.checkFail(resp)) {
        	String error = Core.getError(resp);
            com.openexchange.guard.util.Core.sendFail(response, (error.equals("") ? "Failed to save" : error));
        } else {
            com.openexchange.guard.util.Core.sendOK(response, "OK");
        }
    }

    /**
     * Get the list of recipients from a json email, and get their keys
     * 
     * @param json
     * @return
     * @throws Exception
     */
    public ArrayList<RecipKey> getUsers(JsonObject json, int senderCID, String guestlanguage) throws Exception {

        ArrayList<RecipKey> recipients = new ArrayList<RecipKey>();

        JsonArray recips = json.get("to").getAsJsonArray();
        String from = json.get("from").getAsJsonArray().get(0).getAsJsonArray().get(1).getAsString();
        String fromName = json.get("from").getAsJsonArray().get(0).getAsJsonArray().get(0).getAsString();
        recipients = addUsers(recipients, recips, senderCID, from, fromName, RecipKey.reciptype.to, guestlanguage);
        if (recipients == null)
            return (null);
        recips = json.get("cc").getAsJsonArray();
        recipients = addUsers(recipients, recips, senderCID, from, fromName, RecipKey.reciptype.cc, guestlanguage);
        if (recipients == null)
            return (null);
        recips = json.get("bcc").getAsJsonArray();
        recipients = addUsers(recipients, recips, senderCID, from, fromName, RecipKey.reciptype.bcc, guestlanguage);

        return (recipients);

    }

    private ArrayList<RecipKey> addUsers(ArrayList<RecipKey> recipients, JsonArray recips, int senderCID, String from, String fromName, RecipKey.reciptype type, String guestlanguage) throws Exception {
        for (int i = 0; i < recips.size(); i++) {
            JsonArray recip = recips.get(i).getAsJsonArray();
            String email = recip.get(1).getAsString();
            GetPubKey gk = new GetPubKey();
            try {
                RecipKey key = gk.getPubKey(email, senderCID, from, true, false, guestlanguage, fromName);
                if (key != null) {
                    if (key.pubkey == null) {
                        // Fail to create key
                        return (null);
                    }
                    key.email = email;
                    key.type = type;
                    try {
                        key.name = recip.get(0).getAsString();
                    } catch (Exception ex) {
                        key.name = "";
                        logger.error("No key name set for " + email, ex);
                    }
                    recipients.add(key);
                } else {
                	return (null);
                    // No keys found or made
                }
            } catch (SQLException e) {
            	logger.error("SQL error when getting public key", e);
            } catch (IOException e) {
            	logger.error("IO error when getting public key", e);
            }
        }
        return (recipients);
    }

    private boolean checkReply (JsonArray from, ArrayList<RecipKey> recips, EmailInfo info) {
    	if (info == null) return(false);
    	if (recips == null) return(false);
    	if (!info.fromEmail.toLowerCase().trim().equals(from.get(1).getAsString().toLowerCase().trim())) return(false);
    	for (int i = 0; i < recips.size(); i++) {
    		if (!info.recips.contains(recips.get(i).email)) {
    			if (!info.senderEmail.contains(recips.get(i).email)) return(false);
    		}
    	}
    	return (true);
    }
    /**
     * Send the notice email to our recipients
     * 
     * @param json The original email json
     * @param id The item ID
     * @param ua user agent
     * @param attachments
     */
    private boolean sendEmailNotice(ArrayList<RecipKey> recipients, JsonObject json, String id, String ua, ArrayList<Attachment> attachments, boolean extraEncr, String guest, EmailInfo info, String language, int cid, String guestmessage, JsonObject oldheader, String sentfolder, String mailId, int templId) {
        Mail mf = new Mail();
        boolean success = true;
        String subject = json.get("subject").getAsString();
        JsonArray from = json.get("from").getAsJsonArray().get(0).getAsJsonArray();
        JsonObject head = new JsonObject();
        head.addProperty("X-OxGuard", true);
        head.addProperty("X-OxGuard-ID", id);
        head.addProperty("Message-ID", mailId);
        if (oldheader.get("In-Reply-To") != null) {
            head.addProperty("In-Reply-To", oldheader.get("In-Reply-To").getAsString());
        }
        if (oldheader.get("References") != null) {
            head.addProperty("References", oldheader.get("References").getAsString());
        }
        if (guest != null) {  // Send as
        	if (!info.senderEmail.equals("")) {
        		head.addProperty("Sender", info.senderEmail);
        	}
        }
        Api ap = new Api(cookie, sessionId, ua);
        if (extraEncr) {
            head.addProperty("X-OxGuard-ExtraPass", true);
        }
        Gson gson = new Gson();
        JsonObject guestHeader = gson.fromJson(head, JsonObject.class);
        guestHeader.addProperty("X-OxGuard-Guest", true);
        ArrayList<String> failures = new ArrayList<String>(); // A list of failed emails
        for (int i = 0; i < recipients.size(); i++) {
            RecipKey recip = recipients.get(i);

            if ((recip.cid < 0) && !recip.remote) { // If this is a non-ox member, then we're going to send a different email
                JsonObject email = mf.createBlankGuestMail(recip.lang, templId);
                if (email == null)
                    return (false);
                email = mf.addTo(email, recip.name, recip.email, recip.type);
                email = mf.addFrom(email, from.get(0).getAsString(), from.get(1).getAsString());
                email = mf.addSubject(email, subject);
                email = mf.addPlainText(email, guestmessage);
                email.getAsJsonObject("data").add("headers",  guestHeader);
                email = mf.noSave(email);
                try {
					email = mf.addURL(
					    email,
					    "https://" + Config.OXUrl + Config.externalReaderPath + "?email=" + id +
					    "&user=" + URLEncoder.encode(recip.email, "ISO-8859-1") + (extraEncr ? "&extrapass=true" : "") + (Languages.customTemplateExists(
					        templId,
					        "guesttempl.html") ? ("&templid=" + templId) : "") + "&lang=" + recip.lang);
				} catch (UnsupportedEncodingException e) {
					email = mf.addURL(
						    email,
						    "https://" + Config.OXUrl + Config.externalReaderPath + "?email=" + id);
				}
                if (recip.newGuestPass != null) {
                    // This is a new Guest account with a new password, need to send notice to the recipient
                    if (com.openexchange.guard.util.Core.checkFail(ap.sendMail(
                        mf.getPasswordEmail(
                            recip.email,
                            from.get(0).getAsString(),
                            from.get(0).getAsString(),
                            from.get(1).getAsString(),
                            recip.newGuestPass,
                            recip.lang,
                            templId,
                            guestmessage),
                        null).toString())) {
                        failures.add(recip.email);
                    }
                }
                if (cid > 0) { // Check if OX member sending, or non-ox replying
                    if (com.openexchange.guard.util.Core.checkFail(ap.sendMail(email, attachments).toString())) {
                        failures.add(recip.email);
                    }
                } else {
                    Sender send = new Sender();
                    if (!send.send(email, attachments, info.senderEmail, language, mailId)) {
                        failures.add(recip.email);
                    }
                }
            } else {
                JsonObject email = mf.createBlankOutMail(recip.lang, templId);
                if (email == null)
                    return (false);
                email = mf.addTo(email, recip.name, recip.email, recip.type);
                email = mf.addFrom(email, from.get(0).getAsString(), from.get(1).getAsString());
                /*
                if (attachments.size() > 1) {
                    email = mf.addSubject(email, "OXGrd:: " + subject); // Add double :: if has attachments
                } else
                    email = mf.addSubject(email, "OXGrd: " + subject);
                */
                email = mf.addSubject(email, subject);
                email.getAsJsonObject("data").add("headers", head);
                email = mf.addURL(email, "https://" + Config.OXUrl);
                email = mf.noSave(email);
                if (cid > 0) {
                    if (com.openexchange.guard.util.Core.checkFail(ap.sendMail(email, attachments).toString())) {
                        failures.add(recip.email);
                    }
                } else {
                    Sender send = new Sender();
                    if (!send.send(email, attachments, info.senderEmail, language, mailId)) {
                        failures.add(recip.email);
                    }
                }
            }
        }
        if (failures.size() > 0) {// Some recipients have failed for some reason, notify the sender
            JsonObject failmail = mf.createFailureMail();
            failmail = mf.addTo(failmail, from.get(0).getAsString(), from.get(1).getAsString());
            failmail = mf.addFrom(failmail, "Mailer", from.get(1).getAsString());
            failmail = mf.addSubject(failmail, GetText.gettext("Mail delivery failure", language));
            String content = GetText.gettext("Email delivery to the following recipients failed:", language) + ": ";
            for (int f = 0; f < failures.size(); f++) {
                content += failures.get(f) + ", ";
            }
            failmail = mf.setContent(failmail, content);
            success = false;
            if (cid > 0) // We can only direct into inbox with ox users
                logger.debug(ap.saveInBox(failmail, null).toString());
        }
        // Let's save to the outbox the email, sent to all recipients.
        JsonObject email = mf.createBlankOutMail(language, templId);
        for (int i = 0; i < recipients.size(); i++) {
            email = mf.addTo(email, recipients.get(i).name, recipients.get(i).email, recipients.get(i).type);
        }
        email = mf.addFrom(email, from.get(0).getAsString(), from.get(1).getAsString());
        // email = mf.addSubject(email, "OXGrd: " + subject);
        email = mf.addSubject(email, subject);
        email.getAsJsonObject("data").add("headers", head);
        email = mf.addURL(email, "https://" + Config.OXUrl);
        if (cid > 0)
            logger.debug(ap.saveSentMail(email, attachments, sentfolder).toString());
        return (success);
    }

    /**
     * Send attachment contents to browser with flags to save to disk
     * 
     * @param response
     * @param type
     * @param answer
     * @param filename
     */
    private void sendToSaveAttach(HttpServletResponse response, String type, byte[] answer, String filename) {
        response.setContentType(type);
        response.setHeader("Content-Disposition", "attachment; filename=\"" + filename + "\"");
        response.addHeader("Access-Control-Allow-Origin", "*");
        response.setStatus(HttpServletResponse.SC_OK);
        try {
            OutputStream out = response.getOutputStream();
            out.write(answer);
            out.close();
        } catch (IOException e) {
        	logger.error("Error when writing to output stream", e);
        }
    }

    /*
     * Get exp date of email
     */
    private long getExpDate(JsonObject email) {
        JsonObject header = email.get("headers").getAsJsonObject();
        int hours = header.get("X-OxGuard-Expiration").getAsInt();
        if (hours == 0) {
            return (0);
        }
        final long exphours = hours * 60L * 60L * 1000L;
        Date now = new Date();
        return (now.getTime() + exphours);
    }

    /*
     * Get the public key of a recipient
     */
    public void getRecipKey(HttpServletRequest request, HttpServletResponse response, String email) throws Exception {
    	InternetAddress em = new InternetAddress(email);
    	email = em.getAddress();
  //      if (email.contains("<")) {
  //          email = email.substring(email.indexOf("<") + 1);
  //          email = email.replace(">", "");
   //     }
        String from = request.getParameter("from");
        InternetAddress f = new InternetAddress(from);
        from = f.getAddress();
  //      if (from.contains("<")) {
 //           from = from.substring(from.indexOf("<") + 1).replace(">", "");
 //       }
        GetPubKey acc = new GetPubKey();
        try {
            int cid = Integer.parseInt(request.getParameter("cid"));
            RecipKey recip = acc.getPubKey(email, cid, from, false, false);
            if (recip != null) {
                response.setContentType("application/json");
                response.addHeader("Access-Control-Allow-Origin", "*");
                response.setStatus(HttpServletResponse.SC_OK);
                String keyreply = "unable";
                if (recip.pubkey != null) {// If we have a public key, then we are golden
                    keyreply = recip.pubkey.toString().replace("\r", "").replace("\n", "");
                    response.getWriter().write(
                        "{\"email\" : \"" + email + "\", \"publicKey\" : \"" + keyreply + "\", \"guest\" : \"" + recip.guest + "\"}");
                } else {// If we don't have a public key, will be guest, or was unable to create
                    if (recip.guest) {
                        response.getWriter().write(
                            "{\"email\" : \"" + email + "\", \"publicKey\" : \"" + keyreply + "\", \"guest\" : \"" + recip.guest + "\"}");
                    } else {
                    	if (recip.cid > 0) keyreply = "Will be created on first send";
                        response.getWriter().write("{\"email\" : \"" + email + "\", \"publicKey\" : \"" + keyreply + "\", \"guest\" : \"" + recip.guest + "\"}");
                    }
                }
            } else {
                response.getWriter().write("{\"email\" : \"" + email + "\", \"publicKey\" : \"error\", \"guest\" : \"\"}");
            }
        } catch (SQLException e) {
            logger.error("SQL error getting public key of recipient " + email, e);
            e.printStackTrace();
        } catch (IOException e) {
            logger.error("IO error getting public key of recipient " + email, e);
            e.printStackTrace();
        }

    }

    /**
     * Get the read count of an email item. Will query remote services
     * 
     * @param request
     * @param response
     */
    public void getReadCount(HttpServletRequest request, HttpServletResponse response) {
    	String itemID = request.getParameter("item");
    	try {
            JsonArray users = new JsonArray();
            int userid = Integer.parseInt(request.getParameter("userid"));
            int cid = Integer.parseInt(request.getParameter("cid"));
            if ((itemID == null)) {
                com.openexchange.guard.util.Core.sendFail(response, "Bad item id");
                return;
            }
            Access acc = new Access();
            String xml = acc.getXML(userid, cid, itemID);
            // Need to parse xml for the users that received this email
            if (xml == null) {
                com.openexchange.guard.util.Core.sendFail(response, "Not owner or not found");
                return;
            }
            DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
            DocumentBuilder builder = factory.newDocumentBuilder();
            org.w3c.dom.Document doc = builder.parse(new InputSource(new ByteArrayInputStream(xml.getBytes("utf-8"))));
            doc.getDocumentElement().normalize();
            Element share = (Element) doc.getElementsByTagName("share").item(0);
            NodeList shares = share.getElementsByTagName("u");
            InterOx iox = new InterOx();
            for (int i = 0; i < shares.getLength(); i++) {// For each of the users, check count
                JsonObject readcount = new JsonObject();
                Element user = (Element) shares.item(i);
                String email = user.getElementsByTagName("e").item(0).getFirstChild().getTextContent();
                int rcid = Integer.parseInt(user.getElementsByTagName("c").item(0).getFirstChild().getTextContent());
                int ruserid = Integer.parseInt(user.getElementsByTagName("i").item(0).getFirstChild().getTextContent());
                boolean remote = user.getElementsByTagName("r").item(0).getFirstChild().getTextContent().contains("Y");
                int count = -1;
                if (remote) {// If it is remote, then get it from other ox server
                    count = iox.getRemoteReadCount(email, itemID);
                } else {
                    count = acc.getCount(ruserid, rcid, itemID);
                }
                readcount.add("email", new JsonPrimitive(email));// add to our response
                readcount.add("count", new JsonPrimitive(count));
                users.add(readcount);
            }
            com.openexchange.guard.util.Core.sendOK(response, users.toString());

        } catch (Exception ex) {
            logger.error("Error getting read count for " + itemID, ex);
            com.openexchange.guard.util.Core.sendFail(response, "Error getting count");
        }

    }
    
    public JsonObject getInlineImages (JsonObject data, ArrayList<Attachment> attachments, HttpServletRequest request) {
    	JsonArray contents = data.get("attachments").getAsJsonArray();
    	JsonArray newcontents = new JsonArray();
    	for (int i = 0; i < contents.size(); i++) {
    		if (contents.get(i).getAsJsonObject().get("content").isJsonNull()) {
    			newcontents.add(contents.get(i));
    		}
    		else {
    			JsonObject content = contents.get(i).getAsJsonObject();
    			String cont = content.get("content").getAsString();
    			int start = cont.indexOf("<img");
    			boolean changed = false;
    			while ((start > 0) && (start < cont.length())) {
    				int end = cont.indexOf(">", start);
    				if (end < 0) break;
    				String image = cont.substring(start, end);
    				Pattern pattern = Pattern.compile("src=\"[^\"]*");
    				Matcher match = pattern.matcher(image);
    				if (match.find()) {
    					String url = image.substring(match.start() +5, match.end());
    					if (url.length() > 1) {
	    					Api ap = new Api (cookie, request);
	    					Attachment attach = ap.getInlineImage(url);
	    					if (attach != null) attachments.add(attach);
	    					cont = cont.replace(url, "attached");
	    					changed = true;
    					}
    				}
    				start = cont.indexOf("<img>", end);
    			}
    			if (changed) {
        			content.remove("content");
        			content.addProperty("content", cont);
    			}
    			newcontents.add(content);
    			}
    	}
    	data.remove("attachments");
    	data.add("attachments", newcontents);
    	return (data);
    }

    /**
     * Retract an email item. Will query remote services
     * 
     * @param request
     * @param response
     */
    public void retractEmail(HttpServletRequest request, HttpServletResponse response, OxCookie cookie) {
    	String itemID = request.getParameter("item");
        try {
            int userid = Integer.parseInt(request.getParameter("userid"));
            int cid = Integer.parseInt(request.getParameter("cid"));
            String session = request.getParameter("session");
            String ua = request.getHeader("User-Agent");
            if ((itemID == null)) {
                com.openexchange.guard.util.Core.sendFail(response, "Bad item id");
                return;
            }
            Api ap = new Api(cookie, session, ua);// Verify we are logged in for security
            if (!ap.verifyLogin()) {
                com.openexchange.guard.util.Core.sendFail(response, "Must be logged in");
                return;
            }
            Access acc = new Access();
            String xml = acc.getXML(userid, cid, itemID);
            // Need to parse xml for the users that received this email
            if (xml == null) {
                com.openexchange.guard.util.Core.sendFail(response, "Not owner or not found");
                return;
            }
            DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
            DocumentBuilder builder = factory.newDocumentBuilder();
            org.w3c.dom.Document doc = builder.parse(new InputSource(new ByteArrayInputStream(xml.getBytes("utf-8"))));
            doc.getDocumentElement().normalize();
            Element share = (Element) doc.getElementsByTagName("share").item(0);
            NodeList shares = share.getElementsByTagName("u");
            InterOx iox = new InterOx();
            RetrievedItem obg = acc.getItemID(itemID, userid, cid);
            int retracted = 0;
            boolean delete = Auth.getPermissions(cookie, request).deleteKeyOnRevoke;
            for (int i = 0; i < shares.getLength(); i++) {// For each of the users, check count
                Element user = (Element) shares.item(i);
                String email = user.getElementsByTagName("e").item(0).getFirstChild().getTextContent();
                int ruserid = Integer.parseInt(user.getElementsByTagName("i").item(0).getFirstChild().getTextContent());
                int rcid = Integer.parseInt(user.getElementsByTagName("c").item(0).getFirstChild().getTextContent());
                boolean remote = user.getElementsByTagName("r").item(0).getFirstChild().getTextContent().contains("Y");
                if (remote) {// If it is remote, then get it from other ox server
                    if (iox.retractRemote(email, itemID, obg.salt, delete).contains("OK"))
                        retracted++;
                } else {
                    if (acc.retract(ruserid, rcid, itemID, obg.salt, delete))
                        retracted++;
                }

            }
            if (retracted == shares.getLength()) {
                com.openexchange.guard.util.Core.sendOK(response, "OK");
            } else {
                com.openexchange.guard.util.Core.sendOK(response, "PARTIAL");
            }

        } catch (Exception ex) {
            logger.error("Error retracting an email with itemID " + itemID, ex);
            com.openexchange.guard.util.Core.sendFail(response, "Error retracting an email");
        }

    }

}
