/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2014 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.database;

import java.security.NoSuchAlgorithmException;
import java.security.spec.InvalidKeySpecException;

import javax.servlet.http.HttpServletRequest;

import org.apache.commons.codec.binary.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.gson.JsonObject;
import com.openexchange.guard.cacheitems.Cache;
import com.openexchange.guard.config.Config;
import com.openexchange.guard.config.OxConfiguration;
import com.openexchange.guard.encr.EncrLib;
import com.openexchange.guard.encr.GuardKeys;
import com.openexchange.guard.mailcreator.Mail;
import com.openexchange.guard.ox.Api;
import com.openexchange.guard.ox.InterOx;
import com.openexchange.guard.ox.Languages;
import com.openexchange.guard.server.OxCookie;
import com.openexchange.guard.server.Sender;
import com.openexchange.guard.validator.EmailValidator;

/**
 * Class for looking up public key, and creating keys if needed
 * 
 * @author greg
 */
public class GetPubKey {
	
	private static Logger logger = LoggerFactory.getLogger(GetPubKey.class);

    public RecipKey getPubKey(String email, int senderCID, String from, boolean sending, boolean remote) throws Exception {
        return (getPubKey(email, senderCID, from, sending, remote, "", ""));
    }

    /**
     * Looks up the public key for a recipient. If an OX member, creates if necessary For non-ox members, creates key only if sending
     * 
     * @param email
     * @param senderCID, only needed if not remote - used to create guest
     * @param sending Boolean if sending the email
     * @param remote True if from remote lookup - guest accounts wont be created
     * @return
     * @throws Exception
     */
    public RecipKey getPubKey(String email, int senderCID, String from, boolean sending, boolean remote, String guestlang, String fromName) throws Exception {

    	if (!EmailValidator.validate(email)) return (null); // make sure email address
        // Get connection
        RecipKey result = null;
        // First check if we have in our masterKeyTable
        result = checkMaster(email);

        // If not found, check to see if we have it in cache for outside user
        if ((result == null) && !remote) {
            result = checkCache(email);
        }

        if (result == null) {
            // If not in our keytable, and not in cache, then let's search the OX user database to see if member
            try {
                com.openexchange.guard.database.Utilities util = new com.openexchange.guard.database.Utilities();
                JsonObject userdat = util.getUser(email);
                if (userdat.has(email)) {
	                JsonObject user = userdat.get(email).getAsJsonObject();
	                result = new RecipKey();
	                result.userid = user.get("uid").getAsInt();
	                result.cid = user.get("cid").getAsInt();
	                result.lang = user.get("user").getAsJsonObject().get("language").getAsString();
	                result.name = user.get("user").getAsJsonObject().get("displayName").getAsString();
	                result.email = email;
	                if (sending) {
		                logger.debug("Found id for " + email + " in main ox users, creating key");
		                Keymanager keyman = new Keymanager();
		                result.newGuestPass = EncrLib.randomPass(Config.newPassLength);
		                Mail mf = new Mail();
		                Sender sender = new Sender();
		                int templId = OxConfiguration.getTemplateId(result.cid, result.userid);
		                JsonObject pwEmail = mf.getOxPasswordEmail(
	                            result.email,
	                            fromName,
	                            Config.passwordFromName.equals("") ? (Config.passwordFromAddress.equals("") ? result.name : Config.passwordFromAddress) : Config.passwordFromName,
	                            Config.passwordFromAddress.equals("") ? result.email : Config.passwordFromAddress,
	                            result.newGuestPass,
	                            result.lang,
	                            templId);
		                if (!sender.send(pwEmail, null)) return(null);
		                GuardKeys keys = keyman.createMasterKeys(result.userid, result.cid, email, result.name, result.newGuestPass, result.lang);
		                result.pubkey = (keys == null) ? null : keys.getPublic();
	                }
                }
            } catch (Exception ex) {
            	logger.error("Error getting public key for " + email, ex);
                result = null;
                return (result);
            }
        }

        // OK, local has failed. Lookup remote oxguard server
        if ((result == null) && !remote) {
            result = checkRemote(email, from);
        }

        if (remote)
            return (result); // If remote lookup, we don't create guest accounts, exit here.

        // Finally, if all other options exhausted, create new guest account
        if (result == null) {
            result = createGuest(email, senderCID, sending, guestlang);
            // Create guest account here
        }
        return (result);
    }

    public RecipKey checkMaster(String email) throws Exception {
        RecipKey result = null;
        // First check if we have in our masterKeyTable
        String command = "SELECT * FROM og_email WHERE email = ?";
        DbCommand com = new DbCommand(command);
        com.addVariables(email);
        DbQuery db = new DbQuery();
        db.readOG(com);
        if (db.next()) {
            int id = db.rs.getInt("id");
            int cid = db.rs.getInt("cid");
            int shard = db.rs.getInt("db");
            command = "select cid, id, RSAPublic, lang FROM og_KeyTable WHERE email = ?";
            com = new DbCommand(command);
            com.addVariables(email);
            db.read(com, id, cid);
            if (db.next()) {
                GuardKeys keys = new GuardKeys();
                OxResultSet rs = db.rs;
                try {
                    keys.setPublicKeyFrom64String(rs.getString("RSAPublic"));
                } catch (InvalidKeySpecException | NoSuchAlgorithmException e) {
                    logger.error("Error checking master key for " + email, e);
                    return(null);
                }
                result = new RecipKey();
                result.pubkey = keys.getPublic();
                result.userid = rs.getInt("id");
                result.cid = rs.getInt("cid");
                result.guest = result.cid < 0;
                result.lang = rs.getString("lang");
                if (result.lang == null)
                    result.lang = Languages.default_code;
                logger.debug("Found key for " + email + " in keytable, cid = " + result.cid);
            }
        }

        db.close();
        return (result);
    }

    private RecipKey checkCache(String email) throws Exception {
        RecipKey result = null;
        DbQuery db = new DbQuery();
        String command = "select RSAPublic FROM oxguard.cache WHERE email = ?";
        DbCommand com = new DbCommand(command);
        com.addVariables(email);
        db.readOG(com);
        if (db.next()) {
            GuardKeys keys = new GuardKeys();
            OxResultSet rs = db.rs;
            try {
                keys.setPublicKeyFrom64String(rs.getString("RSAPublic"));
            } catch (InvalidKeySpecException | NoSuchAlgorithmException e) {
                logger.error("Error checking cache for " + email, e);
            }
            result = new RecipKey();
            result.pubkey = keys.getPublic();
            result.userid = -1; // Set as negative number, as these are not local OX users.
            result.cid = -1;
            logger.debug("Found key for " + email + " in cache");
            result.remote = true;
        }
        db.close();
        return (result);
    }

    private RecipKey checkRemote(String email, String from) throws Exception {
        logger.debug("No key found for " + email + " trying remote");
        // try remote lookup and add to cache if found
        RecipKey result = null;
        InterOx iox = new InterOx();
        result = iox.getRemotePublicKey(email, from);
        if (result != null) {
            String command = "INSERT INTO cache VALUES (?, ?, NOW())";
            DbCommand com = new DbCommand(command);
            com.addVariables(email);
            com.addVariables(Base64.encodeBase64String(result.pubkey.getEncoded()));
            DbQuery db = new DbQuery();
            db.writeOxGuard(com);
            db.close();
        }
        return (result);
    }

    private RecipKey createGuest(String email, int senderCID, boolean sending, String guestlang) {
        RecipKey result = null;
        if (sending) {
            result = new RecipKey();
            Keymanager manager = new Keymanager();
            result.newGuestPass = EncrLib.randomPass(Config.newPassLength);
            GuardKeys keys = manager.createMasterKeys(0, (senderCID * -1), email, email, result.newGuestPass, Languages.isavail(guestlang));// Guest
                                                                                                                                            // account
                                                                                                                                            // is
                                                                                                                                            // going
                                                                                                                                            // to
                                                                                                                                            // have
                                                                                                                                            // negative
                                                                                                                                            // contextID
                                                                                                                                            // of
                                                                                                                                            // first
                                                                                                                                            // sender
            result.pubkey = (keys == null) ? null : keys.getPublic();
            result.userid = (keys == null) ? null : keys.userid;
            result.cid = (senderCID * -1);
            result.guest = true;
            result.lang = Languages.isavail(guestlang);
        } else {
            result = new RecipKey();
            result.pubkey = null;
            result.guest = true;
        }
        return (result);
    }
}
