/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.util.custom.i18n;

import static com.openexchange.util.custom.base.NullUtil.immutableCopyOf;
import java.util.HashMap;
import java.util.Map;
import com.openexchange.annotation.Nullable;
import com.openexchange.config.ConfigurationService;
import com.openexchange.config.PropertyFilter;
import com.openexchange.configuration.ConfigurationExceptionCodes;
import com.openexchange.exception.OXException;


/**
 * {@link PropertyTranslations}
 *
 * @author <a href="mailto:pascal.bleser@open-xchange.com">Pascal Bleser</a>
 * @since v7.8.2
 */
public enum PropertyTranslations {
    ;
    public static final PropertyTranslation load(final @Nullable ConfigurationService configService, final String propertyName) {
        if (configService == null) {
            return new InvalidPropertyTranslation(propertyName,
                notNull(ConfigurationExceptionCodes.INVALID_CONFIGURATION.create("ConfigurationService is null")));
        }
        
        final String propertyNamePrefix = propertyName + (propertyName.endsWith(".") ? "" : ".");
        final Map<String, String> properties;
        try {
            properties = new HashMap<String, String>(configService.getProperties(new PropertyFilter() {
                @Override
                public boolean accept(@Nullable String name, @Nullable String value) throws OXException {
                    return name != null && (name.equals(propertyName) || (name.startsWith(propertyNamePrefix) && name.length() > propertyNamePrefix.length()));
                }
            }));
        } catch (final OXException e) {
            return new InvalidPropertyTranslation(propertyName, e);
        }
        
        final String fallback;
        if (properties.containsKey(propertyName)) {
            fallback = notNull(properties.remove(propertyName));
        } else {
            return new InvalidPropertyTranslation(propertyName, notNull(
                ConfigurationExceptionCodes.PROPERTY_MISSING.create(propertyName)));
        }
        
        final Map<String, String> translations = new HashMap<>(properties.size());
        for (final Map.Entry<String, String> e : properties.entrySet()) {
            final String key = e.getKey();
            if (key != null) {
                final String lang = key.length() > propertyNamePrefix.length() ? key.substring(propertyNamePrefix.length()) : null;
                if (lang != null) {
                    final String value = e.getValue();
                    if (value == null || value.isEmpty() || value.trim().isEmpty()) {
                        return new InvalidPropertyTranslation(propertyName, notNull(ConfigurationExceptionCodes.INVALID_CONFIGURATION
                            .create(String.format("property '%s' is empty", value))));
                    }
                    translations.put(lang, value);
                }
            }
        }
        
        return new ValidPropertyTranslation(propertyName, immutableCopyOf(translations), fallback);
    }    

    private static final <X> X notNull(final @Nullable X value) {
        if (value != null) {
            return value;
        }
        throw new IllegalArgumentException("value is null");
    }
    
}
