/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.util.activator.impl;

import static com.openexchange.util.custom.base.NullUtil.immutableCopyOf;
import java.lang.annotation.Annotation;
import java.lang.reflect.Constructor;
import java.lang.reflect.Method;
import java.util.Dictionary;
import java.util.List;
import java.util.Map;
import javax.annotation.concurrent.Immutable;
import com.google.common.base.Optional;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.openexchange.annotation.Nullable;
import com.openexchange.osgi.ServiceListing;
import com.openexchange.osgi.ServiceSet;
import com.openexchange.util.activator.ServiceDependencyResolver;
import com.openexchange.util.activator.ServiceRef;
import com.openexchange.util.custom.collect.ImmutableDictionary;


/**
 * {@link ServiceDependencyResolverImpl}
 *
 * @author <a href="mailto:pascal.bleser@open-xchange.com">Pascal Bleser</a>
 * @since v7.8.2
 */
@Immutable
public final class ServiceDependencyResolverImpl implements ServiceDependencyResolver {

    private final ImmutableMap<MandatoryServiceDependency<?>, Object> mandatoryDependencies;
    private final ImmutableList<OptionalServiceDependency<?>> optionalDependencies;
    private final ImmutableMap<Class<?>, ServiceSet<?>> serviceSets;
    private final ImmutableMap<Class<?>, ServiceListing<?>> serviceListings;
    private final ImmutableDictionary<String, ?> properties;
    
    public ServiceDependencyResolverImpl(Map<MandatoryServiceDependency<?>, Object> mandatoryDependencies,
        List<OptionalServiceDependency<?>> optionalDependencies,
        Map<Class<?>, ServiceSet<?>> serviceSets,
        Map<Class<?>, ServiceListing<?>> serviceListings,
        Map<String, ?> properties) {
        this.mandatoryDependencies = immutableCopyOf(mandatoryDependencies);
        this.optionalDependencies = immutableCopyOf(optionalDependencies);
        this.serviceSets = immutableCopyOf(serviceSets);
        this.serviceListings = immutableCopyOf(serviceListings);
        this.properties = ImmutableDictionary.copyOf(properties);
    }

    public ServiceDependencyResolverImpl(Map<MandatoryServiceDependency<?>, Object> mandatoryDependencies,
        List<OptionalServiceDependency<?>> optionalDependencies,
        Map<Class<?>, ServiceSet<?>> serviceSets,
        Map<Class<?>, ServiceListing<?>> serviceListings,
        Dictionary<String, ?> properties) {
        this.mandatoryDependencies = immutableCopyOf(mandatoryDependencies);
        this.optionalDependencies = immutableCopyOf(optionalDependencies);
        this.serviceSets = immutableCopyOf(serviceSets);
        this.serviceListings = immutableCopyOf(serviceListings);
        this.properties = ImmutableDictionary.copyOf(properties);
    }
    
    @SuppressWarnings("unchecked")
    @Override
    @Nullable
    public <X> X resolveMandatoryService(final Class<X> serviceClass, final @Nullable Annotation[] annotations) {
        for (final Map.Entry<MandatoryServiceDependency<?>, Object> e : mandatoryDependencies.entrySet()) {
            if (e.getKey().matchesDependency(serviceClass, annotations)) {
                return (X) e.getValue();
            }
        }
        return null;
    }
    
    @SuppressWarnings("unchecked")
    @Override
    @Nullable
    public <X> ServiceRef<X> resolveOptionalService(final Class<X> serviceClass, final @Nullable Annotation[] annotations) {
        for (final OptionalServiceDependency<?> dep : optionalDependencies) {
            if (dep.matchesDependency(serviceClass, annotations)) {
                return (ServiceRef<X>) dep.serviceRef();
            }
        }
        return null;
    }

    @Override
    public Optional<Object[]> resolveParameters(Constructor<?> constructor) {
        return RegistrationTools.pickDependencies(
            constructor,
            this.mandatoryDependencies,
            this.optionalDependencies,
            this.serviceSets,
            this.serviceListings,
            this.properties
        );
    }

    @Override
    public Optional<Object[]> resolveParameters(Method method) {
        return RegistrationTools.pickDependencies(
            method,
            this.mandatoryDependencies,
            this.optionalDependencies,
            this.serviceSets,
            this.serviceListings,
            this.properties
        );
    }
}
