/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.util.activator.impl;

import static com.openexchange.util.custom.base.NullUtil.f;
import static com.openexchange.util.custom.base.NullUtil.immutableSet;
import java.lang.annotation.Annotation;
import java.util.Set;
import org.osgi.framework.ServiceReference;
import com.google.common.collect.ImmutableSet;
import com.openexchange.annotation.Nullable;
import com.openexchange.util.activator.Service;

public enum ServiceDependencyQualifiers implements ServiceDependencyQualifier {
    NO_ID_QUALIFIER {

        @Override
        public @Nullable String contributeFilter() {
            return "(!(id=*))";
        }

        @Override
        public @Nullable String contributeIdentification() {
            return "no-id";
        }

        @Override
        public @Nullable Set<String> contributePropertiesOfInterest() {
            return ID_PROPERTY;
        }

        @Override
        public boolean matchesService(Object serviceInstance, ServiceReference<?> reference) {
            return reference.getProperty("id") == null;
        }

        @Override
        public boolean matchesDependency(Class<?> serviceClass, @Nullable Annotation[] annotations) {
            return true;
        }
        
    },
    
    ANY_ID_QUALIFIER {

        @Override
        public @Nullable String contributeFilter() {
            return null;
        }

        @Override
        public @Nullable String contributeIdentification() {
            return "id=*";
        }

        @Override
        public @Nullable Set<String> contributePropertiesOfInterest() {
            return ImmutableSet.of();
        }

        @Override
        public boolean matchesService(Object serviceInstance, ServiceReference<?> reference) {
            return true;
        }

        @Override
        public boolean matchesDependency(Class<?> serviceClass, @Nullable Annotation[] annotations) {
            return true;
        }
        
    }
    ;
    
    private static final ImmutableSet<String> ID_PROPERTY = immutableSet("id");
    
    private static class IdQualifier implements ServiceDependencyQualifier {
        
        private final String id;
        
        public IdQualifier(String id) {
            this.id = id;
        }

        @Override
        public @Nullable String contributeFilter() {
            return f("(id=%s)", id);
        }

        @Override
        public @Nullable String contributeIdentification() {
            return f("id=%s", id);
        }

        @Override
        public @Nullable Set<String> contributePropertiesOfInterest() {
            return ID_PROPERTY;
        }

        @Override
        public boolean matchesService(Object serviceInstance, ServiceReference<?> reference) {
            return this.id.equals(reference.getProperty("id"));
        }

        @Override
        public boolean matchesDependency(Class<?> serviceClass, @Nullable Annotation[] annotations) {
            final @Nullable Service serviceAnnotation = RegistrationTools.serviceAnnotation(annotations);
            return serviceAnnotation != null && serviceAnnotation.id().equals(this.id);
        }
        
    }
    
    public static final IdQualifier mustMatchId(final String id) {
        return new IdQualifier(id);
    }
}
