/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.util.custom.config;

import org.slf4j.Logger;
import com.openexchange.annotation.Nullable;
import com.openexchange.config.ConfigurationService;

/**
 * {@link LoggingConfigHolder}
 *
 * @author <a href="mailto:pascal.bleser@open-xchange.com">Pascal Bleser</a>
 * @since v1.2.0
 */
public final class LoggingConfigHolder<C> extends ConfigHolder<C> {

    public static final <X> LoggingConfigHolder<X> with(final Logger log,
        final String name,
        final @Nullable ConfigurationService configurationService,
        final ConfigLoader<X> loader) {
        return new LoggingConfigHolder<X>(log, name, configurationService, loader);
    }

    public static final <X> LoggingConfigHolder<X> with(final Logger log,
        final String name,
        final String prefix,
        final @Nullable ConfigurationService configurationService,
        final ConfigLoader<X> loader) {
        return new LoggingConfigHolder<X>(log, name, prefix, configurationService, loader);
    }
    
    private final Logger log;
    private final String name;
    
    public LoggingConfigHolder(final Logger log,
        final String name,
        final String prefix,
        final @Nullable ConfigurationService configurationService,
        final ConfigLoader<C> loader) {
        super(configurationService, prefix, loader);
        this.log = log;
        this.name = name;
    }
    
    public LoggingConfigHolder(final Logger log,
        final String name,
        final @Nullable ConfigurationService configurationService,
        final ConfigLoader<C> loader) {
        super(configurationService, loader);
        this.log = log;
        this.name = name;
    }

    /**
     * Override this method if you want to be signalled when the configuration
     * has been reloaded and has changed.
     * <p>
     * The default implementation logs an info message.
     * 
     * @param newConfig the new configuration that is now active
     * @since 1.2.0
     */
    @Override
    protected void onConfigChanged(final C newConfig) {
        log.info("loaded new {}", name);
    }

    /**
     * Override this method if you want to be signalled when the configuration
     * has been reloaded and has not changed.
     * <p>
     * The default implementation logs a debug message.
     * @since 1.2.0
     */
    @Override
    protected void onConfigUnchanged() {
        log.debug("{} unchanged", name);
    }

    /**
     * Override this method if you want to be signalled when the configuration
     * has been reloaded and the new configuration is not valid.
     * <p>
     * The default implementation logs an error.
     * @since 1.2.0
     */
    @Override
    protected void onConfigInvalid() {
        log.error("failed to load {}, keeping previous values", name);
    }

}
