/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.util.custom.base;

import static com.openexchange.util.custom.base.NullUtil.absent;
import static com.openexchange.util.custom.base.NullUtil.optional;
import com.google.common.base.CharMatcher;
import com.google.common.base.Optional;
import com.openexchange.annotation.Nullable;

public enum StringUtil {
    ;
    
    @SuppressWarnings("null")
    public static String trimWhitespaces(final String string) {
        return CharMatcher.whitespace().trimFrom(string);
    }
    public static @Nullable String trimWhitespacesOrNull(final @Nullable String string) {
        return string != null ? trimWhitespaces(string) : null;
    }
    @SuppressWarnings("null")
    public static String rightTrimWhitespaces(final String string) {
        return CharMatcher.whitespace().trimTrailingFrom(string);
    }
    public static String trimTrailingWhitespaces(final String string) {
        return rightTrimWhitespaces(string);
    }
    public static @Nullable String rightTrimWhitespacesOrNull(final @Nullable String string) {
        return string != null ? rightTrimWhitespaces(string) : null;
    }
    @SuppressWarnings("null")
    public static String leftTrimWhitespaces(final String string) {
        return CharMatcher.whitespace().trimLeadingFrom(string);
    }
    public static String trimLeadingWhitespaces(final String string) {
        return leftTrimWhitespaces(string);
    }
    public static @Nullable String leftTrimWhitespacesOrNull(final @Nullable String string) {
        return string != null ? leftTrimWhitespaces(string) : null;
    }
    
    public static boolean isEmptyOrWhitespaces(final @Nullable String string) {
        return string == null || CharMatcher.whitespace().trimFrom(string).isEmpty();
    }
    public static String orIfEmpty(final @Nullable String string, final String defaultValue) {
        if (string == null || isEmptyOrWhitespaces(string)) {
            return defaultValue;
        } else {
            return string;
        }
    }
    
    public static Optional<String> fromEmptyOrWhitespaces(final @Nullable String string) {
        if (string == null || isEmptyOrWhitespaces(string)) {
            return absent();
        } else {
            return optional(string);
        }
    }

}
