/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.util.activator.impl;

import static com.openexchange.util.custom.base.NullUtil.build;
import static com.openexchange.util.custom.base.NullUtil.className;
import static com.openexchange.util.custom.base.NullUtil.emptySet;
import java.util.Dictionary;
import javax.servlet.http.HttpServlet;
import org.osgi.framework.BundleContext;
import org.osgi.service.http.HttpService;
import org.osgi.util.tracker.ServiceTrackerCustomizer;
import com.google.common.collect.ImmutableSet;
import com.openexchange.annotation.Nullable;
import com.openexchange.config.ConfigurationService;
import com.openexchange.util.activator.DependencyProvider;
import com.openexchange.util.activator.ServiceDependencyResolver;
import com.openexchange.util.activator.ServiceRegistrationHandler;


/**
 * {@link ServletInstanceServiceWithDependencies}
 *
 * @author <a href="mailto:pascal.bleser@open-xchange.com">Pascal Bleser</a>
 * @since v1.0.0
 */
public class ServletInstanceServiceWithDependencies<T extends HttpServlet> extends ServiceInstanceServiceWithDependencies<HttpServlet, T> {

    private static final ImmutableSet<MandatoryServiceDependency<?>> deps(final String path) {
        final ImmutableSet.Builder<MandatoryServiceDependency<?>> b = ImmutableSet.builder();
        b.add(new MandatoryServiceDependency<>(HttpService.class));
        if (path.contains("${")) {
            b.add(new MandatoryServiceDependency<>(ConfigurationService.class));
        }
        return build(b);
    }
    
    private final String path;
    
    public ServletInstanceServiceWithDependencies(final T serviceInstance, final String path) {
        super(serviceInstance, HttpServlet.class, deps(path), null);
        this.path = path;
    }

    @Override
    protected ServiceTrackerCustomizer<HttpServlet, T> createRegisterer(final BundleContext bundleContext,
        final ActivatorRegistry registry,
        final ImmutableSet<Class<?>> serviceClasses,
        final Dictionary<String, ?> properties,
        final @Nullable ServiceRegistrationHandler<T> serviceRegistrationHandler) {
        return new ServletServiceRegistererTemplate<T>(path,
            bundleContext,
            serviceClasses,
            className(serviceInstance),
            properties,
            mandatoryDependencies,
            optionalDependencies,
            RegistrationTools.serviceSets(bundleContext, registry, emptySet()),
            RegistrationTools.serviceListings(bundleContext, registry, emptySet()),
            RegistrationTools.collectPropertiesOfInterest(mandatoryDependencies, optionalDependencies)) {
            @Override
            protected T createInstance(final ServiceDependencyResolver resolver)
                throws Exception {
                if (serviceInstance instanceof DependencyProvider) {
                    ((DependencyProvider) serviceInstance).onDependenciesAvailable(resolver);
                }
                if (serviceRegistrationHandler != null) {
                    serviceRegistrationHandler.onServiceRegistered(serviceInstance);
                }
                return serviceInstance;
            }
        };
    }

}
