/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.util.activator.impl;

import static com.openexchange.util.custom.base.NullUtil.immutableSetCopyOf;
import java.lang.reflect.Constructor;
import java.util.Dictionary;
import java.util.HashSet;
import java.util.Set;
import javax.annotation.concurrent.Immutable;
import javax.servlet.http.HttpServlet;
import org.osgi.framework.BundleContext;
import org.osgi.service.http.HttpService;
import org.osgi.util.tracker.ServiceTrackerCustomizer;
import com.google.common.collect.ImmutableSet;
import com.openexchange.annotation.Nullable;
import com.openexchange.config.ConfigurationService;
import com.openexchange.util.activator.ServiceRegistrationHandler;

/**
 * {@link ServletClassServiceWithDependencies}
 *
 * @author <a href="mailto:pascal.bleser@open-xchange.com">Pascal Bleser</a>
 * @since v1.0.2
 * @param <S>
 * @param <T>
 */
@Immutable
public final class ServletClassServiceWithDependencies<T extends HttpServlet> extends ClassServiceWithDependenciesTemplate<HttpServlet, T> {
    
    public static final <Y extends HttpServlet> ServiceWithDependencies<Y> find(final Class<Y> serviceInstanceClass, final String path) {
        final DependencyScan<HttpServlet, Y> scan = scan(serviceInstanceClass);
        
        final Set<MandatoryServiceDependency<?>> mandatoryDeps = new HashSet<>();
        mandatoryDeps.add(new MandatoryServiceDependency<>(HttpService.class));
        if (path.contains("${")) {
            mandatoryDeps.add(new MandatoryServiceDependency<>(ConfigurationService.class));
        }
        mandatoryDeps.addAll(scan.mandatoryDependencies);
        
        return new ServletClassServiceWithDependencies<Y>(path,
            scan.constructor,
            serviceInstanceClass,
            immutableSetCopyOf(mandatoryDeps),
            scan.optionalDependencies,
            scan.serviceSets,
            scan.serviceListings
        );    
    }
    
    private final String path;
    
    public ServletClassServiceWithDependencies(final String path,
        final Constructor<T> constructor,
        final Class<T> serviceInstanceClass,
        final ImmutableSet<MandatoryServiceDependency<?>> mandatoryDependencies,
        final ImmutableSet<OptionalServiceDependency<?>> optionalDependencies,
        final ImmutableSet<Class<?>> serviceSets,
        final ImmutableSet<Class<?>> serviceListings) {
        super(constructor, serviceInstanceClass, HttpServlet.class, mandatoryDependencies, optionalDependencies, serviceSets, serviceListings);
        this.path = path;
    }

    @Override
    protected ServiceTrackerCustomizer<HttpServlet, T> createRegisterer(final BundleContext bundleContext,
        final ActivatorRegistry registry,
        final ImmutableSet<Class<?>> serviceClasses,
        final Dictionary<String, ?> properties,
        final @Nullable ServiceRegistrationHandler<T> serviceRegistrationHandler) {
        return new ServletServiceRegisterer<T>(path,
            bundleContext,
            serviceClasses,
            serviceInstanceClass,
            properties,
            mandatoryDependencies,
            optionalDependencies,
            RegistrationTools.serviceSets(bundleContext, registry, serviceSets),
            RegistrationTools.serviceListings(bundleContext, registry, serviceListings),
            RegistrationTools.collectPropertiesOfInterest(mandatoryDependencies, optionalDependencies),
            constructor,
            serviceRegistrationHandler
        );        
    }
    
}
