/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.util.activator.impl;

import static com.openexchange.util.activator.impl.RegistrationTools.TO_SIMPLE_CLASS_NAME;
import static com.openexchange.util.custom.base.NullUtil.className;
import static com.openexchange.util.custom.base.NullUtil.f;
import static com.openexchange.util.custom.base.NullUtil.logger;
import java.util.Collection;
import java.util.Dictionary;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;
import javax.annotation.concurrent.ThreadSafe;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceRegistration;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.openexchange.annotation.Nullable;
import com.openexchange.osgi.ExceptionUtils;
import com.openexchange.osgi.ServiceListing;
import com.openexchange.osgi.ServiceSet;
import com.openexchange.util.activator.ServiceDependencyResolver;

/**
 * Template for registering service instances.
 *
 * @author <a href="mailto:pascal.bleser@open-xchange.com">Pascal Bleser</a>
 * @author <a href="mailto:marcus.klein@open-xchange.com">Marcus Klein</a>
 */
@ThreadSafe
public abstract class InstanceFactoryServiceAwareDependentServiceRegistererTemplate<S, T extends S> extends ServiceAwareDependentServiceRegistererTemplate<S, T> {

    private static final org.slf4j.Logger LOG = logger(InstanceFactoryServiceAwareDependentServiceRegistererTemplate.class);
    
    private final AtomicReference<ServiceRegistrationSet<T>> registrationRef;
    
    public InstanceFactoryServiceAwareDependentServiceRegistererTemplate(
        BundleContext context,
        ImmutableSet<Class<?>> serviceTypes,
        String serviceInstanceName,
        Dictionary<String, ?> properties,
        Collection<MandatoryServiceDependency<?>> mandatoryServices,
        Collection<OptionalServiceDependency<?>> optionalServices,
        ImmutableMap<Class<?>, ServiceSet<?>> serviceSets,
        ImmutableMap<Class<?>, ServiceListing<?>> serviceListings,
        Set<String> propertiesOfInterest) {
        super(context, serviceTypes, serviceInstanceName, properties, mandatoryServices, optionalServices,
            serviceSets, serviceListings, propertiesOfInterest);
        this.registrationRef = new AtomicReference<ServiceRegistrationSet<T>>(null);
    }

    protected abstract T createInstance(final ServiceDependencyResolver resolver) throws Exception;

    @Override
    protected @Nullable Object register(final @Nullable ServiceDependencyResolver resolver) {
        if (resolver == null) {
            throw new IllegalArgumentException(f("%s parameter is null", ServiceDependencyResolver.class.getName()));
        }
        try {
            final T serviceInstance = createInstance(resolver);
            if (LOG.isTraceEnabled()) {
                LOG.trace("registering service {} ({})",
                    className(serviceInstance),
                    serviceTypes.stream().map(TO_SIMPLE_CLASS_NAME).collect(Collectors.joining(", ")));
            }
            final ServiceRegistration<?> r = RegistrationTools
                .registerIntoContext(context, serviceTypes, serviceInstance, properties);
            registrationRef.set(new ServiceRegistrationSet<T>(serviceInstance, r));
            return serviceInstance;
        } catch (Throwable t) {
            ExceptionUtils.handleThrowable(t);
            LOG.error(f("failed to register %s (%s)",
                serviceInstanceName,
                serviceTypes.stream().map(TO_SIMPLE_CLASS_NAME).collect(Collectors.joining(", "))),
                t);
            return null;
        }
    }
    
    @Override
    protected void unregister(final Object service) {
        final ServiceRegistrationSet<T> unregister = registrationRef.getAndSet(null);
        if (LOG.isTraceEnabled()) {
            LOG.trace("Unregistering service {} ({})",
                serviceInstanceName,
                serviceTypes.stream().map(TO_SIMPLE_CLASS_NAME).collect(Collectors.joining(", "))
            );
        }
        
        if (unregister != null) {
            unregister.registration.unregister();
            RegistrationTools.closeInstance(unregister.registeredService, serviceInstanceName, serviceTypes);
        }
    }

}
