/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.session.tasks;

import java.util.HashSet;
import java.util.Set;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.usm.api.database.StorageAccessException;
import com.openexchange.usm.api.session.assets.ContextSessionID;
import com.openexchange.usm.session.impl.SessionImpl;
import com.openexchange.usm.session.impl.SessionManagerImpl;

/**
 * {@link USMSessionCleanupTask}
 *
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 */
public class USMSessionCleanupTask implements Runnable {

    private static final Logger log = LoggerFactory.getLogger(USMSessionCleanupTask.class);

    /**
     * Initial delay (in seconds) after startup of SessionManager/TimerService after which the cleanup task is first performed
     * 
     * Default: 180000 milliseconds (3 minutes)
     */
    public static final long INITIAL_DELAY = 1000L * 60L * 3L; // 3 minutes

    /**
     * Interval (in milliseconds) between executions of the cleanup task
     * 
     * Default: 86400000 milliseconds (1 day)
     */
    public static final long EXECUTION_INTERVAL = 24L * 60L * 60L * 1000L; // 1 day

    private final SessionManagerImpl _sessionManager;

    /**
     * Initializes a new {@link USMSessionCleanupTask}.
     * 
     * @param sessionManager
     */
    public USMSessionCleanupTask(SessionManagerImpl sessionManager) {
        _sessionManager = sessionManager;
    }

    /*
     * (non-Javadoc)
     * 
     * @see java.lang.Runnable#run()
     */
    @Override
    public void run() {
        log.debug("Checking for old USM sessions");
        try {
            // TODO: Implement something more efficient (e.g. LRU mechanism)
            int count = 0;
            Set<Integer> remainingContextIDs = new HashSet<Integer>(_sessionManager.getSessionDataStorage().getAllContextIDs());
            while (!remainingContextIDs.isEmpty()) {
                int context = remainingContextIDs.iterator().next();
                count += checkForOldSessions(context, remainingContextIDs);
                for (int contextId : _sessionManager.getSessionDataStorage().getContextsInSameSchema(context)) {
                    remainingContextIDs.remove(contextId);
                }
            }
            log.info("Checked for old USM sessions, " + count + " were removed");
        } catch (StorageAccessException e) {
            log.error("Couldn't access DB to check for old USM sessions", e);
        } catch (Exception e) {
            log.error("Uncaught error occurred while checking for old USM sessions", e);
        }
    }

    /**
     * Check db for old USM sessions for the specified context ID
     * 
     * @param cid
     * @return
     * @throws StorageAccessException
     */
    private int checkForOldSessions(int cid, Set<Integer> validContextIDs) throws StorageAccessException {
        log.debug("Checking for old USM sessions in same schema as context " + cid);
        Set<ContextSessionID> sessionIDsToRemove = _sessionManager.getSessionDataStorage().getSessionsToRemove(cid, _sessionManager.getSessionStorageTimeLimit(), validContextIDs);

        // Check if any of the sessions to remove is currently active in the SessionManager, do not remove those sessions
        if (!sessionIDsToRemove.isEmpty()) {
            for (SessionImpl session : _sessionManager.getSessionStorage().getSessionList()) {
                sessionIDsToRemove.remove(new ContextSessionID(session.getContextId(), session.getSessionId()));
            }
        }

        int count = _sessionManager.getSessionDataStorage().purgeSessions(cid, sessionIDsToRemove);
        log.debug(count + " USM sessions removed in same schema as context " + cid);
        return count;
    }
}
