/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.session.sync;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import com.openexchange.usm.api.session.assets.SessionID;

/**
 * {@link SynchronizationLock}
 * 
 * @author <a href="mailto:afe@microdoc.de">Alexander Feess</a>
 */
public final class SynchronizationLock {

    private static final SimpleDateFormat _LOCK_TIME_FORMAT = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSS Z");

    private static class LockKey {

        private final SessionID _sessionID;

        private final String _id;

        public LockKey(SessionID sessionID, String id) {
            _sessionID = sessionID;
            _id = id;
        }

        @Override
        public int hashCode() {
            return _sessionID.hashCode() * 31 + _id.hashCode();
        }

        @Override
        public boolean equals(Object obj) {
            if (this == obj)
                return true;
            if (!(obj instanceof LockKey))
                return false;
            LockKey other = (LockKey) obj;
            return _id.equals(other._id) && _sessionID.equals(other._sessionID);
        }
    }

    private static class LockValue {

        private final String _acquirer;

        private final CountDownLatch _countdownLatch;

        public LockValue(String acquirer) {
            _acquirer = acquirer;
            _countdownLatch = new CountDownLatch(1);
        }

        public String getAcquirer() {
            return _acquirer;
        }

        public CountDownLatch getCountdownLatch() {
            return _countdownLatch;
        }
    }

    private static long _lockTimeout;

    private static final ConcurrentHashMap<LockKey, LockValue> _lockedKeys = new ConcurrentHashMap<LockKey, LockValue>(16, 0.75f, 1);

    /**
     * Configures the timeout to use for acquiring locks on specific session locks.
     * 
     * @param timeout timeout in milliseconds: The lock() method will wait for upto this time when trying to gain exclusive access to the specified lock
     */
    public static void setLockTimeout(long timeout) {
        _lockTimeout = timeout;
    }

    /**
     * Tries to acquire a lock for a given ID in a given session. Locks are acquired per session per ID. For different sessions and/or IDs a
     * different lock is used. If acquiring the lock is successful, returns null. If it is not successful, returns the "acquirer" text that
     * was stored when the lock was acquired by someone else.
     * 
     * @param session Session for which to acquire the lock
     * @param id ID for which to acquire the lock
     * @param acquirer Text that will be stored with the lock (anybody else that fails to acquire the lock while it is held will get this
     *            String which may contain some description who/why this lock was acquired)
     * @return null if lock was acquired successfully, String stored as acquirer by the code that currently holds the lock if the lock is
     *         currently not available
     */
    public static String lock(SessionID sessionID, String id, String acquirer) {
        LockKey key = new LockKey(sessionID, id);
        LockValue value = new LockValue(buildAcquirerWithDateTime(acquirer));
        LockValue oldValue = _lockedKeys.putIfAbsent(key, value);
        if (oldValue == null)
            return null;
        long now = System.currentTimeMillis();
        long limit = now + _lockTimeout;
        for (; now < limit; now = System.currentTimeMillis()) {
            try {
                if (oldValue.getCountdownLatch().await(limit - now, TimeUnit.MILLISECONDS)) {
                    oldValue = _lockedKeys.putIfAbsent(key, value);
                    if (oldValue == null)
                        return null;
                }
            } catch (InterruptedException e) {
                break;
            }
        }
        return oldValue.getAcquirer();
    }

    private static String buildAcquirerWithDateTime(String acquirer) {
        synchronized (_LOCK_TIME_FORMAT) {
            return acquirer + " (since " + _LOCK_TIME_FORMAT.format(new Date()) + ')';
        }
    }

    /**
     * Releases a previously acquired lock. No security check is made if the lock is actually currently locked. After the call the
     * session/ID combination is available for locking.
     * 
     * @param session
     * @param id
     */
    public static void unlock(SessionID sessionID, String id) {
        LockValue lock = _lockedKeys.remove(new LockKey(sessionID, id)); // Remove lock from table
        if (lock != null)
            lock.getCountdownLatch().countDown(); // Notify any threads that were waiting on lock
    }
}
