/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.session.cache;

import com.openexchange.usm.api.cache.SyncStateCache;
import com.openexchange.usm.api.cache.SyncStateCacheProvider;
import com.openexchange.usm.api.session.assets.CompleteSessionID;
import com.openexchange.usm.session.impl.SessionImpl;
import com.openexchange.usm.session.impl.SessionManagerImpl;
import com.openexchange.usm.session.impl.SessionReference;

/**
 * {@link MemorySyncStateCacheProvider} Default provider for the SyncState cache, uses simple in-memory storage, creates 1 simple cache per USM session.
 * 
 * @author <a href="mailto:afe@microdoc.de">Alexander Feess</a>
 */
public class MemorySyncStateCacheProvider implements SyncStateCacheProvider {

    private final SessionManagerImpl _sessionManager;

    public MemorySyncStateCacheProvider(SessionManagerImpl sessionManager) {
        _sessionManager = sessionManager;
    }

    @Override
    public SyncStateCache getCache(CompleteSessionID session) {
        return new MemorySyncStateCache(_sessionManager.getMaxSyncStatesInDB());
    }

    @Override
    public void freeCache(CompleteSessionID session) {
        // Nothing to do, cache will be garbage collected
    }

    @Override
    public void removeCacheCompletely(CompleteSessionID session) {
        // Nothing to do, cache will be garbage collected
    }

    @Override
    public int getSyncStateCount() {
        int count = 0;
        for (SessionReference reference : _sessionManager.getSessionStorage().getSessionReferences()) {
            SessionImpl session = reference.poll();
            if (session != null) {
                SyncStateCache syncStateCache = session.getDataObjectCache().getSyncStateCache();
                if (syncStateCache instanceof MemorySyncStateCache)
                    count += ((MemorySyncStateCache) syncStateCache).getSyncStateCount();
            }
        }
        return count;
    }

    @Override
    public int removeUnusedSyncStates(long limit) {
        int count = 0;
        for (SessionReference reference : _sessionManager.getSessionStorage().getSessionReferences()) {
            SessionImpl session = reference.poll();
            if (session != null) {
                SyncStateCache syncStateCache = session.getDataObjectCache().getSyncStateCache();
                if (syncStateCache instanceof MemorySyncStateCache)
                    count += ((MemorySyncStateCache) syncStateCache).removeUnusedSyncStates(limit);
            }
        }
        return count;
    }
}
