/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */
package com.openexchange.usm.connector.commands;

import static com.openexchange.usm.connector.commands.CommandConstants.FOLDERID;
import static com.openexchange.usm.connector.commands.CommandConstants.OBJECTIDS;
import static com.openexchange.usm.connector.commands.CommandConstants.PROPERTIES;
import static com.openexchange.usm.connector.commands.CommandConstants.SESSIONID;
import static com.openexchange.usm.connector.commands.CommandConstants.SYNCID;
import java.util.ArrayList;
import java.util.List;
import javax.servlet.http.HttpServletRequest;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.usm.api.datatypes.DataType;
import com.openexchange.usm.api.exceptions.InvalidUUIDException;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.Folder;
import com.openexchange.usm.connector.exceptions.DataObjectNotFoundException;
import com.openexchange.usm.connector.exceptions.MultipleOperationsOnDataObjectException;
import com.openexchange.usm.json.ConnectorBundleErrorCodes;
import com.openexchange.usm.json.USMJSONAPIException;
import com.openexchange.usm.json.USMJSONServlet;
import com.openexchange.usm.json.response.ResponseObject;
import com.openexchange.usm.json.response.ResponseStatusCode;
import com.openexchange.usm.session.dataobject.DataObjectSet;
import com.openexchange.usm.util.UUIDToolkit;

/**
 * Handler for the ReadData USM-JSON-Command.
 * @author ldo
 *
 */
public class ReadDataHandler extends NormalCommandHandler {
	private static final String UUID = "uuid";
	private static final String[] REQUIRED_PARAMETERS = { SESSIONID, SYNCID, FOLDERID, OBJECTIDS, PROPERTIES };
	private static final String[] OPTIONAL_PARAMETERS = {};

    public ReadDataHandler(USMJSONServlet servlet, HttpServletRequest request) throws USMJSONAPIException {
        super(servlet, request);
    }

	@Override
	public ResponseObject handleRequest() throws USMJSONAPIException {
		String folderID = getStringParameter(FOLDERID, null);
		long syncId = getSyncID();
		JSONArray objectids = getJSONArray(_parameters, OBJECTIDS);
		JSONArray properties = getJSONArray(_parameters, PROPERTIES);
		List<String> fields = createFieldsListFromParameter(properties);
		DataObject[] resultElements = readDataElements(syncId, folderID, objectids, fields);
		JSONArray data = createResponseData(fields, resultElements);
		return new ResponseObject(ResponseStatusCode.SUCCESS, data);
	}

	private List<String> createFieldsListFromParameter(JSONArray properties) throws USMJSONAPIException {
		int length = properties.length();
		List<String> fields = new ArrayList<String>(length);
		for (int i = 0; i < length; i++) {
			try {
				fields.add(properties.getString(i));
			} catch (JSONException e) {
				throw USMJSONAPIException.createJSONError(ConnectorBundleErrorCodes.READ_DATA_JSON_ERROR_2, e);
			}
		}
		return fields;
	}

	private JSONArray createResponseData(List<String> fields, DataObject[] resultElements) throws USMJSONAPIException {
		JSONArray data = new JSONArray();
		if (resultElements != null) {
			for (DataObject dataObject : resultElements) {
				try {
					data.put(storeRequestedFieldsInJSONObject(dataObject, fields));
				} catch (JSONException e) {
					throw USMJSONAPIException.createJSONError(ConnectorBundleErrorCodes.READ_DATA_JSON_ERROR, e);
				}
			}
		}
		return data;
	}

    private DataObject[] readDataElements(long syncID, String folderID, JSONArray objectids, List<String> fields) throws USMJSONAPIException {
        try {
            Folder folder = getFolderByUUID(folderID);
			DataObjectSet cachedFolderElements = new DataObjectSet(_session.getCachedFolderElements(folder.getID(), folder
					.getElementsContentType(), syncID));
            List<DataObject> resultElements = new ArrayList<DataObject>();
            String[] fieldsArray = fields.toArray(new String[fields.size()]);
            int length = objectids.length();
            for (int i = 0; i < length; i++) {
                String objectUUID = objectids.getString(i);
                try {
                    java.util.UUID uuid = UUIDToolkit.extractUUIDFromString(objectUUID);
                    String oxObjectID = getDataObjectByUUID(cachedFolderElements, uuid).getID();
                    DataObject readObject = _session.readDataObject(folder.getID(), oxObjectID, fieldsArray);
                    if (readObject != null) {
                        if (readObject.getUUID() == null)
                            readObject.setUUID(uuid);
                        resultElements.add(readObject);
                    }
                } catch (DataObjectNotFoundException e) {
                    throw USMJSONAPIException.createObjectNotFoundException(e);
                } catch (MultipleOperationsOnDataObjectException e) {
                    throw USMJSONAPIException.createMultipleOperationsOnDataObjectException(e);
                } catch (InvalidUUIDException e) {
                    throw USMJSONAPIException.createInvalidUUIDException(e);
                }
            }
            return resultElements.toArray(new DataObject[resultElements.size()]);
        } catch (USMJSONAPIException e) {
            throw e;
        } catch (USMException e) {
            throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.GET_CACHED_DATA_GET_ELEMENTS_ERROR, e);
        } catch (JSONException e) {
            throw USMJSONAPIException.createJSONError(ConnectorBundleErrorCodes.READ_DATA_JSON_ERROR, e);
        }
    }

	protected JSONObject storeRequestedFieldsInJSONObject(DataObject o, List<String> fields) throws JSONException {
		JSONObject result = new JSONObject();
		for (String field : fields) {
			Object fieldContent = o.getFieldContent(field);
			DataType<?> fieldType = o.getContentType().getFields()[o.getFieldIndex(field)].getFieldType();
			fieldType.storeInJSONObject(o.getSession(), result, field, fieldContent);
		}
		result.put(UUID, o.getUUID().toString());
		return result;
	}

	@Override
	protected String[] getOptionalParameters() {
		return OPTIONAL_PARAMETERS;
	}

	@Override
	protected String[] getRequiredParameters() {
		return REQUIRED_PARAMETERS;
	}
}
