/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.connector.commands;

import static com.openexchange.usm.connector.commands.CommandConstants.AVAILABLE_CONTENT_TYPES;
import static com.openexchange.usm.connector.commands.CommandConstants.CLIENT_FIELD_PREFIX;
import static com.openexchange.usm.connector.commands.CommandConstants.CONFLICT_RESOLUTION;
import static com.openexchange.usm.connector.commands.CommandConstants.CONTEXT_ID;
import static com.openexchange.usm.connector.commands.CommandConstants.CONTEXT_UUID;
import static com.openexchange.usm.connector.commands.CommandConstants.CUSTOM_PROPERTIES;
import static com.openexchange.usm.connector.commands.CommandConstants.DEVICE;
import static com.openexchange.usm.connector.commands.CommandConstants.END_DATE;
import static com.openexchange.usm.connector.commands.CommandConstants.FOLDER_TREE;
import static com.openexchange.usm.connector.commands.CommandConstants.START_DATE;
import static com.openexchange.usm.connector.commands.CommandConstants.SYNCID;
import static com.openexchange.usm.connector.commands.CommandConstants.SYNC_CONTENT_TYPES;
import static com.openexchange.usm.connector.commands.CommandConstants.TIME_ZONE;
import static com.openexchange.usm.connector.commands.CommandConstants.USER;
import static com.openexchange.usm.connector.commands.CommandConstants.USERID;
import static com.openexchange.usm.connector.commands.CommandConstants.UUID_KEY;
import static com.openexchange.usm.connector.commands.CommandConstants.START_DATE_CALENDAR;
import static com.openexchange.usm.connector.commands.CommandConstants.END_DATE_CALENDAR;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.util.BitSet;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.annotation.NonNull;
import com.openexchange.usm.api.contenttypes.common.ContentType;
import com.openexchange.usm.api.contenttypes.common.ContentTypeField;
import com.openexchange.usm.api.contenttypes.common.ContentTypeManager;
import com.openexchange.usm.api.contenttypes.common.DefaultContentTypes;
import com.openexchange.usm.api.contenttypes.folder.FolderContentType;
import com.openexchange.usm.api.exceptions.InvalidUUIDException;
import com.openexchange.usm.api.exceptions.USMAccessDeniedException;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.ox.json.OXJSONPropertyNames;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.Folder;
import com.openexchange.usm.api.session.Session;
import com.openexchange.usm.api.session.assets.ChangeState;
import com.openexchange.usm.api.session.assets.ConflictResolution;
import com.openexchange.usm.connector.exceptions.DataObjectNotFoundException;
import com.openexchange.usm.connector.exceptions.MultipleOperationsOnDataObjectException;
import com.openexchange.usm.json.ConnectorBundleErrorCodes;
import com.openexchange.usm.json.JSONSessionInitializer;
import com.openexchange.usm.json.ServletConstants;
import com.openexchange.usm.json.USMJSONAPIException;
import com.openexchange.usm.json.USMJSONAccessDeniedException;
import com.openexchange.usm.json.USMJSONServlet;
import com.openexchange.usm.json.USMSessionCredentials;
import com.openexchange.usm.json.response.ResponseObject;
import com.openexchange.usm.json.response.ResponseStatusCode;
import com.openexchange.usm.mimemail.ExternalMailContentTypeFields;
import com.openexchange.usm.session.dataobject.DataObjectSet;
import com.openexchange.usm.util.BitSetEncoder;
import com.openexchange.usm.util.JSONToolkit;
import com.openexchange.usm.util.TempFileStorage;
import com.openexchange.usm.util.Toolkit;
import com.openexchange.usm.util.UUIDToolkit;

/**
 * General Handler for all USM-JSON-Commands. All command handlers for specific commands should extend this handler.
 * 
 * @author ldo
 */
public abstract class BaseCommandHandler implements CommandHandler {

    private static final String HOST = "host";

    private static final String CUSTOM_HOST_HEADER = "X-Host";

    private static final String DEFAULT_ADDRESS = "defaultAddress";

    private static final String MAX_INLINE_ATTACHMENT_SIZE = "max_inline_attachment_size";

    private static final String TEMP_FILES_TIMEOUT = "temp_files_timeout";

    private static JSONObject extractJSONObjectFromRequest(HttpServletRequest request) throws USMJSONAPIException {
        // read the JSON object from the request
        BufferedReader reader = null;
        try {
            reader = new BufferedReader(new InputStreamReader(request.getInputStream(), ServletConstants.UTF_ENCODING_8));
            char[] buffer = new char[4096];
            StringBuilder sb = new StringBuilder(4096);
            for (int count = reader.read(buffer); count > 0; count = reader.read(buffer)) {
                sb.append(buffer, 0, count);
            }
            JSONObject requestObject = new JSONObject(sb.toString());
            return requestObject;
        } catch (UnsupportedEncodingException e) {
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.UTF_8_NOT_SUPPORTED,
                ResponseStatusCode.INTERNAL_ERROR,
                "UTF-8 Encoding is not supported");
        } catch (IOException e) {
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.ERROR_ON_READ_REQUEST,
                ResponseStatusCode.BAD_REQUEST,
                "Error while reading request",
                e);
        } catch (JSONException e) {
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.BAD_JSONOBJECT_IN_REQUEST,
                ResponseStatusCode.BAD_REQUEST,
                "Invalid JSONObject as parameter",
                e);
        } finally {
            Toolkit.close(reader);
        }
    }

    public static Session retrieveUSMSession(USMJSONServlet servlet, HttpServletRequest request, USMSessionCredentials creds) throws USMJSONAPIException {
        return retrieveUSMSession(servlet, request, creds.getUser(), creds.getPassword(), creds.getDevice());
    }

    public static Session retrieveUSMSession(USMJSONServlet servlet, HttpServletRequest request, String user, String password, String device) throws USMJSONAPIException {
        try {
            return servlet.getSession(user, password, device, request.getRemoteAddr(), getXHeadersFromRequest(request));
        } catch (USMAccessDeniedException e) {
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.COMMAND_ACCESS_DENIED,
                ResponseStatusCode.ACCESS_DENIED,
                e.getErrorMessage());
        } catch (USMException e) {
            throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.COMMAND_INTERNAL_ERROR, e);
        }
    }

    public static Map<String, String> getXHeadersFromRequest(HttpServletRequest request) {
        Map<String, String> headers = new HashMap<String, String>();
        Enumeration<?> headerNames = request.getHeaderNames();
        while (headerNames.hasMoreElements()) {
            String headerName = (String) headerNames.nextElement();
            if (headerName.toLowerCase().startsWith("x-"))
                headers.put(headerName, request.getHeader(headerName));
            else if (HOST.equalsIgnoreCase(headerName))
                headers.put(CUSTOM_HOST_HEADER, request.getHeader(headerName));
        }
        return headers;
    }

    public static void initSessionRequestData(USMJSONServlet servlet, Session session) throws USMJSONAccessDeniedException {
        if (servlet.getAccessLimiter().isUSMAccessDenied(session))
            throw new USMJSONAccessDeniedException(
                ConnectorBundleErrorCodes.COMMAND_REJECTED_TOO_MANY_SYNCHRONIZATIONS,
                "Too many unnecessary synchronizations");
        session.setCustomProperty(OXJSONPropertyNames.NUMBER_OF_CALLS, null);
        session.setCustomProperty(OXJSONPropertyNames.ACCUMULATED_TIME, null);
    }

    protected final USMJSONServlet _servlet;

    protected final JSONObject _parameters;

    protected final HttpServletRequest _currentHttpRequest;

    private final Map<ContentType, BitSet> _extraFieldsMap = new HashMap<ContentType, BitSet>();

    protected Session _session;

    protected BaseCommandHandler(USMJSONServlet servlet, HttpServletRequest request) throws USMJSONAPIException {
        _servlet = servlet;
        _parameters = extractJSONObjectFromRequest(request);
        _currentHttpRequest = request;
        int requiredCount = 0;
        for (String key : JSONToolkit.keys(_parameters)) {
            if (checkParameter(key))
                requiredCount++;
        }
        if (requiredCount != getRequiredParameters().length) {
            for (String key : getRequiredParameters()) {
                if (!_parameters.has(key))
                    throw new USMJSONAPIException(
                        ConnectorBundleErrorCodes.COMMAND_MISSING_REQUIRED_PARAMETER,
                        ResponseStatusCode.WRONG_MISSING_PARAMETERS,
                        "Missing required parameter " + key);
            }
        }
    }

    private final boolean checkParameter(String key) throws USMJSONAPIException {
        for (String requiredKey : getRequiredParameters()) {
            if (key.equals(requiredKey))
                return true;
        }
        for (String requiredKey : getOptionalParameters()) {
            if (key.equals(requiredKey))
                return false;
        }
        throw new USMJSONAPIException(
            ConnectorBundleErrorCodes.COMMAND_UNKNOWN_PARAMETER,
            ResponseStatusCode.WRONG_MISSING_PARAMETERS,
            "Unknown parameter " + key);
    }

    /*
     * (non-Javadoc)
     * @see com.openexchange.usm.connector.commands.CommandHandler#handleRequest(org.json.JSONObject,
     * com.openexchange.usm.json.ConnectorServlet, javax.servlet.http.HttpSession)
     */
    @Override
    public ResponseObject handleRequest() throws USMJSONAPIException {
        throw new USMJSONAPIException(
            ConnectorBundleErrorCodes.COMMAND_NOT_IMPLEMENTED,
            ResponseStatusCode.INTERNAL_ERROR,
            "No handler for request");
    }

    protected boolean getBooleanParameter(String name) throws USMJSONAPIException {
        return getBooleanParameter(name, false);
    }

    protected boolean getBooleanParameter(String name, boolean defaultValue) throws USMJSONAPIException {
        return _parameters.has(name) ? getBoolean(_parameters, name) : defaultValue;
    }

    protected String getStringParameter(String name) throws USMJSONAPIException {
        return getStringParameter(name, null);
    }

    protected String getStringParameter(String name, String defaultValue) throws USMJSONAPIException {
        return _parameters.has(name) ? getString(_parameters, name) : defaultValue;
    }

    protected long getLongParameter(String name, long defaultValue) throws USMJSONAPIException {
        return _parameters.has(name) ? getLong(_parameters, name) : defaultValue;
    }

    protected boolean hasSyncID() {
        return _parameters.has(SYNCID);
    }

    protected long getSyncID() throws USMJSONAPIException {
        return getLong(_parameters, SYNCID);
    }

    protected ConflictResolution getOptionalConflictResolution() throws USMJSONAPIException {
        try {
            if (!_parameters.has(CONFLICT_RESOLUTION))
                return null;
            return ConflictResolution.valueOf(_parameters.getString(CONFLICT_RESOLUTION));
        } catch (IllegalArgumentException ignored) {
            // fall through
        } catch (JSONException ignored) {
            // fall through
        }
        throw new USMJSONAPIException(
            ConnectorBundleErrorCodes.COMMAND_BAD_CONFLICT_RESOLUTION_PARAMETER,
            ResponseStatusCode.WRONG_MISSING_PARAMETERS,
            "Bad value for parameter " + CONFLICT_RESOLUTION);
    }

    protected String getUUIDString(JSONObject object) {
        return object.optString(UUID_KEY, "<not set>");
    }

    protected UUID extractUUIDFromJSONObject(JSONObject object) throws InvalidUUIDException {
        return UUIDToolkit.extractUUIDFromString(getUUIDString(object));
    }

    
    protected JSONArray getJSONArray(JSONObject o, String key) throws USMJSONAPIException {
        try {
            return o.getJSONArray(key);
        } catch (JSONException e) {
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.COMMAND_NO_JSON_ARRAY_IN_JSON_OBJECT,
                ResponseStatusCode.WRONG_MISSING_PARAMETERS,
                "No JSONArray for key " + key);
        }
    }

    protected JSONObject getJSONObject(JSONObject o, String key) throws USMJSONAPIException {
        try {
            return o.getJSONObject(key);
        } catch (JSONException e) {
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.COMMAND_NO_JSON_OBJECT_IN_JSON_OBJECT,
                ResponseStatusCode.WRONG_MISSING_PARAMETERS,
                "No JSONObject for key " + key);
        }
    }

    protected boolean getBoolean(JSONObject o, String key) throws USMJSONAPIException {
        try {
            return o.getBoolean(key);
        } catch (JSONException e) {
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.COMMAND_NO_BOOLEAN_IN_JSON_OBJECT,
                ResponseStatusCode.WRONG_MISSING_PARAMETERS,
                "No boolean for key " + key);
        }
    }

    protected String getString(JSONObject o, String key) throws USMJSONAPIException {
        try {
            return o.getString(key);
        } catch (JSONException e) {
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.COMMAND_NO_STRING_IN_JSON_OBJECT,
                ResponseStatusCode.WRONG_MISSING_PARAMETERS,
                "No String for key " + key);
        }
    }

    protected long getLong(JSONObject o, String key) throws USMJSONAPIException {
        try {
            return o.getLong(key);
        } catch (JSONException e) {
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.COMMAND_NO_LONG_IN_JSON_OBJECT,
                ResponseStatusCode.WRONG_MISSING_PARAMETERS,
                "No long for key " + key);
        }
    }

    protected long getLong(JSONArray o, int index) throws USMJSONAPIException {
        try {
            return o.getLong(index);
        } catch (JSONException e) {
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.COMMAND_NO_LONG_IN_JSON_ARRAY,
                ResponseStatusCode.WRONG_MISSING_PARAMETERS,
                "No long for index " + index);
        }
    }

    protected int getInt(JSONObject o, String key) throws USMJSONAPIException {
        try {
            return o.getInt(key);
        } catch (JSONException e) {
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.COMMAND_NO_INT_IN_JSON_OBJECT,
                ResponseStatusCode.WRONG_MISSING_PARAMETERS,
                "No int for key " + key);
        }
    }

    protected JSONObject getJSONObject(JSONArray a, int index) throws USMJSONAPIException {
        try {
            return a.getJSONObject(index);
        } catch (JSONException e) {
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.COMMAND_NO_JSON_OBJECT_IN_JSON_ARRAY,
                ResponseStatusCode.WRONG_MISSING_PARAMETERS,
                "No JSONObject in JSONArray at index " + index);
        }
    }

    protected String getString(JSONArray a, int index) throws USMJSONAPIException {
        try {
            return a.getString(index);
        } catch (JSONException e) {
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.COMMAND_NO_STRING_IN_JSON_ARRAY,
                ResponseStatusCode.WRONG_MISSING_PARAMETERS,
                "No String in JSONArray for index " + index);
        }
    }

    protected BitSet getExtraFields(ContentType type) {
        BitSet result = _extraFieldsMap.get(type);
        if (result == null) {
            String extraFieldStorage = _session.getPersistentField(JSONSessionInitializer.getTypePropertyKey(type));
            result = (extraFieldStorage == null || extraFieldStorage.length() == 0) ? new BitSet() : BitSetEncoder.fromBase64(extraFieldStorage);
            _extraFieldsMap.put(type, result);
        }
        return result;
    }

    protected JSONObject getConfiguration() throws USMJSONAPIException {
        boolean availableContentTypes = getBooleanParameter(AVAILABLE_CONTENT_TYPES);
        boolean syncContentTypes = getBooleanParameter(SYNC_CONTENT_TYPES);
        ContentTypeManager contentTypeManager = _servlet.getContentTypeManager();
        try {
            JSONObject config = new JSONObject();
            // mandatory
            config.put(USER, _session.getUser());
            config.put(DEVICE, _session.getDevice());
            config.put(TIME_ZONE, _session.getUserTimeZone().getID());
            config.put(START_DATE, String.valueOf(_session.getStartDate()));
            config.put(END_DATE, String.valueOf(_session.getEndDate()));
            config.put(START_DATE_CALENDAR, String.valueOf(_session.getCalendarStartDate()));
            config.put(END_DATE_CALENDAR, String.valueOf(_session.getCalendarEndDate()));
            config.put(FOLDER_TREE, _session.getPersistentField(FolderContentType.FOLDER_TREE));
            config.put(CONFLICT_RESOLUTION, _session.getConflictResolution());
            config.put(CONTEXT_UUID, String.valueOf(_session.getContextUUID()));
            config.put(CONTEXT_ID, _session.getContextId());
            config.put(USERID, _session.getUserIdentifier());
            config.put(DEFAULT_ADDRESS, _session.getDefaultEmailAddress());
            config.put(MAX_INLINE_ATTACHMENT_SIZE, _servlet.getMaxInlineAttachmentSize());
            config.put(TEMP_FILES_TIMEOUT, TempFileStorage.getTempFileTimeout());

            config.put(CUSTOM_PROPERTIES, getCustomProperties());
            // optional
            if (syncContentTypes) {
                config.put(SYNC_CONTENT_TYPES, getSyncContentTypes(contentTypeManager));
            }
            if (availableContentTypes) {
                config.put(AVAILABLE_CONTENT_TYPES, getAvailableContentTypes(contentTypeManager));
            }
            return config;
        } catch (JSONException e) {
            throw USMJSONAPIException.createJSONError(ConnectorBundleErrorCodes.GET_CONFIG_JSON_ERROR, e);
        } catch (USMException e) {
            throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.GET_CONFIG_INTERNAL_ERROR, e);
        }
    }

    private JSONObject getCustomProperties() throws JSONException {
        JSONObject result = new JSONObject();
        for (Map.Entry<String, String> entry : _session.getPersistentFields().entrySet()) {
            String key = entry.getKey();
            if (key.startsWith(CLIENT_FIELD_PREFIX))
                result.put(key.substring(CLIENT_FIELD_PREFIX.length()), entry.getValue());
        }
        return result;
    }

    protected JSONObject getAvailableContentTypes(ContentTypeManager contentTypeManager) throws JSONException, USMException {
        JSONObject contentTypesObject = new JSONObject();
        ContentType[] availableContentTypes = contentTypeManager.getRegisteredContentTypes();

        for (ContentType type : availableContentTypes) {
            ContentTypeField[] fields = type.getFields();
            BitSet availableFields = new BitSet();
            availableFields.set(0, fields.length, true);
            JSONSessionInitializer.excludeInvalidFieldsFromFilter(type, availableFields);
            JSONArray fieldsArray = new JSONArray();
            for (int i = 0; i < fields.length; i++) {
                if (availableFields.get(i)) {
                    String fieldName = fields[i].getFieldName();
                    if (CommandConstants.ATTACHMENTS_LAST_MODIFIED.equals(fieldName))
                        fieldsArray.put(CommandConstants.ATTACHMENTS);
                    else
                        fieldsArray.put(fieldName);
                }
            }
            if (DefaultContentTypes.MAIL_ID.equals(type.getID())) {
                String[] externalFields = ExternalMailContentTypeFields.getFields();
                for (int i = 0; i < externalFields.length; i++) {
                    fieldsArray.put(externalFields[i]);
                }
            }

            contentTypesObject.put(type.getID(), fieldsArray);
        }
        return contentTypesObject;
    }

    private JSONObject getSyncContentTypes(ContentTypeManager contentTypeManager) throws JSONException, USMException {
        JSONObject contentTypesObject = new JSONObject();
        ContentType[] availableContentTypes = contentTypeManager.getRegisteredContentTypes();

        for (ContentType type : availableContentTypes) {
            BitSet bitSet = _session.getFieldFilter(type);
            if (bitSet != null) {
                if (!DefaultContentTypes.MAIL_ID.equals(type.getID()))
                    bitSet.or(getExtraFields(type));
                ContentTypeField[] fields = type.getFields();
                JSONArray fieldsArray = new JSONArray();
                for (int i = 0; i < fields.length; i++) {
                    if (bitSet.get(i)) {
                        String fieldName = fields[i].getFieldName();
                        if (CommandConstants.ATTACHMENTS_LAST_MODIFIED.equals(fieldName))
                            fieldsArray.put(CommandConstants.ATTACHMENTS);
                        else
                            fieldsArray.put(fieldName);

                    }
                }

                if (DefaultContentTypes.MAIL_ID.equals(type.getID())) {
                    BitSet extraFieldsBitSet = getExtraFields(type);
                    String[] externalFields = ExternalMailContentTypeFields.getFields();
                    for (int i = 0; i < externalFields.length; i++) {
                        if (extraFieldsBitSet.get(i))
                            fieldsArray.put(externalFields[i]);
                    }
                }
                contentTypesObject.put(type.getID(), fieldsArray);
            }
        }
        return contentTypesObject;
    }

    @Override
    public Session getSession() {
        return _session;
    }

    protected void setSession(Session session) throws USMJSONAPIException {
        _session = session;
        initSessionRequestData(_servlet, session);
    }

    protected abstract String[] getRequiredParameters();

    protected abstract String[] getOptionalParameters();

    protected @NonNull DataObject getDataObjectByUUID(DataObjectSet objects, UUID uuid) throws DataObjectNotFoundException, MultipleOperationsOnDataObjectException {
        if (objects == null || !objects.contains(uuid))
            throw new DataObjectNotFoundException(uuid);
        DataObject o = objects.get(uuid);
        if (o.getChangeState() != ChangeState.UNMODIFIED)
            throw new MultipleOperationsOnDataObjectException(o);
        return o;
    }

    // TODO Optimize this
    protected Folder getFolderByUUID(String folderUUID) throws USMJSONAPIException {
        try {
            if (DefaultContentTypes.GROUPS_ID.equals(folderUUID) || DefaultContentTypes.RESOURCES_ID.equals(folderUUID))
                return _session.getCachedFolder(folderUUID);
            UUID uuid = UUIDToolkit.extractUUIDFromString(folderUUID);
            Folder[] folders = _session.getCachedFolders();
            DataObject folder = new DataObjectSet(folders).get(uuid);
            if (folder != null)
                return (Folder) folder;
        } catch (USMException e) {
            throw USMJSONAPIException.createInternalError(ConnectorBundleErrorCodes.COMMAND_CANNOT_FIND_FOLDER_BY_UUID, e);
        }
        throw new USMJSONAPIException(
            ConnectorBundleErrorCodes.COMMAND_UNKNOWN_FOLDER_ID,
            ResponseStatusCode.UNKNOWN_UUID,
            "Unknown folderid " + folderUUID);
    }

    protected Map<String, String> getXHeadersFromCurrentRequest() {
        return getXHeadersFromRequest(_currentHttpRequest);
    }

    public HttpSession getHttpSession() {
        return _currentHttpRequest.getSession();
    }

    @Override
    public void disposeResources() {
        // default: do nothing
    }
}
