/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.database.ox;

import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Statement;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import com.openexchange.usm.api.database.StorageAccessException;
import com.openexchange.usm.api.exceptions.USMStorageException;
import com.openexchange.usm.util.SQLToolkit;
import com.openexchange.usm.util.Toolkit;

public class TableDefinition {

    final static String SESSION_TABLE_NAME = "USMSession";

    private static final Log LOG = LogFactory.getLog(TableDefinition.class);

    // Names of added tables
    public final static String[] TABLE_NAMES = { "USMSession", "usmIdMapping", "easUIDMapping", "USMDataStorage", "USMSessionFields" };

    public final static String[] SEQUENCE_TABLES_NAMES = { "sequence_usmSessionId", "sequence_usmIdMapping" };

    // SQL statements to create tables
    public final static String[] SQL_STATEMENTS = {
        "CREATE TABLE USMSession (cid INT4 UNSIGNED NOT NULL, id INT4 UNSIGNED NOT NULL, device VARCHAR(64) NOT NULL, protocol VARCHAR(32) NOT NULL, usmSessionId INT4 UNSIGNED NOT NULL, PRIMARY KEY (cid, usmSessionId), UNIQUE (cid, id, device, protocol)) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_bin",
        "CREATE TABLE usmIdMapping(cid INT4 UNSIGNED NOT NULL, usmSessionId INT4 UNSIGNED NOT NULL, longId VARCHAR(255) NOT NULL, shortId INT4 UNSIGNED NOT NULL, PRIMARY KEY (cid, usmSessionId, shortId), FOREIGN KEY(cid, usmSessionId) REFERENCES USMSession(cid, usmSessionId)) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_bin",
        "CREATE TABLE easUIDMapping(cid INT4 UNSIGNED NOT NULL, usmSessionId INT4 UNSIGNED NOT NULL, folderId VARCHAR(64) NOT NULL, longId VARCHAR(255) NOT NULL, shortId INT4 UNSIGNED NOT NULL, PRIMARY KEY (cid, usmSessionId, folderId, shortId), FOREIGN KEY(cid, usmSessionId) REFERENCES USMSession(cid, usmSessionId)) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_bin",
        "CREATE TABLE USMDataStorage (cid INT4 UNSIGNED NOT NULL, usmSessionId INT4 UNSIGNED NOT NULL, ObjectID VARCHAR(64) NOT NULL, SyncKey INT8 NOT NULL, Element LONGTEXT NOT NULL, PRIMARY KEY (cid, usmSessionId, objectId, syncKey), FOREIGN KEY(cid, usmSessionId) REFERENCES USMSession(cid, usmSessionId)) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_bin",
        "CREATE TABLE USMSessionFields (cid INT4 UNSIGNED NOT NULL, usmSessionId INT4 UNSIGNED NOT NULL, field VARCHAR(20) NOT NULL, value VARCHAR(32) NOT NULL, PRIMARY KEY (cid, usmSessionId, field), FOREIGN KEY(cid, usmSessionId) REFERENCES USMSession(cid, usmSessionId)) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_bin" };

    public final static String[] UUID_SQL_STATEMENTS = {
        "CREATE TABLE UUIDHeader(cid INT4 UNSIGNED NOT NULL, header VARCHAR(36) NOT NULL, PRIMARY KEY (header)) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_bin",
        "CREATE TABLE UUIDMapping (cid INT4 UNSIGNED NOT NULL, contentType INT4 UNSIGNED NOT NULL, objectId INT4 UNSIGNED NOT NULL, uuid BINARY(16) NOT NULL, PRIMARY KEY (cid, uuid), UNIQUE (cid, contentType, objectId)) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_bin" };

    public static void generateSequence(String sequenceName, Connection con, boolean isAutocommit, boolean createTable) throws USMStorageException, StorageAccessException {
        Statement stmt = null;
        try {
            try {
                stmt = con.createStatement();
                if (isAutocommit)
                    con.setAutoCommit(false);
                if (createTable)
                    stmt.execute(SQLToolkit.createSequenceStatement(sequenceName));
                // do not create stored procedures anymore, they will be deleted by USMDeleteStoredProceduresUpdateTaskV2 anyway
                //stmt.execute(SQLToolkit.dropSequenceProcedure(sequenceName));
                //stmt.execute(SQLToolkit.createSequenceProcedure(sequenceName));
                con.commit();
                LOG.info("Creation of sequence " + sequenceName + " successfully performed");
            } catch (SQLException e) {
                if (isAutocommit)
                    con.rollback();
                throw new USMStorageException(USMDatabaseOXErrorCodes.CREATE_SEQUENCE_ERROR, e.getMessage(), e);
            } finally {
                if (isAutocommit)
                    con.setAutoCommit(true);
                Toolkit.close(stmt);
            }
        } catch (SQLException e) {
            throw new USMStorageException(USMDatabaseOXErrorCodes.CREATE_SEQUENCE_ERROR_AUTOCOMMIT_SETTINGS, e.getMessage(), e);
        }
    }

}
