/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.contenttypes.util;

import java.util.Arrays;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.usm.api.contenttypes.common.CommonConstants;

/**
 * {@link OXObjectIDList} List of OX Object IDs and their timestamps as parsed from a JSONArray result, sorted by OX object ID (numerically or lexically)
 * 
 * @author <a href="mailto:afe@microdoc.de">Alexander Feess</a>
 */
public class OXObjectIDList {

    private static class OXTimestamp implements Comparable<OXTimestamp> {

        private final static int parseIdNumber(String id) {
            try {
                return Integer.parseInt(id);
            } catch (NumberFormatException e) {
                return 0;
            }
        }

        public final String _id;

        public final int _idNumber;

        public final long _lastModified;

        public OXTimestamp(String id, long lastModified) {
            _id = id;
            _lastModified = lastModified;
            _idNumber = parseIdNumber(_id);
        }

        @Override
        public int compareTo(OXTimestamp o) {
            if (_idNumber == 0 || o._idNumber == 0)
                return _id.compareTo(o._id);
            return _idNumber - o._idNumber;
        }
    }

    private final static OXTimestamp[] _EMPTY_ARRAY = {};

    private final long _timestamp;

    private final OXTimestamp[] _data;

    public OXObjectIDList(long defaultTimestamp, JSONObject oxObjectIDdata) throws JSONException {
        _timestamp = oxObjectIDdata.optLong(CommonConstants.RESULT_TIMESTAMP, defaultTimestamp);
        JSONArray jsonar = oxObjectIDdata.getJSONArray(CommonConstants.RESULT_DATA);
        int length = jsonar.length();
        if (length == 0) {
            _data = _EMPTY_ARRAY;
        } else {
            _data = new OXTimestamp[length];
            for (int i = 0; i < length; i++) {
                JSONArray columnsArray = jsonar.getJSONArray(i);
                _data[i] = new OXTimestamp(columnsArray.getString(0), columnsArray.getLong(1));
            }
        }
        Arrays.sort(_data); // Make sure that data is sorted by id so that series will be before any exceptions of the series
    }

    public long getTimestamp() {
        return _timestamp;
    }

    public int size() {
        return _data.length;
    }

    public String getID(int i) {
        return _data[i]._id;
    }

    public long getLastModified(int i) {
        return _data[i]._lastModified;
    }
}
