/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */
package com.openexchange.usm.api.ox.json;

import java.util.Map;
import org.apache.commons.httpclient.MultiThreadedHttpConnectionManager;
import org.json.JSONArray;
import org.json.JSONObject;
import com.openexchange.usm.api.contenttypes.resource.ResourceInputStream;
import com.openexchange.usm.api.exceptions.AuthenticationFailedException;
import com.openexchange.usm.api.exceptions.OXCommunicationException;
import com.openexchange.usm.api.io.InputStreamProvider;
import com.openexchange.usm.api.session.OXConnectionInformation;
import com.openexchange.usm.api.session.Session;

/**
 * The OXAjaxAccess service wraps the http communication with the OX Ajax-Server interface
 * for easy access from other bundles, especially ContentTypes and Authentication.
 * 
 * <br>TODO Check if POST operation is also required
 * 
 * @author afe
 *
 */
public interface OXJSONAccess {
	/**
	 * Login user into OX server
	 * 
	 * @param session
	 * @throws AuthenticationFailedException if login failed, details are stored in the Exception
	 * @throws OXCommunicationException if some error occurred during the communication with the OX server
	 */
	void login(Session session) throws AuthenticationFailedException, OXCommunicationException;

	/**
	 * End session that was previously started using login()
	 * 
	 * @param session id of session to end
	 */
	void logout(Session session);

	/**
	 * Can be called after a session has been removed to cleanup any OX resources that might still be allocated,
	 * e.g. closing the OX session for the USM session.
	 * 
	 * @param data
	 */
	void logout(OXConnectionInformation data);

	/**
	 * Retrieve the OX Ajax configuration for the user of the Session
	 * 
	 * @param session
	 * @param path optional sub-path of configuration to retrieve
	 * @return
	 * @throws AuthenticationFailedException if the authentication failed on the OX server
	 * @throws OXCommunicationException if some error occurred during the communication with the OX server
	 */
	JSONObject getConfiguration(Session session, String... path) throws AuthenticationFailedException,
			OXCommunicationException;

	/**
	 * Perform a GET operation on the Ajax-Interface using the specified sub-path
	 * and request parameters.
	 * <br>TODO Check if using a Map is the best method for providing request parameters
	 * 	(impossible to specify parameters with multiple values, no types for parameters)
	 * 
	 * @param path sub-path to use for request url, e.g. "login", "folders", ...
	 * @param action value for always required parameter action (if set to null, it will be omitted in the request)
	 * @param session value for always required parameter session (if set to null, it will be omitted in the request, but no cookie resolving for sessions can be performed)
	 * @param parameters Map containing all request parameters to use
	 * @return JSONData containing the parsed response
	 * @throws AuthenticationFailedException if the authentication failed on the OX server
	 * @throws OXCommunicationException if some error occurred during the communication with the OX server
	 */
	JSONResult doGet(String path, String action, Session session, Map<String, String> parameters)
			throws AuthenticationFailedException, OXCommunicationException;

	/**
	 * Perform a PUT operation on the Ajax-Interface using the specified sub-path
	 * and request parameters. The provided JSONArray is sent in the request body
	 * <br>TODO Check if using a Map is the best method for providing request parameters
	 * 	(impossible to specify parameters with multiple values, no types for parameters)
	 * 
	 * @param path sub-path to use for request url, e.g. "login", "folders", ...
	 * @param action value for always required parameter action (if set to null, it will be omitted in the request)
	 * @param session value for always required parameter session (if set to null, it will be omitted in the request, but no cookie resolving for sessions can be performed)
	 * @param parameters Map containing all request parameters to use
	 * @param array JSONArray to send in request body
	 * @return JSONData containing the parsed response
	 * @throws AuthenticationFailedException if the authentication failed on the OX server (e.g. password was changed, user was deleted)
	 * @throws OXCommunicationException if some error occurred during the communication with the OX server
	 */
	JSONResult doPut(String path, String action, Session session, Map<String, String> parameters, JSONArray array)
			throws AuthenticationFailedException, OXCommunicationException;

	/**
	 * Perform a PUT operation on the Ajax-Interface using the specified sub-path
	 * and request parameters. The provided JSONObject is sent in the request body
	 * <br>TODO Check if using a Map is the best method for providing request parameters
	 * 	(impossible to specify parameters with multiple values, no types for parameters)
	 * 
	 * @param path sub-path to use for request url, e.g. "login", "folders", ...
	 * @param action value for always required parameter action (if set to null, it will be omitted in the request)
	 * @param session value for always required parameter session (if set to null, it will be omitted in the request, but no cookie resolving for sessions can be performed)
	 * @param parameters Map containing all request parameters to use
	 * @param object JSONObject to send in request body
	 * @return JSONData containing the parsed response
	 * @throws AuthenticationFailedException if the authentication failed on the OX server (e.g. password was changed, user was deleted)
	 * @throws OXCommunicationException if some error occurred during the communication with the OX server
	 */
	JSONResult doPut(String path, String action, Session session, Map<String, String> parameters, JSONObject object)
			throws AuthenticationFailedException, OXCommunicationException;

	/**
	 * Perform a PUT operation on the Ajax-Interface using the specified sub-path
	 * and request parameters. The provided String is sent in the request body (encoded using UTF-8)
	 * <br>TODO Check if using a Map is the best method for providing request parameters
	 * 	(impossible to specify parameters with multiple values, no types for parameters)
	 * 
	 * @param path sub-path to use for request url, e.g. "login", "folders", ...
	 * @param action value for always required parameter action (if set to null, it will be omitted in the request)
	 * @param session value for always required parameter session (if set to null, it will be omitted in the request, but no cookie resolving for sessions can be performed)
	 * @param parameters Map containing all request parameters to use
	 * @param data String to send in request body
	 * @return JSONData containing the parsed response
	 * @throws AuthenticationFailedException if the authentication failed on the OX server (e.g. password was changed, user was deleted)
	 * @throws OXCommunicationException if some error occurred during the communication with the OX server
	 */
	JSONResult doPut(String path, String action, Session session, Map<String, String> parameters, String data)
			throws AuthenticationFailedException, OXCommunicationException;

	/**
	 * Perform a PUT operation on the Ajax-Interface using the specified sub-path
	 * and request parameters. The provided byte array is sent in the request body
	 * 
	 * @param path sub-path to use for request url, e.g. "login", "folders", ...
	 * @param action value for always required parameter action (if set to null, it will be omitted in the request)
	 * @param session value for always required parameter session (if set to null, it will be omitted in the request, but no cookie resolving for sessions can be performed)
	 * @param parameters Map containing all request parameters to use
	 * @param data data to send in request body
	 * @return JSONData containing the parsed response
	 * @throws AuthenticationFailedException if the authentication failed on the OX server (e.g. password was changed, user was deleted)
	 * @throws OXCommunicationException if some error occurred during the communication with the OX server
	 */
	JSONResult doPut(String path, String action, Session session, Map<String, String> parameters, byte[] data)
			throws AuthenticationFailedException, OXCommunicationException;

    /**
     * Perform a PUT operation on the Ajax-Interface using the specified sub-path
     * and request parameters. The data provided by the InputStream is sent in the request body
     * 
     * @param path sub-path to use for request url, e.g. "login", "folders", ...
     * @param action value for always required parameter action (if set to null, it will be omitted in the request)
     * @param session value for always required parameter session (if set to null, it will be omitted in the request, but no cookie resolving for sessions can be performed)
     * @param parameters Map containing all request parameters to use
     * @param streamProvider InputStreamProvider for the data to send in request body
     * @return JSONData containing the parsed response
     * @throws AuthenticationFailedException if the authentication failed on the OX server (e.g. password was changed, user was deleted)
     * @throws OXCommunicationException if some error occurred during the communication with the OX server
     */
    JSONResult doPut(String path, String action, Session session, Map<String, String> parameters, InputStreamProvider streamProvider) throws AuthenticationFailedException, OXCommunicationException;

	OXResource getResource(Session session, String path, Map<String, String> parameters)
			throws AuthenticationFailedException, OXCommunicationException;

	/**
	 * Get a resource as InputStream. The returned InputStream must be closed before any other accesses to OX can be performed.
	 * 
	 * @param session
	 * @param oxAjaxPath
	 * @param parameters
	 * @return
	 * @throws AuthenticationFailedException
	 * @throws OXCommunicationException
	 */
    ResourceInputStream getResourceAsStream(Session session, String oxAjaxPath, Map<String, String> parameters)
        throws AuthenticationFailedException, OXCommunicationException;
	
	JSONResult storeResource(String path, String action, Session session, Map<String, String> parameters,
			JSONObject jsonData, byte[] data, String fileContentType) throws AuthenticationFailedException,
			OXCommunicationException;

	JSONResult storeResource(String path, String action, Session session, Map<String, String> parameters,
			JSONObject jsonData, byte[] data, String fileContentType, String jsonPartName, String filePartName)
			throws AuthenticationFailedException, OXCommunicationException;

    JSONResult storeResourceFromStream(String path, String action, Session session, Map<String, String> parameters,
            JSONObject jsonData, String fileContentType, long size, InputStreamProvider streamProvider)
            throws AuthenticationFailedException, OXCommunicationException;

    JSONResult storeResourceFromStream(String path, String action, Session session, Map<String, String> parameters,
            JSONObject jsonData, String fileContentType, String jsonPartName, String filePartName, long size, InputStreamProvider streamProvider)
            throws AuthenticationFailedException, OXCommunicationException;
	
	JSONResult doPost(String path, String action, Session session, Map<String, String> parameters)
			throws AuthenticationFailedException, OXCommunicationException;
	
	
	/**
	 * @return the used HttpConnectionManager
	 */
	MultiThreadedHttpConnectionManager getHttpConnectionManager();
}
