/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */
package com.openexchange.usm.api.exceptions;

import org.json.JSONObject;

/**
 * Exception that is generated if authentication on the OX server failed.<br>
 * Details about the failed authentication are stored in a JSONObject. This is either
 * directly the JSONObject returned from the OX server, or if the result could not be
 * parsed into a JSONObject, a JSONObject containing two fields:<br>
 * <ul>
 * <li>error: Error message from the OXJSONAccess service
 * <li>data: data returned from the OX server, either a JSONArray (if it could be parsed
 * as such an array), or a String containing the response
 * </ul>
 * Note that on severe communication problems (No http OK(200) status code or
 * HttpException/IOException) a OXCommunicationException is thrown.
 * 
 * @author afe
 *
 */
public class AuthenticationFailedException extends USMException {

	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;

	private final JSONObject _errorDetails;

	public AuthenticationFailedException(int errorCode, String message, JSONObject errorDetails) {
		super(errorCode, message);
		_errorDetails = errorDetails;
	}

	public JSONObject getErrorDetails() {
		return _errorDetails;
	}

	@Override
	public JSONObject getOxErrorForJSONResponse() {
		return _errorDetails;
	}

	@Override
	public JSONObject getErrorDetailsForJSONResponse() {
		return null;
	}

	@Override
    public String getErrorMessage() {
        String message = super.getMessage();
        if (_errorDetails != null) {
            if (message == null) {
                message = _errorDetails.toString();
            }
            else {
                message = super.getMessage() + ": " + _errorDetails.toString();
            }
        }
        return message;
    }

	@Override
    public String getMessage() {
        return getErrorMessage();
    }

	@Override
	public String toString() {
		String superToString = super.toString();
		if (_errorDetails == null)
			return superToString;
		return superToString + ':' + _errorDetails;
	}
}
