/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.api.datatypes;

import java.io.Serializable;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.usm.api.session.Session;

/**
 * DataType for attachments to OX PIM objects
 * 
 * @author afe
 */
public class PIMAttachmentsDataType extends AbstractDataType<PIMAttachments> {

    public PIMAttachmentsDataType() {
        super("Attachments");
    }

    @Override
    public PIMAttachments checkValue(Object o) {
        if (o != null && !(o instanceof PIMAttachments))
            generateTypeError(o);
        return (PIMAttachments) o;
    }

    @Override
    public void addToJSONArray(Session session, JSONArray parent, Object value) {
        parent.put((value == null) ? JSONObject.NULL : ((PIMAttachments) value).getTimestamp());
    }

    @Override
    public PIMAttachments extractFromJSONArray(Session session, JSONArray array, int index) throws JSONException {
        return (array.isNull(index) || array.getLong(index) == 0) ? null : new PIMAttachments(array.getLong(index));
    }

    @Override
    public void storeInJSONObject(Session session, JSONObject parent, String field, Object value) throws JSONException {
        parent.put(field, (value == null) ? JSONObject.NULL : ((PIMAttachments) value).getTimestamp());
    }

    @Override
    public PIMAttachments extractFromJSONObject(Session session, JSONObject jsonObject, String field) throws JSONException {
        return (jsonObject.isNull(field) || jsonObject.getLong(field) == 0) ? null : new PIMAttachments(jsonObject.getLong(field));
    }

    /**
     * Override standard method to make sure that created copies of DataObjects do not have the forceModified flag set for their
     * PIMAttachments fields
     */
    @Override
    public PIMAttachments createCopy(Object source) {
        PIMAttachments v = checkValue(source);
        return (v != null && v.isForceModified()) ? new PIMAttachments(v.getTimestamp(), v.getAttachments()) : v;
    }

    @Override
    public Serializable serialize(Object value) {
        return (value instanceof PIMAttachments) ? new SerializedPIMAttachments((PIMAttachments) value) : null;
    }

    @Override
    public PIMAttachments deserialize(Serializable value) {
        if (value instanceof SerializedPIMAttachments)
            return ((SerializedPIMAttachments) value).toPIMAttachments();
        return super.deserialize(value);
    }
}
