/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.api.datatypes;

import java.io.Serializable;
import org.json.JSONException;

/**
 * Abstract base class for common DataTypes.
 * 
 * @author afe
 * @param <T> Java type of object that is stored by the DataType
 */
public abstract class AbstractDataType<T extends Serializable> implements DataType<T> {
    
    private final String name;
    
    /**
     * Initializes a new {@link AbstractDataType}.
     * @param n
     */
    public AbstractDataType(String n) {
        name = n;
    }

    protected void generateTypeError(Object o) {
        throw new IllegalArgumentException(o.getClass().getName() + " is no " + getName());
    }

    @Override
    public T createNewObject() {
        return null;
    }

    /**
     * Must be overwritten for all DataTypes that do not use immutable Objects
     */
    @Override
    public T createCopy(Object source) {
        return checkValue(source);
    }

    @Override
    public Serializable serialize(Object value) {
        return createCopy(value);
    }

    @Override
    public T deserialize(Serializable value) {
        return createCopy(value);
    }

    @Override
    public boolean isEqual(Object object, Object object2) {
        return (object == null) ? object2 == null : object.equals(object2);
    }

    @Override
    public boolean isDefaultValue(Object object) {
        return object == null;
    }

    protected Number convertToNumber(Object o) throws JSONException {
        if (o instanceof Number)
            return (Number) o;
        try {
            return Double.valueOf(o.toString());
        } catch (NumberFormatException e) {
            throw new JSONException("Element is not a Number");
        }
    }
    
    /*
     * (non-Javadoc)
     * @see com.openexchange.usm.api.datatypes.DataType#getName()
     */
    @Override
    public String getName() {
        return name;
    }
}
