/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */
package com.openexchange.usm.api.contenttypes.common;

import com.openexchange.usm.api.datatypes.ArrayDataType;
import com.openexchange.usm.api.datatypes.DataType;
import com.openexchange.usm.api.datatypes.UnsortedArrayDataType;

/**
 * This class defines all relevant information about one field of a ContentType
 * 
 * @author afe
 *
 */
public class ContentTypeField implements Comparable<ContentTypeField> {

	private final String _name;
	private final DataType<?> _type;
	private final int _id;
	private final int _fieldWeight;
	private final boolean _isIDField;
	private final boolean _isParentFolderIDField;
	private final boolean _isRequiredForTransferToServer;

	/**
	 * Constructs a non-indexed ContentTypeField with the given name and type
	 * 
	 * @param id ContentType specific internal ID
	 * @param name non-null field name
	 * @param type non-null unique type name
	 */
	public ContentTypeField(int id, String name, DataType<?> type) {
		this(id, name, 0, type, false, false, false, false);
	}

	public ContentTypeField(int id, String name, int fieldWeight, DataType<?> type) {
		this(id, name, fieldWeight, type, false, false, false, false);
	}

	/**
	 * Constructs a ContentTypeField with the given name and type which may be indexed.
	 * The indexed parameter is just for convenience, the provided DataType will be
	 * encapsulated by an ArrayDataType to generate the DataType used for this
	 * ContentTypeField.
	 * 
	 * @param id ContentType specific internal ID
	 * @param name non-null field name
	 * @param type non-null unique type name
	 * @param isIndexed true if ContentTypeField is indexed
	 */
	public ContentTypeField(int id, String name, DataType<?> type, boolean isIndexed) {
		this(id, name, 0, type, isIndexed, false, false, false);
	}
	
	public ContentTypeField(int id, String name, DataType<?> type, boolean isIndexed, boolean isSortedIndexed) {
		this(id, name, 0, type, isIndexed, false, false, false, isSortedIndexed);
	}

	/**
	 * Constructs a ContentTypeField with the given name and type which may be indexed.
	 * The indexed parameter is just for convenience, the provided DataType will be
	 * encapsulated by an ArrayDataType to generate the DataType used for this
	 * ContentTypeField.
	 * 
	 * @param id ContentType specific internal ID
	 * @param name non-null field name
	 * @param type non-null unique type name
	 * @param isIndexed true if ContentTypeField is indexed
	 * @param idField If set to true, this field will be used as the ID field for DataObjects of the ContentType. Exactly the first field of a ContentType must have this flag set
	 * @param parentFolderIDField If set to true, this field will be used as the parent folder ID field for DataObjects of the ContentType. Exactly the second field of a ContentType must have this flag set
	 * @param requiredForTransferToServer If set to true, this field will be transferred to the server for changes even if it is not modified
	 */
	public ContentTypeField(int id, String name, DataType<?> type, boolean isIndexed, boolean idField,
			boolean parentFolderIDField, boolean requiredForTransferToServer) {
		this(id, name, 0, type, isIndexed, idField, parentFolderIDField, requiredForTransferToServer);
	}

	public ContentTypeField(int id, String name, int fieldWeight, DataType<?> type, boolean isIndexed, boolean idField,
			boolean parentFolderIDField, boolean requiredForTransferToServer) {
		this(id, name, fieldWeight, type, isIndexed, idField, parentFolderIDField, requiredForTransferToServer, true);
	}
	
	public ContentTypeField(int id, String name, int fieldWeight, DataType<?> type, boolean isIndexed, boolean idField,
			boolean parentFolderIDField, boolean requiredForTransferToServer, boolean isSortedIndexed) {
		if (name == null || type == null)
			throw new IllegalArgumentException("Name and/or type of field not specified");
		if (parentFolderIDField && idField)
			throw new IllegalArgumentException("Field can not contain both ID and parent folder ID");
		_id = id;
		_name = name;
		_isIDField = idField;
		_isParentFolderIDField = parentFolderIDField;
		_isRequiredForTransferToServer = requiredForTransferToServer;
		if(isIndexed) {
			_type = isSortedIndexed ? new ArrayDataType(type) : new UnsortedArrayDataType(type);
		} else {
			_type = type;
		}
		_fieldWeight = fieldWeight;
	}

	/**
	 * @return Name of this ContentTypeField, must be unique within the containing ContentType
	 */
	public String getFieldName() {
		return _name;
	}

	/**
	 * This method must return the datatype for a given field. This can be either
	 * one of the default datatypes specified in DataTypes or a unique custom data type
	 * 
	 * @return field type for the field
	 */
	public DataType<?> getFieldType() {
		return _type;
	}

	/**
	 * @return ContentType specific internal ID
	 */
	public int getFieldID() {
		return _id;
	}

	@Override
    public int compareTo(ContentTypeField o) {
		return _name.compareTo(o._name);
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + _name.hashCode();
		result = prime * result + _type.hashCode();
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		ContentTypeField other = (ContentTypeField) obj;
		if (!_name.equals(other._name))
			return false;
		if (!_type.equals(other._type))
			return false;
		return true;
	}

	@Override
	public String toString() {
		return _name + '(' + _type.getName() + ',' + _id + ')';
	}

	/**
	 * 
	 * @return true if this field stores the ID of a DataObject (which needs special treatment for some cases)
	 */
	public boolean isIDField() {
		return _isIDField;
	}

	/**
	 * 
	 * @return true if this field stores the parent folder ID of a DataObject (for sanity check of ContentTypes)
	 */
	public boolean isParentFolderIDField() {
		return _isParentFolderIDField;
	}

	/**
	 * 
	 * @return true if this field should be transferred to the OX server on changes even if it is not modified
	 */
	public boolean isRequiredForTransferToServer() {
		return _isRequiredForTransferToServer;
	}

	public int getFieldWeight() {
		return _fieldWeight;
	}
}
