/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */
package com.openexchange.usm.api.cache;

import java.io.Serializable;

/**
 * The Cache class extends the ICacheAccess interface, adding a
 * methods to store in and read  data objects from cache.
 *
 * @author afe
 *
 * @param <K> key type
 * @param <V> value type
 */
public interface Cache<K extends Serializable, V extends Serializable> {
	/**
	 * Basic get method.
	 * <p>
	 * @param name
	 * @return Object or null if not found.
	 */
	V get(K name);

	/**
	 * Puts in cache if an item does not exist with the name in that region.
	 * <p>
	 * @param name - any serializable object.
	 *               If removing key/value pairs in a group is needed than the name should be String in a hierarchical form X:Y:Z
	 * @param obj  - can be any serializable object
	 */
	void putSafe(K name, V obj);

	/**
	 * Puts and/or overrides an element with the name in that region.
	 * <p>
	 * @param name - any serializable object.
	 *               If removing key/value pairs in a group is needed than the name should be String in a hierarchical form X:Y:Z
	 * @param obj - can be any serializable object
	 * @param invalidate - <code>true</code> to trigger remote invalidation processing for the cache entry, <code>false</code>, otherwise.
	 */
	void put(K name, V obj, boolean invalidate);

	/**
	 * Remove an object for this key if one exists, else do nothing.
	 * To remove more key/value pairs at once use hierarchical String keys in a form X:Y:Z
	 * ex. remove ("X:Y:") will remove all keys which start with "X:Y:"
	 * <p>
	 * @param name
	 */
	void remove(K name);

	/**
	 * Can be called to free resources if the cache will not be used any more.
	 */
	void dispose();
}
