/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.util;

import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import com.openexchange.usm.api.database.EncapsulatedConnection;

public class SQLToolkit {

	public static final String SQL_SEQUENCE_STATEMENT_PART1 = "CREATE TABLE ";

	public static final String SQL_SEQUENCE_STATEMENT_PART2 = " (cid INT4 UNSIGNED NOT NULL, id INT4 UNSIGNED NOT NULL, PRIMARY KEY (cid)) ENGINE=InnoDB DEFAULT CHARSET=utf8 COLLATE=utf8_bin";

	public static final String INITIALIZE_SQL_SEQUENCE_STATEMENT_PART1 = "INSERT INTO ";

	public static final String INITIALIZE_SQL_SEQUENCE_STATEMENT_PART2 = " VALUES( ";

	public static final String INITIALIZE_SQL_SEQUENCE_STATEMENT_PART3 = ", 0)";

	public static final String INITIALIZE_SQL_SEQUENCE_STATEMENT_PART3_INIT1 = ", 1)";

	public static String createSequenceStatement(String tableName) {
		return SQL_SEQUENCE_STATEMENT_PART1 + tableName + SQL_SEQUENCE_STATEMENT_PART2;
	}

	public static String createInitilizeSequenceStatement(String tableName, int contextID) {
		return INITIALIZE_SQL_SEQUENCE_STATEMENT_PART1 + tableName + INITIALIZE_SQL_SEQUENCE_STATEMENT_PART2 + contextID + INITIALIZE_SQL_SEQUENCE_STATEMENT_PART3;
	}

	public static String createInsertFirstValueSequenceStatement(String tableName, int contextID) {
		return INITIALIZE_SQL_SEQUENCE_STATEMENT_PART1 + tableName + INITIALIZE_SQL_SEQUENCE_STATEMENT_PART2 + contextID + INITIALIZE_SQL_SEQUENCE_STATEMENT_PART3_INIT1;
	}

	public static String createSequenceProcedure(String tableName) {
		StringBuilder result = new StringBuilder(500);
		result.append("CREATE PROCEDURE ");
		result.append("get_");
		result.append(tableName);
		result.append("_id(IN context INT4 UNSIGNED)");
		result.append(" NOT DETERMINISTIC MODIFIES SQL DATA BEGIN UPDATE ");
		result.append(tableName);
		result.append(" SET id=id+1 WHERE cid=context;  SELECT id FROM ");
		result.append(tableName);
		result.append(" WHERE cid=context; END");
		return result.toString();
	}

	public static String createGetNextCall(String tableName, int contextID) {
		return "CALL get_" + tableName + "_id(" + contextID + ")";
	}

	public static String dropSequenceProcedure(String sequenceName) {
		return "DROP PROCEDURE IF EXISTS get_" + sequenceName + "_id";
	}

	public static int getNextIdFromSequence(EncapsulatedConnection con, String sequenceName, int contextID) throws SQLException {
		PreparedStatement stmt = null;
		ResultSet result = null;
		try {
			con.setAutoCommit(false);
			stmt = con.prepareStatement("UPDATE " + sequenceName + " SET id=id+1 WHERE cid = ?");
			stmt.setInt(1, contextID);
			stmt.execute();
			stmt.close();
			stmt = con.prepareStatement("SELECT id FROM " + sequenceName + " WHERE cid = ?");
			stmt.setInt(1, contextID);
			result = stmt.executeQuery();
			if (result.next()) {
				int rv = result.getInt(1);
				stmt.close();
				con.commit();
				return rv;
			}
			stmt.close();

			return getInitialIdFromSequence(con, sequenceName, contextID);
			
		} catch (SQLException e) {
			con.rollback();
			throw e;
		} finally {
			try {
				con.setAutoCommit(true);
			} finally {
				Toolkit.close(result);
				Toolkit.close(stmt);
			}
		}
	}

	/**
	 * Returns the initial id for the specified sequence and context.<br/>
	 * In case of a deadlock (Bug 26351), the statement will be executed 5 times before failing completely.<br/><br/>
	 * 
	 * If a deadlock is detected, it is possible that on the next iteration/execution, the id is already set to '1'.
	 * Therefore, we invoke the getNextIdFromSequence which guarantees to avoid duplicates.
	 * 
	 * @see <a href="http://dev.mysql.com/doc/refman/5.6/en/connector-j-reference-error-sqlstates.html">
	 * Mapping MySQL Error Numbers to JDBC SQLState Codes</a>
	 * 
	 * @param con
	 * @param sequenceName
	 * @param contextID
	 * @return
	 * @throws SQLException
	 */
	private static int getInitialIdFromSequence(EncapsulatedConnection con, String sequenceName, int contextID) throws SQLException {
		boolean exception = false;
		int tries = 1;
		int ret = 0;
		do {
			PreparedStatement stmt = null;
			ResultSet result = null;

			try {
				con.setAutoCommit(false);
				stmt = con.prepareStatement("INSERT INTO " + sequenceName + " VALUES(?, 1)");
				stmt.setInt(1, contextID);
				stmt.execute();
				stmt.close();
				con.commit();
				ret = 1;
				return ret;
			} catch (SQLException e) {
				exception = true;
				
				if (e.getSQLState().equals("41000") || e.getSQLState().equals("40001")) { //ER_LOCK_DEADLOCK
					if (tries >= 5) {
						throw e;
					} else {
						try {
							Thread.sleep(500);
						} catch (InterruptedException e1) {/*ignore*/}
					}
				} else if (e.getSQLState().equals("S1009") || e.getSQLState().equals("23000")) { //ER_DUP_KEY
					return getNextIdFromSequence(con, sequenceName, contextID);
				}
			} finally {
				try {
					con.setAutoCommit(true);
				} finally {
					Toolkit.close(result);
					Toolkit.close(stmt);
				}
			}
		} while (exception && (tries++ < 5));
		
		return ret;
	}
}