/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.util;

import java.util.HashSet;
import java.util.Set;

import org.json.JSONException;
import org.json.JSONObject;

public class OXJSONUtil {

    private static final String WARNING_CATEGORY = "13";

    private static final String CATEGORY_FIELD = "category";

    public static final String CATEGORY_RESOURCE_TEMPORARY_DOWN = "4";

    public static final String CATEGORY_SUBSYSTEM_DOWN = "5";

    public static final String CATEGORY_SOCKET_CONNECTION_CORRUPT = "6";

    public static final String CATEGORY_RESOURCE_FULL_OR_BUSY = "11";

    public static final String ERROR_CODE_OBJECT_CHANGED = "FLD-0001";

    public static final String ERROR_CODE_FOLDER_MUST_NOT_BE_DELETED = "FLD-0090";

    public static final String ERROR_FIELD = "error";

    public static final String ERROR_CODE_FIELD = "code";

    public static final String DATA_FIELD = "data";

    public static final String UNKNOWN_ERROR_CODE = "unknown";

    public static final Set<String> _RELOGIN_ERROR_CODES = new HashSet<String>();
    
    private static final Set<String> _OBJECT_MISSING_ERROR_CODES = new HashSet<String>();
    
    

    public static boolean isOXError(JSONObject object) {
        if (!object.has(ERROR_FIELD))
            return false;
        if (!object.has(CATEGORY_FIELD))
            return true;
        if (!WARNING_CATEGORY.equals(getJSONErrorCategory(object)))
            return true;
        return getJSONErrorCode(object).equals(ERROR_CODE_FOLDER_MUST_NOT_BE_DELETED);
    }

    public static String getJSONErrorCode(JSONObject object) {
        if (!object.has(ERROR_FIELD))
            return "";
        try {
            return object.getString(ERROR_CODE_FIELD);
        } catch (JSONException e) {
            return UNKNOWN_ERROR_CODE;
        }
    }

    public static String getJSONErrorCategory(JSONObject object) {
        return object.optString(CATEGORY_FIELD, null);
    }
    
    public static void setReloginErrorCodes(String reloginErrorCodes) {
    	setErrorCodes(reloginErrorCodes,_RELOGIN_ERROR_CODES);
    }
    
    public static void setMissingObjectErrorCodes(String missingObjectErrorCodes) {
    	setErrorCodes(missingObjectErrorCodes,_OBJECT_MISSING_ERROR_CODES);
    }

    private static void setErrorCodes(String errorCodesString, Set<String> errorCodesCollection) {
    	errorCodesCollection.clear();
        for (String s : errorCodesString.split(",")) {
            String code = s.trim();
            if (code.length() > 0)
            	errorCodesCollection.add(code);
        }
    }
    
    public static boolean isMissingObjectErrorCode(JSONObject oxError) {
        return _OBJECT_MISSING_ERROR_CODES.contains(getJSONErrorCode(oxError));
    }
    
    public static boolean isReloginErrorCode(JSONObject oxError) {
        return  _RELOGIN_ERROR_CODES.contains(OXJSONUtil.getJSONErrorCode(oxError));
    }

}
