/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.session.impl;

import java.lang.ref.ReferenceQueue;
import java.lang.ref.SoftReference;
import java.lang.ref.WeakReference;

/**
 * Stores a SoftReference to a SessionImpl object, and also tracks the last access and allows access to the time of the last access to the
 * SessionImpl. {@link SessionReference}
 * 
 * @author <a href="mailto:afe@microdoc.de">Alexander Feess</a>
 */
public class SessionReference {

    /**
     * SoftReference to keep SessionImpl in memory until the memory is needed for other objects
     */
    private final SoftReference<SessionImpl> _softReference;

    /**
     * WeakReference to allow access to SessionImpl without updating the last access timestamp in the SoftReference
     */
    private final WeakReference<SessionImpl> _weakReference;

    /**
     * Own timestamp of last access since the timestamp in SoftReference is not accessible
     */
    private long _lastAccess;

    public SessionReference(SessionImpl session, ReferenceQueue<? super SessionImpl> queue) {
        _softReference = new SoftReference<SessionImpl>(session, queue);
        _weakReference = new WeakReference<SessionImpl>(session);
        _lastAccess = System.currentTimeMillis();
    }

    /**
     * @return SoftReference that is referencing the SessionImpl
     */
    public SoftReference<SessionImpl> getReference() {
        return _softReference;
    }

    /**
     * @return Cached SessionImpl or null if it has been removed by JVM, updates the last access timestamp
     */
    public SessionImpl get() {
        _lastAccess = System.currentTimeMillis();
        return _softReference.get();
    }

    /**
     * @return Cached SessionImpl or null if it has been removed by JVM, does not update the last access timestamp
     */
    public SessionImpl poll() {
        return _weakReference.get();
    }

    /**
     * @return Timestamp of last access via get() (or since creation if get() has not been called)
     */
    public long getLastAccess() {
        return _lastAccess;
    }
}
