/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.session.impl;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import javax.management.MalformedObjectNameException;
import javax.management.NotCompliantMBeanException;
import javax.management.ObjectName;
import org.apache.commons.httpclient.MultiThreadedHttpConnectionManager;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.json.JSONObject;
import com.openexchange.annotation.NonNull;
import com.openexchange.cluster.timer.ClusterTimerService;
import com.openexchange.exception.OXException;
import com.openexchange.log.LogProperties;
import com.openexchange.management.ManagementService;
import com.openexchange.timer.ScheduledTimerTask;
import com.openexchange.timer.TimerService;
import com.openexchange.usm.api.USMVersion;
import com.openexchange.usm.api.cache.SyncStateCacheProvider;
import com.openexchange.usm.api.configuration.ConfigurationManager;
import com.openexchange.usm.api.configuration.ConfigurationProperties;
import com.openexchange.usm.api.contenttypes.common.ContentType;
import com.openexchange.usm.api.contenttypes.common.ContentTypeManager;
import com.openexchange.usm.api.contenttypes.common.DefaultContentTypes;
import com.openexchange.usm.api.contenttypes.folder.FolderContentType;
import com.openexchange.usm.api.database.StorageAccessException;
import com.openexchange.usm.api.exceptions.AuthenticationFailedException;
import com.openexchange.usm.api.exceptions.OXCommunicationException;
import com.openexchange.usm.api.exceptions.USMAccessDeniedException;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.exceptions.USMInvalidConfigurationException;
import com.openexchange.usm.api.exceptions.USMStartupException;
import com.openexchange.usm.api.exceptions.USMStorageException;
import com.openexchange.usm.api.mapping.storage.ContextUUIDMappingStorage;
import com.openexchange.usm.api.mapping.storage.FolderIDMappingStorage;
import com.openexchange.usm.api.ox.json.OXJSONAccess;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.Folder;
import com.openexchange.usm.api.session.Session;
import com.openexchange.usm.api.session.SessionInitializer;
import com.openexchange.usm.api.session.SessionManager;
import com.openexchange.usm.api.session.assets.SessionID;
import com.openexchange.usm.api.session.assets.UserID;
import com.openexchange.usm.api.session.storage.PersistentSessionDataStorage;
import com.openexchange.usm.api.session.storage.PersistentSyncStateStorage;
import com.openexchange.usm.session.cache.MemorySyncStateCacheProvider;
import com.openexchange.usm.session.dataobject.DataObjectSet;
import com.openexchange.usm.session.dataobject.FolderImpl;
import com.openexchange.usm.session.jmx.USMCacheInformation;
import com.openexchange.usm.session.jmx.USMSessionInformation;
import com.openexchange.usm.session.jmx.USMSessionInformationMBean;
import com.openexchange.usm.session.storage.SessionStorage;
import com.openexchange.usm.session.sync.IncrementalContentSyncer;
import com.openexchange.usm.session.sync.OXDataCache;
import com.openexchange.usm.session.sync.SlowContentSyncer;
import com.openexchange.usm.session.sync.SynchronizationLock;
import com.openexchange.usm.session.tasks.USMSessionCacheCleanupLogInfoTask;
import com.openexchange.usm.session.tasks.USMSessionCacheCleanupTask;
import com.openexchange.usm.session.tasks.USMSessionCleanupTask;
import com.openexchange.usm.util.JSONToolkit;
import com.openexchange.usm.util.TempFileStorage;
import com.openexchange.usm.util.UUIDToolkit;

public class SessionManagerImpl implements SessionManager {

    // Stores interval after which an automatic access check is performed.
    // LATER we might want to make this constant configurable
    private final static long ACCESS_CHECK_INTERVAL = 300000L;

    private final static long ONE_HOUR_IN_MILLISECONDS = 60L * 60L * 1000L;

    private final static long ONE_DAY_IN_MILLISECONDS = 24L * ONE_HOUR_IN_MILLISECONDS;

    private static final String LAST_ACCESS_FIELD_NAME = "USM_ACCESS";

    private static final long LAST_ACCESS_UPDATE_INTERVAL = 3L * ONE_HOUR_IN_MILLISECONDS;

    public final static List<String> SPECIAL_FOLDERS = new ArrayList<String>();

    private final static List<String> SPECIAL_FOLDERS_VIEW = Collections.unmodifiableList(SPECIAL_FOLDERS);

    static {
        SPECIAL_FOLDERS.add(DefaultContentTypes.GROUPS_ID);
        SPECIAL_FOLDERS.add(DefaultContentTypes.RESOURCES_ID);
    }

    private static final Log LOG = LogFactory.getLog(SessionManagerImpl.class);

    private final OXJSONAccess _oxAjaxAccess;

    private final ContentTypeManager _contentTypeManager;

    private final FolderContentType _folderContentType;

    private final SessionStorage _sessionStorage;

    private final OXDataCache _dataCache;

    private final ConfigurationManager _configurationManager;

    private final FolderIDMappingStorage _mappingService;

    private final ContextUUIDMappingStorage _uuidService;

    private ManagementService _managementService;

    private final USMCacheInformation _usmCacheInformation = new USMCacheInformation(this);

    private ObjectName _jmxSessionObjectName;

    private String _usmInterface;

    // A value < 0 indicates percent, >= 0 an absolute value
    private int _waitForChangesEmailPullDelay;

    private int _waitForChangesEmailMinPullDelay = -1;

    private int _maxSyncStatesInDB = ConfigurationProperties.SYNC_MAX_STATES_IN_DB_DEFAULT;

    private int _maxObjectsPerFolder = ConfigurationProperties.OBJECTS_IN_FOLDER_SYNC_LIMIT_DEFAULT;

    private long _sessionStorageTimeLimit = ConfigurationProperties.USM_SESSION_STORAGE_TIME_LIMIT_DEFAULT * ONE_DAY_IN_MILLISECONDS;

    private TimerService _timerService;

    private ScheduledTimerTask _cleanupTask;

    private ScheduledTimerTask _sessionCacheCleanupTask;

    private ScheduledTimerTask _sessionCacheCleanupLogInfoTask;

    private int _syncRetryCount;

    private int _sessionCacheTimeout;

    private int _syncStateCacheTimeout;

    private int _attachmentCountLimitPerObject;

    private long _attachmentSizeLimitPerObject;

    private final SyncStateCacheProvider _defaultCacheProvider = new MemorySyncStateCacheProvider(this);

    private final PersistentSyncStateStorage syncStateStorage;

    private final PersistentSessionDataStorage sessionDataStorage;

    private SyncStateCacheProvider _syncStateCacheProvider;

    private final List<String> _waitEnabledProtocols = new ArrayList<String>(); // List because we only have 1 or 2 enabled protocols

    private static final ConcurrentMap<SessionID, AcquiredLatch> SYNCHRONIZER = new ConcurrentHashMap<SessionID, AcquiredLatch>(256);

    private ClusterTimerService clusterTimerService;

    /**
     * This constructor is used only for tests of the IncrementalContentSyncer and SlowContentSyncer.
     * 
     * @param log
     */
    public SessionManagerImpl() {
        _dataCache = new OXDataCache(0, 0);
        _oxAjaxAccess = null;
        _contentTypeManager = null;
        _configurationManager = null;
        _folderContentType = null;
        _sessionStorage = null;
        _mappingService = null;
        _uuidService = null;
        syncStateStorage = null;
        sessionDataStorage = null;
    }

    /**
     * Initializes a new {@link SessionManagerImpl}.
     * 
     * @param ajaxAccess
     * @param contentTypeManager
     * @param folderContentType
     * @param configurationManager
     * @param dbAccess
     * @param mappingService
     * @param uuidService
     * @param ssStorage
     * @param psdStorage
     */
    public SessionManagerImpl(final OXJSONAccess ajaxAccess, final ContentTypeManager contentTypeManager, final FolderContentType folderContentType, final ConfigurationManager configurationManager, final FolderIDMappingStorage folderIdMappingService, final ContextUUIDMappingStorage uuidService, final PersistentSyncStateStorage ssStorage, final PersistentSessionDataStorage psdStorage) {
        _oxAjaxAccess = ajaxAccess;
        _contentTypeManager = contentTypeManager;
        _folderContentType = folderContentType;
        _configurationManager = configurationManager;
        _mappingService = folderIdMappingService;
        _uuidService = uuidService;
        syncStateStorage = ssStorage;
        sessionDataStorage = psdStorage;
        try {
            _syncRetryCount = _configurationManager.getProperty(ConfigurationProperties.SYNC_CONCURRENT_MODIFICATION_MAX_RETRIES_PROPERTY, ConfigurationProperties.SYNC_CONCURRENT_MODIFICATION_MAX_RETRIES_DEFAULT, false);
        } catch (final USMInvalidConfigurationException e) {
            throw new USMStartupException(USMSessionManagementErrorCodes.CONFIGURATION_ERROR, "Configuration for Sync retries is invalid", e);
        }
        _waitForChangesEmailPullDelay = readEmailPullDelayProperty();
        _waitForChangesEmailMinPullDelay = readEmailMinPullDelayProperty();
        try {
            _maxSyncStatesInDB = _configurationManager.getProperty(ConfigurationProperties.SYNC_MAX_STATES_IN_DB_PROPERTY, ConfigurationProperties.SYNC_MAX_STATES_IN_DB_DEFAULT, false);
            if (_maxSyncStatesInDB < 2) {
                throw new USMStartupException(USMSessionManagementErrorCodes.CONFIGURATION_ERROR_5, "Configuration for max_states_in_db must be greater or equal to 2");
            }
        } catch (final USMInvalidConfigurationException e) {
            throw new USMStartupException(USMSessionManagementErrorCodes.CONFIGURATION_ERROR_6, "Configuration for max_states_in_db is invalid", e);
        }

        try {
            _usmInterface = _configurationManager.getProperty(ConfigurationProperties.USM_INTERFACE_CONFIG_PROPERTY, ConfigurationProperties.USM_INTERFACE_CONFIG_DEFAULT, true);
        } catch (final USMInvalidConfigurationException e) {
            throw new USMStartupException(USMSessionManagementErrorCodes.CONFIGURATION_ERROR_8, "Configuration for usm interface module is invalid", e);
        }

        try {
            _sessionStorageTimeLimit = _configurationManager.getProperty(ConfigurationProperties.USM_SESSION_STORAGE_TIME_LIMIT_PROPERTY, ConfigurationProperties.USM_SESSION_STORAGE_TIME_LIMIT_DEFAULT, false) * ONE_DAY_IN_MILLISECONDS;
        } catch (final USMInvalidConfigurationException e) {
            throw new USMStartupException(USMSessionManagementErrorCodes.CONFIGURATION_ERROR_9, "Invalid configuration for " + ConfigurationProperties.USM_SESSION_STORAGE_TIME_LIMIT_PROPERTY, e);
        }

        int cacheTimeLimitSyncComplete;
        int cacheTimeLimitSyncIncomplete;
        try {
            cacheTimeLimitSyncComplete = _configurationManager.getProperty(ConfigurationProperties.SYNC_CACHE_TIME_LIMIT_COMPLETE_PROPERTY, ConfigurationProperties.SYNC_CACHE_TIME_LIMIT_COMPLETE_DEFAULT, false);
        } catch (final USMInvalidConfigurationException e) {
            throw new USMStartupException(USMSessionManagementErrorCodes.CONFIGURATION_ERROR_10, "Invalid configuration for " + ConfigurationProperties.SYNC_CACHE_TIME_LIMIT_COMPLETE_PROPERTY, e);
        }
        try {
            cacheTimeLimitSyncIncomplete = _configurationManager.getProperty(ConfigurationProperties.SYNC_CACHE_TIME_LIMIT_INCOMPLETE_PROPERTY, ConfigurationProperties.SYNC_CACHE_TIME_LIMIT_INCOMPLETE_DEFAULT, false);
        } catch (final USMInvalidConfigurationException e) {
            throw new USMStartupException(USMSessionManagementErrorCodes.CONFIGURATION_ERROR_11, "Invalid configuration for " + ConfigurationProperties.SYNC_CACHE_TIME_LIMIT_INCOMPLETE_PROPERTY, e);
        }
        try {
            _sessionCacheTimeout = _configurationManager.getProperty(ConfigurationProperties.USM_CACHE_SESSION_TIMEOUT_PROPERTY, ConfigurationProperties.USM_CACHE_SESSION_TIMEOUT_DEFAULT, false);
        } catch (final USMInvalidConfigurationException e) {
            throw new USMStartupException(USMSessionManagementErrorCodes.CONFIGURATION_ERROR_14, "Invalid configuration for " + ConfigurationProperties.USM_CACHE_SESSION_TIMEOUT_PROPERTY, e);
        }
        try {
            _syncStateCacheTimeout = _configurationManager.getProperty(ConfigurationProperties.USM_CACHE_SYNC_STATE_TIMEOUT_PROPERTY, ConfigurationProperties.USM_CACHE_SYNC_STATE_TIMEOUT_DEFAULT, false);
        } catch (final USMInvalidConfigurationException e) {
            throw new USMStartupException(USMSessionManagementErrorCodes.CONFIGURATION_ERROR_15, "Invalid configuration for " + ConfigurationProperties.USM_CACHE_SYNC_STATE_TIMEOUT_PROPERTY, e);
        }

        try {
            _maxObjectsPerFolder = _configurationManager.getProperty(ConfigurationProperties.OBJECTS_IN_FOLDER_SYNC_LIMIT_PROPERTY, ConfigurationProperties.OBJECTS_IN_FOLDER_SYNC_LIMIT_DEFAULT, false);
        } catch (final USMInvalidConfigurationException e) {
            throw new USMStartupException(USMSessionManagementErrorCodes.CONFIGURATION_ERROR_16, "Invalid configuration for " + ConfigurationProperties.OBJECTS_IN_FOLDER_SYNC_LIMIT_PROPERTY, e);
        }

        try {
            _attachmentCountLimitPerObject = _configurationManager.getProperty(ConfigurationProperties.ATTACHMENT_COUNT_LIMIT_PER_OBJECT_PROPERTY, ConfigurationProperties.ATTACHMENT_COUNT_LIMIT_PER_OBJECT_DEFAULT, false);
        } catch (final USMInvalidConfigurationException e) {
            throw new USMStartupException(USMSessionManagementErrorCodes.CONFIGURATION_ERROR_17, "Invalid configuration for " + ConfigurationProperties.ATTACHMENT_COUNT_LIMIT_PER_OBJECT_PROPERTY, e);
        }

        try {
            _attachmentSizeLimitPerObject = (long) _configurationManager.getProperty(ConfigurationProperties.ATTACHMENT_SIZE_LIMIT_PER_OBJECT_PROPERTY, ConfigurationProperties.ATTACHMENT_SIZE_LIMIT_PER_OBJECT_DEFAULT, false);
        } catch (final USMInvalidConfigurationException e) {
            throw new USMStartupException(USMSessionManagementErrorCodes.CONFIGURATION_ERROR_18, "Invalid configuration for " + ConfigurationProperties.ATTACHMENT_SIZE_LIMIT_PER_OBJECT_PROPERTY, e);
        }

        try {
            TempFileStorage.setTempFileTimeout(_configurationManager.getProperty(ConfigurationProperties.TEMP_FILES_TIMEOUT_PROPERTY, ConfigurationProperties.TEMP_FILES_TIMEOUT_DEFAULT, false));
        } catch (final USMInvalidConfigurationException e) {
            throw new USMStartupException(USMSessionManagementErrorCodes.CONFIGURATION_ERROR_19, "Invalid configuration for " + ConfigurationProperties.TEMP_FILES_TIMEOUT_PROPERTY, e);
        }

        try {
            SynchronizationLock.setLockTimeout(_configurationManager.getProperty(ConfigurationProperties.SYNC_LOCK_TIMEOUT_PROPERTY, ConfigurationProperties.SYNC_LOCK_TIMEOUT_DEFAULT, false));
        } catch (final USMInvalidConfigurationException e) {
            throw new USMStartupException(USMSessionManagementErrorCodes.CONFIGURATION_ERROR_20, "Invalid configuration for " + ConfigurationProperties.SYNC_LOCK_TIMEOUT_PROPERTY, e);
        }

        _sessionStorage = new SessionStorage(this);
        _dataCache = new OXDataCache(cacheTimeLimitSyncComplete, cacheTimeLimitSyncIncomplete);
        setTimerService(_timerService); // Initialize timers if TimerService was already set

        LOG.info("USM SessionManager activated");
        LOG.info("Open-Xchange USM Version " + USMVersion.VERSION + " (Build " + USMVersion.BUILD_NUMBER + ") successfully started");
    }

    public UUID getContextUUID(final int cid) throws StorageAccessException, USMStorageException {
        return _uuidService == null ? null : _uuidService.getContextUUID(cid);
    }

    public void deactivate() {
        deactivateTimers();
        deactivateUSMSessionCleanupTask();
        _sessionStorage.shutdown(_oxAjaxAccess);
        _waitForChangesEmailPullDelay = 0;
        _waitForChangesEmailMinPullDelay = -1;
        LOG.info("USM SessionManager deactivated");
    }

    @Override
    public Session getSession(final String user, final String password, final String protocol, final String device) throws USMException {
        return getSession(user, password, protocol, device, null, "127.0.0.1", new HashMap<String, String>());
    }

    /*
     * Optimised code. Does not iterate over all contexts anymore. Instead, the code first looks up in the cache for the USM session. If
     * it's not there then tries to login and determine the context id from the retrieved session. Finally it performs a database check only
     * on the previously determined context id.
     */
    // TODO This method is only called by 1 special handler. We may need to talk to the Outlook-Addin-developer(s) if the special
    // error that is reported if the session is not known to USM is really required. If not, we could remove this code
    @Override
    public boolean existsSessionInDB(final String user, final String password, final String protocol, final String device) throws USMException {
        final SessionID id = new SessionID(user, protocol, device);
        SessionImpl session = _sessionStorage.getSession(id);
        if (session != null) {
            return session.checkProperDBStorage();
        }

        // FIXME getSession() already creates the session in the DB if it hasn't existed yet. So existsSessionInDB() will now always return
        // true (and the code following the getSession() will always find the session in the DB)
        session = (SessionImpl) getSession(user, password, protocol, device);
        return sessionDataStorage.sessionExistsForDevice(session);
    }

    private static AcquiredLatch acquireFor(final SessionID key) {
        AcquiredLatch latch = SYNCHRONIZER.get(key);
        if (null == latch) {
            final AcquiredLatch newLatch = new AcquiredLatch(Thread.currentThread(), new CountDownLatch(1));
            // it could happen that a concurrent thread is faster
            latch = SYNCHRONIZER.putIfAbsent(key, newLatch);
            if (null == latch) {
                latch = newLatch;
            }
        }
        return latch;
    }

    private static void releaseFor(final SessionID key) {
        SYNCHRONIZER.remove(key);
    }

    @Override
    public @NonNull Session getSession(final String user, final String password, final String protocol, final String device, final SessionInitializer initializer, final String clientIP, final Map<String, String> xHeaders) throws USMException {
        final SessionID id = new SessionID(user, protocol, device);
        final SessionImpl session = _sessionStorage.getSession(id);
        if (session != null) {
            boolean removeSession = true;
            try {
                if (!session.getPassword().equals(password)) {
                    session.setPassword(password);
                    session.setClientIp(clientIP);
                    _oxAjaxAccess.login(session);
                }
                session.updateClientIp(clientIP);
                if (recheckUSMAccess(initializer, session)) {
                    updateLastUSMAccess(session);
                    removeSession = false;
                    applyMDCProperties(session);
                    return session;
                }
            } finally {
                if (removeSession) {
                    _sessionStorage.removeSession(session);
                }
            }
            // Cached Session is no longer present in DB, start from scratch
            _sessionStorage.removeSession(session);
        }

        final AcquiredLatch acquiredLatch = acquireFor(id);
        final CountDownLatch latch = acquiredLatch.latch;
        if (Thread.currentThread() == acquiredLatch.owner) {
            // Initialize a new session
            try {
                final Session newSession = initNewSessionFor(user, password, protocol, device, initializer, clientIP, xHeaders);
                acquiredLatch.result.set(newSession);
                applyMDCProperties(newSession);
                return newSession;
            } catch (final USMException e) {
                acquiredLatch.result.set(e);
                throw e;
            } catch (final Exception e) {
                final USMException usmExc = new USMException(USMSessionManagementErrorCodes.SESSION_INITIALIZATION_FAILED, e);
                acquiredLatch.result.set(usmExc);
                throw usmExc;
            } finally {
                latch.countDown();
                releaseFor(id);
            }
        }
        // A concurrent thread is already running
        try {
            LOG.debug("I need to wait for a concurrent thread which does the login already for session " + id);

            // Need to await 'til login done by concurrent thread
            latch.await();

            // Check if already locally available...
            final Object result = acquiredLatch.result.get();

            if (result instanceof USMException) {
                throw (USMException) result;
            }
            // return the session which was created by the concurrent thread
            final Session newSession = (Session) result;
            applyMDCProperties(newSession);
            return newSession;
        } catch (final InterruptedException e) {
            throw new USMException(USMSessionManagementErrorCodes.SESSION_INITIALIZATION_FAILED, e);
        }
    }

    @Override
    public void removeSession(Session session) {
        if (session != null && session instanceof SessionImpl)
            _sessionStorage.removeSession((SessionImpl) session);
    }
    
    /**
     * Applies the {@link LogProperties.Name.SESSION_CONTEXT_ID}, {@link LogProperties.Name.SESSION_USER_ID}
     * and the {@link LogProperties.Name.SESSION_SESSION_ID} (masqueraded as an USM session id) values
     * to the MDC filter
     * 
     * @param session The USM {@link Session}
     */
    private void applyMDCProperties(final Session session) {
        LogProperties.put(LogProperties.Name.SESSION_SESSION_ID, session.getSessionId());
        LogProperties.put(LogProperties.Name.SESSION_CONTEXT_ID, session.getContextId());
        LogProperties.put(LogProperties.Name.SESSION_USER_ID, session.getUserIdentifier());
    }

    private @NonNull Session initNewSessionFor(final String user, final String password, final String protocol, final String device, final SessionInitializer initializer, final String clientIP, final Map<String, String> xHeaders) throws USMException {
        final SessionImpl session = new SessionImpl(this, user, password, protocol, device, initializer);
        session.setClientIp(clientIP);
        session.setXRequestHeaders(xHeaders);
        _oxAjaxAccess.login(session);
        try {
            final JSONObject configuration = _oxAjaxAccess.getConfiguration(session);
            checkUSMAccess(session, configuration);
            session.setLastAccessCheck(System.currentTimeMillis());
            session.initialize(configuration);
        } catch (final USMException e) {
            _oxAjaxAccess.logout(session);
            throw e;
        }
        _sessionStorage.storeSession(session);
        _usmCacheInformation.newSessionCreated();
        updateLastUSMAccess(session);
        return session;
    }

    private static void updateLastUSMAccess(final SessionImpl session) throws USMStorageException, StorageAccessException {
        final long now = System.currentTimeMillis();
        final String oldStringValue = session.getPersistentField(LAST_ACCESS_FIELD_NAME);
        if (oldStringValue != null) {
            try {
                final long lastAccess = Long.parseLong(oldStringValue);
                if (now - lastAccess < LAST_ACCESS_UPDATE_INTERVAL)
                 {
                    return; // Last access timestamp is not too old, do nothing
                }
            } catch (final NumberFormatException ignored) {
                // ignore any invalid data, and simple fall through and store the current timestamp
            }
        }
        session.setPersistentField(LAST_ACCESS_FIELD_NAME, String.valueOf(now));
    }

    private boolean recheckUSMAccess(final SessionInitializer initializer, final SessionImpl session) throws USMException {
        final long now = System.currentTimeMillis();
        if (session.getLastAccessCheck() >= now - ACCESS_CHECK_INTERVAL) {
            return true;
        }
        checkUSMAccess(session, null);
        initializer.checkAccess(session);
        session.setLastAccessCheck(now);
        return session.checkProperDBStorage();
    }

    private void checkUSMAccess(final SessionImpl session, final JSONObject configuration) throws USMAccessDeniedException, AuthenticationFailedException {
        if (_usmInterface == null || _usmInterface.length() == 0) {
            final JSONObject usmModule = getUSMModule(session, configuration, "com.openexchange.usm");
            if (usmModule != null && usmModule.optBoolean("active")) {
                return;
            }
        } else {
            final JSONObject oxInterfaces = getUSMModule(session, configuration, "interfaces");
            if (oxInterfaces != null && oxInterfaces.optBoolean(_usmInterface)) {
                return;
            }
        }
        _oxAjaxAccess.logout(session);
        throw new USMAccessDeniedException(USMSessionManagementErrorCodes.USM_ACCESS_DISABLED, "USM access disabled for user " + session.getUserIdentifier() + '/' + session.getUser() + " in context " + session.getContextId());
    }

    private JSONObject getUSMModule(final SessionImpl session, final JSONObject configuration, final String module) throws AuthenticationFailedException {
        if (configuration != null) {
            return JSONToolkit.getJSONObject(configuration, "modules", module);
        }
        try {
            return _oxAjaxAccess.getConfiguration(session, "modules", module);
        } catch (final OXCommunicationException e) {
            return null;
        }
    }

    public OXJSONAccess getOXAjaxAccess() {
        return _oxAjaxAccess;
    }

    public Log getJournal() {
        return LOG;
    }

    public ContentTypeManager getContentTypeManager() {
        return _contentTypeManager;
    }

    public FolderContentType getFolderContentType() {
        return _folderContentType;
    }

    public FolderIDMappingStorage getMappingService() {
        return _mappingService;
    }

    public ContextUUIDMappingStorage getUUIDService() {
        return _uuidService;
    }

    public int getMaxSyncStatesInDB() {
        return _maxSyncStatesInDB;
    }

    // --- Interface OXEventListener ---

    @Override
    public void folderChanged(final int contextId, final int userId, final String folderId, long timestamp) {
        final SessionID[] sessions = _sessionStorage.getActiveSessions(new UserID(contextId, userId));
        if (sessions != null) {
            if (timestamp == 0) {
                timestamp = System.currentTimeMillis();
            }
            for (final SessionID session : sessions) {
                final SessionImpl s = _sessionStorage.getSession(session);
                if (s != null) {
                    s.foldersChanged(timestamp);
                }
            }
        }
    }

    @Override
    public void folderContentChanged(final int contextId, final int userId, final String folderId, long timestamp) {
        final SessionID[] sessions = _sessionStorage.getActiveSessions(new UserID(contextId, userId));
        if (sessions != null) {
            if (timestamp == 0) {
                timestamp = System.currentTimeMillis();
            }
            for (final SessionID session : sessions) {
                final SessionImpl s = _sessionStorage.getSession(session);
                if (s != null) {
                    if (folderId == null) {
                        s.folderContentChanged(timestamp);
                    } else {
                        s.folderContentChanged(timestamp, folderId);
                    }
                }
            }
        }
    }

    @Override
    public void defaultFoldersChanged(final int contextId, final int userId) {
        final SessionID[] sessions = _sessionStorage.getActiveSessions(new UserID(contextId, userId));
        if (sessions != null) {
            for (final SessionID session : sessions) {
                final SessionImpl s = _sessionStorage.getSession(session);
                if (s != null) {
                    s.markFolderHierarchyDirty();
                    try {
                        s.getSessionInitializer().updateDefaultFolders(s, null);
                    } catch (final USMException e) {
                        LOG.error("Error while updating the default mail folders: " + e.getErrorMessage(), e);
                    }
                }
            }
        }
    }

    public boolean isEmailPullEnabled() {
        return _waitForChangesEmailMinPullDelay >= 0;
    }

    public long computePullTime(final long start, final long end) {
        final long result = start + _waitForChangesEmailMinPullDelay;
        // If disabled or minimum wait time greater than interval, return -1 to indicate no email pull
        if (_waitForChangesEmailMinPullDelay < 0 || result > end) {
            return -1L;
        }
        // Compute pulltime as percentage of interval or absolute delay from start
        final long pullTime = (_waitForChangesEmailPullDelay < 0) ? (start - (end - start) * _waitForChangesEmailPullDelay / 100L) : (start + _waitForChangesEmailPullDelay);
        // return at least minimum wait time, at most normal end time
        return Math.min(Math.max(result, pullTime), end);
    }

    // internal methods

    private int readEmailMinPullDelayProperty() {
        try {
            return _configurationManager.getProperty(ConfigurationProperties.SYNC_EMAIL_PULL_MIN_DELAY_PROPERTY, ConfigurationProperties.SYNC_EMAIL_PULL_MIN_DELAY_DEFAULT, true);
        } catch (final USMInvalidConfigurationException e) {
            throw new USMStartupException(USMSessionManagementErrorCodes.CONFIGURATION_ERROR_4, "Invalid configuration value for " + ConfigurationProperties.SYNC_EMAIL_PULL_MIN_DELAY_PROPERTY, e);
        }
    }

    private int readEmailPullDelayProperty() {
        try {
            final String pullValue = _configurationManager.getProperty(ConfigurationProperties.SYNC_EMAIL_PULL_DELAY_PROPERTY, ConfigurationProperties.SYNC_EMAIL_PULL_DELAY_DEFAULT, true);
            if (pullValue.endsWith("%")) {
                return -Integer.parseInt(pullValue.substring(0, pullValue.length() - 1));
            }
            return Integer.parseInt(pullValue);
        } catch (final NumberFormatException e) {
            throw new USMStartupException(USMSessionManagementErrorCodes.CONFIGURATION_ERROR_2, "Invalid configuration value for " + ConfigurationProperties.SYNC_EMAIL_PULL_DELAY_PROPERTY, e);
        } catch (final USMInvalidConfigurationException e) {
            throw new USMStartupException(USMSessionManagementErrorCodes.CONFIGURATION_ERROR_3, "Invalid configuration value for " + ConfigurationProperties.SYNC_EMAIL_PULL_DELAY_PROPERTY, e);
        }
    }

    public static DataObjectSet insertStoredUUID(final DataObject o, final DataObjectSet oldData, DataObjectSet newestCachedData) throws StorageAccessException, USMStorageException {
        final boolean useModifiedUUID = o.getSession().getUserIdentifier() != o.getParentFolderOwnerID();
        if (oldData != null) {
            final DataObject oldObject = oldData.get(o.getID());
            if (oldObject != null) {
                o.setUUID(useModifiedUUID ? UUIDToolkit.getModifedUUID(oldObject.getUUID(), o.getParentFolderOwnerID()) : oldObject.getUUID());
                return newestCachedData;
            }
        }
        final int contentTypeCode = o.getContentType().getCode();
        switch (contentTypeCode) {
            case DefaultContentTypes.FOLDER_CODE:
                if (newestCachedData == null) {
                    newestCachedData = new DataObjectSet(o.getSession().getCachedFolders());
                }
                final DataObject cachedFolder = newestCachedData.get(o.getID());
                if (cachedFolder != null) {
                    if (oldData != null) {
                        final DataObject oldFolder = oldData.get(cachedFolder.getUUID());
                        if (oldFolder != null) {
                            if (!oldFolder.getID().equals(cachedFolder.getID()) && newestCachedData.get(oldFolder.getID()) != null) {
                                // we have a swap situation
                                final DataObject swappedFolder = newestCachedData.get(oldFolder.getID());
                                if (swappedFolder != null && swappedFolder.getUUID() != null) {
                                    o.setUUID(swappedFolder.getUUID());
                                    break;
                                }
                            }
                        }
                    }
                    o.setUUID(cachedFolder.getUUID());
                }
                break;
            case DefaultContentTypes.MAIL_CODE:
                if (newestCachedData == null) {
                    newestCachedData = new DataObjectSet(o.getSession().getCachedFolderElements(o.getParentFolderID(), o.getContentType()));
                }
                final DataObject cachedMail = newestCachedData.get(o.getID());
                if (cachedMail != null) {
                    o.setUUID(useModifiedUUID ? UUIDToolkit.getModifedUUID(cachedMail.getUUID(), o.getParentFolderOwnerID()) : cachedMail.getUUID());
                }
                break;
            default:
                if (newestCachedData == null) {
                    newestCachedData = new DataObjectSet(o.getSession().getCachedFolderElements(o.getParentFolderID(), o.getContentType()));
                }
                final DataObject cachedObject = newestCachedData.get(o.getID());
                UUID uuid = (cachedObject == null) ? null : cachedObject.getUUID();
                if (uuid == null) {
                    uuid = o.getSession().getUUID(o.getContentType(), o.getID());
                }
                o.setUUID(useModifiedUUID ? UUIDToolkit.getModifedUUID(uuid, o.getParentFolderOwnerID()) : uuid);
                break;
        }
        return newestCachedData;
    }

    public synchronized void setManagementService(final ManagementService managementService) {
        if (_managementService != null) {
            if (_jmxSessionObjectName != null) {
                try {
                    _managementService.unregisterMBean(_jmxSessionObjectName);
                } catch (final OXException e) {
                    LOG.error(e.getMessage(), e);
                } finally {
                    _jmxSessionObjectName = null;
                    _managementService = null;
                }
            }
        }
        _managementService = managementService;
        if (managementService != null) {
            try {
                _jmxSessionObjectName = getObjectName(USMSessionInformation.class.getName(), USMSessionInformationMBean.SESSION_DOMAIN);
                _managementService.registerMBean(_jmxSessionObjectName, new USMSessionInformation(this, _usmCacheInformation));
            } catch (final MalformedObjectNameException e) {
                LOG.error(e.getMessage(), e);
            } catch (final OXException e) {
                LOG.error(e.getMessage(), e);
            } catch (final NotCompliantMBeanException e) {
                LOG.error(e.getMessage(), e);
            }
        }
    }

    /**
     * Creates an appropriate instance of {@link ObjectName} from specified class name and domain name.
     * 
     * @param className The class name to use as object name
     * @param domain The domain name
     * @return An appropriate instance of {@link ObjectName}
     * @throws MalformedObjectNameException If instantiation of {@link ObjectName} fails
     */
    private static ObjectName getObjectName(final String className, final String domain) throws MalformedObjectNameException {
        final int pos = className.lastIndexOf('.');
        return new ObjectName(domain, "name", pos == -1 ? className : className.substring(pos + 1));
    }

    public SessionStorage getSessionStorage() {
        return _sessionStorage;
    }

    public void setWaitForChangesEmailPullDelay(final int waitForChangesEmailPullDelay) {
        _waitForChangesEmailPullDelay = waitForChangesEmailPullDelay;
    }

    public int getWaitForChangesEmailPullDelay() {
        return _waitForChangesEmailPullDelay;
    }

    public void setWaitForChangesEmailMinPullDelay(final int waitForChangesEmailMinPullDelay) {
        _waitForChangesEmailMinPullDelay = waitForChangesEmailMinPullDelay;
    }

    public int getWaitForChangesEmailMinPullDelay() {
        return _waitForChangesEmailMinPullDelay;
    }

    public void setMaxSyncStatesInDB(final int states) {
        _maxSyncStatesInDB = states;
    }

    public SlowContentSyncer getSlowSyncer() {
        return new SlowContentSyncer(this, _syncRetryCount);
    }

    public IncrementalContentSyncer getIncrementalSyncer() {
        return new IncrementalContentSyncer(this, _syncRetryCount);
    }

    private void deactivateTimers() {
        if (_sessionCacheCleanupTask != null) {
            _sessionCacheCleanupTask.cancel();
            _sessionCacheCleanupTask = null;
        }
        if (_sessionCacheCleanupLogInfoTask != null) {
            _sessionCacheCleanupLogInfoTask.cancel();
            _sessionCacheCleanupLogInfoTask = null;
        }
    }

    public void setTimerService(final TimerService timerService) {
        deactivateTimers();
        _timerService = timerService;
        if (_timerService != null && _configurationManager != null) { // Only start new tasks if bundle is activated and TimerService is available

            // The ClusterTimerService is preferred, thus only initialise and set it to local timer service
            // if it wasn't previously initialised (use it as a fall-back in case the ClusterTimerService is
            // not available
            if (_cleanupTask == null) {
                _cleanupTask = _timerService.scheduleAtFixedRate(new USMSessionCleanupTask(this), USMSessionCleanupTask.INITIAL_DELAY, USMSessionCleanupTask.EXECUTION_INTERVAL);
            }

            long sessionCacheCheckInterval = ConfigurationProperties.USM_SESSION_CACHE_CHECK_INTERVAL_DEFAULT;
            long sessionCacheCheckIntervalLogInfo = ConfigurationProperties.USM_SESSION_CACHE_CHECK_INTERVAL_LOG_INFO_DEFAULT;

            try {
                sessionCacheCheckInterval = _configurationManager.getProperty(ConfigurationProperties.USM_SESSION_CACHE_CHECK_INTERVAL_PROPERTY, ConfigurationProperties.USM_SESSION_CACHE_CHECK_INTERVAL_DEFAULT, true);

                sessionCacheCheckIntervalLogInfo = _configurationManager.getProperty(ConfigurationProperties.USM_SESSION_CACHE_CHECK_INTERVAL_LOG_INFO_PROPERTY, ConfigurationProperties.USM_SESSION_CACHE_CHECK_INTERVAL_LOG_INFO_DEFAULT, true);

            } catch (final USMInvalidConfigurationException e) {
                LOG.error(e.getStackTrace(), e);
            }
            _sessionCacheCleanupTask = _timerService.scheduleAtFixedRate(new USMSessionCacheCleanupTask(this), sessionCacheCheckInterval, sessionCacheCheckInterval);

            _sessionCacheCleanupLogInfoTask = _timerService.scheduleAtFixedRate(new USMSessionCacheCleanupLogInfoTask(this), sessionCacheCheckIntervalLogInfo, sessionCacheCheckIntervalLogInfo);
        }
    }

    public long getSessionStorageTimeLimit() {
        return _sessionStorageTimeLimit;
    }

    public OXDataCache getOXDataCache() {
        return _dataCache;
    }

    public Folder getSpecialFolder(final SessionImpl session, final String id) {
        if (!DefaultContentTypes.GROUPS_ID.equals(id) && !DefaultContentTypes.RESOURCES_ID.equals(id)) {
            return null;
        }
        final ContentType type = getContentTypeManager().getContentType(id);
        if (type == null) {
            return null;
        }
        final Folder folder = new FolderImpl(session, getFolderContentType());
        folder.setID(id);
        folder.setTimestamp(System.currentTimeMillis());
        folder.setElementsContentType(type);
        return folder;
    }

    public List<String> getSpecialFolders() {
        return SPECIAL_FOLDERS_VIEW;
    }

    public int getSessionCacheTimeout() {
        return _sessionCacheTimeout;
    }

    public void setSessionCacheTimeout(final int timeout) {
        _sessionCacheTimeout = timeout;
    }

    public int getSyncStateCacheTimeout() {
        return _syncStateCacheTimeout;
    }

    public void setSyncStateCacheTimeout(final int timeout) {
        _syncStateCacheTimeout = timeout;
    }

    public int getMaxObjectsPerFolder() {
        return _maxObjectsPerFolder;
    }

    public int getAttachmentCountLimitPerObject() {
        return _attachmentCountLimitPerObject;
    }

    public long getAttachmentSizeLimitPerObject() {
        return _attachmentSizeLimitPerObject;
    }

    public void setMaxObjectsPerFolder(final int maxObjectsPerFolder) {
        _maxObjectsPerFolder = maxObjectsPerFolder;
    }

    public void updateCacheRemovals(final int removedSessions, final int jvmRemovals, final int removedSyncStates) {
        _usmCacheInformation.sessionsRemovedDueToInactivity(removedSessions);
        _usmCacheInformation.sessionsRemovedByJVM(jvmRemovals);
        _usmCacheInformation.syncStatesRemovedDueToInactivity(removedSyncStates);
    }

    public USMCacheInformation getUSMCacheInformation() {
        return _usmCacheInformation;
    }

    public void syncStateSavedToDatabase() {
        _usmCacheInformation.syncStateSavedToDatabase();
    }

    public void syncStatesLoadedFromDatabase(final int count) {
        _usmCacheInformation.syncStatesLoadedFromDatabase(count);
    }

    public MultiThreadedHttpConnectionManager getHttpConnectionManager() {
        return _oxAjaxAccess.getHttpConnectionManager();
    }

    public void setSyncStateCacheProvider(final SyncStateCacheProvider provider) {
        _syncStateCacheProvider = provider;
    }

    public SyncStateCacheProvider getSyncStateCacheProvider() {
        final SyncStateCacheProvider provider = _syncStateCacheProvider;
        return (provider == null) ? _defaultCacheProvider : provider;
    }

    public PersistentSyncStateStorage getSyncStateStorage() {
        return syncStateStorage;
    }

    public PersistentSessionDataStorage getSessionDataStorage() {
        return sessionDataStorage;
    }

    @Override
    public void objectDeleted(final int contextID, final int objectID, final int contentType) {
        // nothing to do with new UUID storage mechanism
    }

    @Override
    public void enableWaitingForChanges(final String protocol) {
        if (!_waitEnabledProtocols.contains(protocol)) {
            _waitEnabledProtocols.add(protocol);
        }
    }

    @Override
    public void disableWaitForChanges(final String protocol) {
        _waitEnabledProtocols.remove(protocol);
        for (final SessionImpl session : _sessionStorage.getSessionList()) {
            if (session.getProtocol().equals(protocol)) {
                session.notifyOnWaitForChanges();
            }
        }
    }

    public boolean isWaitForChangesEnabled(final String protocol) {
        return _waitEnabledProtocols.contains(protocol);
    }

    /**
     * Stops and deactivates the {@link USMSessionCleanupTask}
     */
    private void deactivateUSMSessionCleanupTask() {
        if (_cleanupTask != null) {
            _cleanupTask.cancel();
            _cleanupTask = null;
        }
    }

    /**
     * Sets the {@link ClusterTimerService} and restarts the {@link USMSessionCleanupTask} if the {@link ClusterTimerService} is present
     * 
     * @param clusterTimerService The {@link ClusterTimerService} to set; <code>null</code> to deactivate
     */
    public void setClusterTimerService(final ClusterTimerService clusterTimerService) {
        this.clusterTimerService = clusterTimerService;
        deactivateUSMSessionCleanupTask();
        if (clusterTimerService != null) {
            _cleanupTask = clusterTimerService.scheduleAtFixedRate("USMSessionCleanupTask", new USMSessionCleanupTask(this), USMSessionCleanupTask.INITIAL_DELAY, USMSessionCleanupTask.EXECUTION_INTERVAL, TimeUnit.MILLISECONDS);
        }
    }

    /**
     * Returns the {@link ClusterTimerService}
     * 
     * @return the {@link ClusterTimerService}
     */
    public ClusterTimerService getClusterTimerService() {
        return clusterTimerService;
    }
}
