/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.session.impl;

/**
 * {@link CustomPropertyLimit}
 * 
 * @author <a href="mailto:afe@microdoc.de">Alexander Feess</a>
 */
public class CustomPropertyLimit implements Comparable<CustomPropertyLimit> {

    private final long _storageLimit;

    private final String _key;

    /**
     * Initializes a new {@link CustomPropertyLimit} that will store the timeout for a given key
     * 
     * @param key CustomProperty key
     */
    public CustomPropertyLimit(String key, long timeout) {
        _key = key;
        if (timeout <= 0L)
            throw new IllegalArgumentException("Timeout must be > 0");
        long now = System.currentTimeMillis();
        _storageLimit = now + timeout;
        if (_storageLimit < now || _storageLimit < timeout)
            throw new IllegalArgumentException("Timeout too high");
    }

    /**
     * Initializes a new {@link CustomPropertyLimit} that will match any CustomPropertyLimit with the same key (useful for searching without
     * knowing the timeout)
     * 
     * @param key CustomProperty key
     */
    public CustomPropertyLimit(String key) {
        _key = key;
        _storageLimit = 0L;
    }

    /**
     * @return CustomProperty key referenced by this {@link CustomPropertyLimit}
     */
    public String getKey() {
        return _key;
    }

    /**
     * @return timestamp after which this {@link CustomPropertyLimit} should be removed from the USM-Session.
     */
    public long getStorageLimit() {
        return _storageLimit;
    }

    @Override
    public int compareTo(CustomPropertyLimit o) {
        if (_storageLimit < o._storageLimit)
            return -1;
        if (_storageLimit > o._storageLimit)
            return 1;
        return 0;
    }

    @Override
    public int hashCode() {
        // Only key is used for hashCode() so that object is stored/retrieved correctly with specific / unspecified timeout
        return _key.hashCode();
    }

    /**
     * Returns true if obj is a CustomPropertyLimit the keys match, and the storageLimit either matches or one of the two storageLimits is 0
     * (which matches any limit).
     */
    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        CustomPropertyLimit other = (CustomPropertyLimit) obj;
        if (_key == null) {
            if (other._key != null)
                return false;
        } else if (!_key.equals(other._key))
            return false;
        // Special handling of storageLimit: Allow 0 to be a wild card that matches all limits
        if (_storageLimit == 0L || other._storageLimit == 0L)
            return true;
        return _storageLimit == other._storageLimit;
    }

    @Override
    public String toString() {
        if (_storageLimit == 0L)
            return "Limit(" + _key + ')';
        return "Limit(" + _key + ':' + _storageLimit + ')';
    }
}
