/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.session.dataobject;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.UUID;
import com.openexchange.usm.api.session.DataObject;

public class DataObjectSet implements Collection<DataObject> {

	private final static String MISSING_ID_PREFIX = "DATAOBJECT_WITHOUT_ID_";

	private final Map<String, DataObject> _idMap;
	private final Map<UUID, DataObject> _uuidMap;
	private int _noIDCounter = 0;

	public DataObjectSet() {
		_idMap = new HashMap<String, DataObject>();
		_uuidMap = new HashMap<UUID, DataObject>();
	}

	public DataObjectSet(int size) {
		_idMap = new HashMap<String, DataObject>(size);
		_uuidMap = new HashMap<UUID, DataObject>(size);
	}

	public DataObjectSet(DataObject[] elements) {
		this(elements == null ? 100 : elements.length);
		if (elements != null)
			addAll(elements);
	}

	public DataObjectSet(Collection<? extends DataObject> elements) {
		this(elements == null ? 100 : elements.size());
		if (elements != null)
			addAll(elements);
	}

	@Override
    public boolean add(DataObject e) {
		String id = e.getID();
		if (id == null)
			id = generateNoIDKey(_noIDCounter++);
		_idMap.put(id, e);
		UUID uuid = e.getUUID();
		if (uuid != null)
			_uuidMap.put(uuid, e);
		return true;
	}

	@Override
    public boolean addAll(Collection<? extends DataObject> c) {
		if (c == null || c.size() == 0)
			return false;
		for (DataObject o : c)
			add(o);
		return true;
	}

	public boolean addAll(DataObject[] c) {
		if (c == null || c.length == 0)
			return false;
		for (DataObject o : c)
			add(o);
		return true;
	}

	@Override
    public void clear() {
		_idMap.clear();
		_uuidMap.clear();
	}

	public boolean contains(String id) {
		return _idMap.containsKey(id);
	}

	public boolean contains(UUID uuid) {
		return _uuidMap.containsKey(uuid);
	}

	public DataObject get(String id) {
		return _idMap.get(id);
	}

	public DataObject get(UUID uuid) {
		return _uuidMap.get(uuid);
	}

	@Override
    public boolean isEmpty() {
		return _idMap.isEmpty();
	}

	@Override
    public Iterator<DataObject> iterator() {
		return _idMap.values().iterator();
	}

	public DataObject remove(String id) {
		DataObject o = _idMap.remove(id);
		if (o != null) {
			UUID uuid = o.getUUID();
			if (uuid != null)
				_uuidMap.remove(uuid);
		}
		return o;
	}

	public DataObject remove(UUID uuid) {
		DataObject o = _uuidMap.remove(uuid);
		if (o != null) {
			String id = o.getID();
			if (id != null)
				_idMap.remove(id);
		}
		return o;
	}

	@Override
    public int size() {
		return _idMap.size();
	}

	@Override
    public DataObject[] toArray() {
		Collection<DataObject> values = _idMap.values();
		return values.toArray(new DataObject[values.size()]);
	}

	public List<DataObject> toList() {
		List<DataObject> list = new ArrayList<DataObject>(_idMap.size());
		for (DataObject o : this)
			list.add(o);
		return list;
	}

	/**
	 * This method only checks if a DataObject with the same id is stored in this DataObjectSet.
	 * It does not check for equality of the DataObjects themselves.
	 * 
	 */
	@Override
    public boolean contains(Object o) {
		if (!(o instanceof DataObject))
			return false;
		DataObject ob = get(((DataObject) o).getID());
		if (ob != null)
			return true;
		if (_noIDCounter > 0) {
			for (int i = 0; i < _noIDCounter; i++) {
				ob = get(generateNoIDKey(i));
				if (o == ob)
					return true;
			}
		}
		return false;
	}

	@Override
    public boolean containsAll(Collection<?> c) {
		for (Object e : c) {
			if (!contains(e))
				return false;
		}
		return true;
	}

	@Override
    public boolean remove(Object o) {
		if (!(o instanceof DataObject))
			return false;
		DataObject ob = remove(((DataObject) o).getID());
		DataObject ob2 = null;

        if (ob == null) {
            for (Entry<String, DataObject> entry : _idMap.entrySet()) {
                if (entry.getValue() == o) {
                    ob = _idMap.remove(entry.getKey());
                    break;
                }
            }
            for (Entry<UUID, DataObject> entry : _uuidMap.entrySet()) {
                if (entry.getValue() == o) {
                    ob2 = _uuidMap.remove(entry.getKey());
                    break;
                }
            }
        } else {
            ob2 = ob;
        }


		//		if (ob == null && _noIDCounter > 0) {
		//			for (int i = 0; i < _noIDCounter; i++) {
		//				String key = generateNoIDKey(i);
		//				DataObject ob2 = get(key);
		//				if (ob2 == o) {
		//					_idMap.remove(key);
		//					break;
		//				}
		//			}
		//		}

		return ob != null && ob2 != null;
	}

	@Override
    public boolean removeAll(Collection<?> c) {
		boolean result = false;
		for (Object o : c) {
			result |= remove(o);
		}
		return result;
	}

	@Override
    public boolean retainAll(Collection<?> c) {
		throw new UnsupportedOperationException();
	}

	@Override
    public <T> T[] toArray(T[] a) {
		throw new UnsupportedOperationException();
	}

	@Override
	public String toString() {
		return _idMap.toString();
	}

	private String generateNoIDKey(int n) {
		return MISSING_ID_PREFIX + n;
	}

	/**
	 * Checks whether this DataObjectSet contains exactly the given DataObjects, i.e. same number, same IDs of
	 * DataObjects, every ID must be present and the local DataObject with the given ID must be equal to the given
	 * DataObject (i.e. same ChangeState,all fields equal, timestamps *do not* need to be equal) and the contained
	 * ProtocolInformations and UUID must also be equal.
	 * 
	 * @param objects
	 * @return
	 */
	public boolean isEqualTo(DataObject[] objects) {
        if(objects == null)
            return isEmpty();
        if (size() != objects.length)
            return false;
        for (DataObject o : objects) {
            if (o == null)
                return false;
            DataObject local = get(o.getID());
            if (!o.equalsWithLastModified(local) || !DataObjectUtil.isEqual(o.getProtocolInformation(), local.getProtocolInformation())
                    || !DataObjectUtil.isEqual(o.getUUID(), local.getUUID()))
                return false;
        }
        return true;
    }

    public Set<String> getIDs() {
        return _idMap.keySet();
    }
    
    public void swapUUIDs(Map<UUID, UUID> swapUUIDsMap) {
        Set<UUID> processedSwapUuids = new HashSet<UUID>();
        for (UUID uuid1: swapUUIDsMap.keySet()) {
            UUID uuid2 = swapUUIDsMap.get(uuid1);
            if (!uuid1.equals(uuid2) && uuid2 != null && !processedSwapUuids.contains(uuid1) && contains(uuid1) && contains(uuid2)) {
                DataObject obj1 = remove(uuid1);
                DataObject obj2 = remove(uuid2);
                DataObject copy1 = obj1.createCopy(false);
                DataObject copy2 = obj2.createCopy(false);
                copy1.setUUID(uuid2);
                copy2.setUUID(uuid1);
                add(copy1);
                add(copy2);
                processedSwapUuids.add(uuid2);
            }
        }
    }
}
