/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.connector.sync;

import java.util.Comparator;
import com.openexchange.usm.api.contenttypes.common.ContentType;
import com.openexchange.usm.api.contenttypes.common.ContentTypeField;
import com.openexchange.usm.api.datatypes.DataTypes;
import com.openexchange.usm.api.session.DataObject;

public abstract class DataObjectSorter implements Comparator<DataObject> {

    protected DataObjectSorter() {
        super();
    }

    protected final int getFieldIndex(ContentType c, String field) {
        ContentTypeField[] fields = c.getFields();
        for (int i = 0; i < fields.length; i++) {
            if (fields[i].getFieldName().equals(field))
                return i;
        }
        throw new IllegalStateException("ContentType " + c.getID() + " has no field named " + field);
    }

    protected int compareTimestamp(DataObject o1, DataObject o2, int index) {
        long l1 = getTimestamp(o1, index);
        long l2 = getTimestamp(o2, index);
        return (l1 > l2) ? -1 : (l1 < l2) ? 1 : 0;
    }

    protected int compareString(DataObject o1, DataObject o2, int index) {
        Object v1 = o1.getFieldContent(index);
        Object v2 = o2.getFieldContent(index);
        if (v1 == null)
            return (v2 == null) ? 0 : -1;
        return (v2 == null) ? 1 : v1.toString().compareTo(v2.toString());
    }

    protected long getTimestamp(DataObject o1, int index) {
        Object o = o1.getFieldContent(index);
        return (o instanceof Number) ? ((Number) o).longValue() : 0L;
    }

    protected int compareBoolean(DataObject o1, DataObject o2, int index) {
        Object v1 = o1.getFieldContent(index);
        Object v2 = o2.getFieldContent(index);
        if (v1 == null)
            return (v2 == null) ? 0 : -1;
        if (v2 == null)
            return 1;
        Boolean b1 = DataTypes.BOOLEAN.checkValue(v1);
        Boolean b2 = DataTypes.BOOLEAN.checkValue(v2);
        if (b1.equals(b2))
            return 0;
        if (Boolean.TRUE.equals(b1))
            return 1;
        return -1;
    }
}
