/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */
package com.openexchange.usm.api.session;

import com.openexchange.usm.api.contenttypes.common.ContentType;
import com.openexchange.usm.api.contenttypes.folder.FolderContentType;

/**
 * Extension to the DataObject for the ContentType "folder" that provides
 * easy access methods for special fields.
 *
 *
 * @author afe
 *
 */
public interface Folder extends DataObject {

	/*
	 * (non-Javadoc)
	 * @see com.openexchange.usm.api.session.DataObject#getContentType()
	 */
    @Override
    FolderContentType getContentType();

	/**
	 * This method will retrieve the ID of the ContentType, or if none is set yet,
	 * it will return the value of the appropriate field that contains the id
	 * (i.e. normally "module" for OX).
	 *
	 * @return ID of the ContentType of the elements in this Folder
	 */
	String getElementsContentTypeID();

	/**
	 *
	 * @return ContentType of the elements in this Folder
	 */
	ContentType getElementsContentType();

	/**
	 * This method is used by the sync system to set the ContentType of elements of the Folder.
	 * It can be called only once after creation, later calls throw an IllegalStateException.
	 *
	 *
	 * @param type
	 * @throws IllegalStateException if the ContentType has already been set
	 */
	void setElementsContentType(ContentType type);

	/**
	 * @return original timestamp for last content synchronization
	 */
	long getOriginalContentSyncTimestamp();

	/**
	 * @return currently set timestamp for current content synchronization
	 */
	long getContentSyncTimestamp();

	/**
	 * Sets the timestamp for the current content synchronization. All elements
	 * that are changed will have their timestamps set to this value, too.
	 *
	 * @param timestamp milliseconds since midnight, January 1, 1970 UTC.
	 */
	void setContentSyncTimestamp(long timestamp);

	/**
	 * Create an exact copy of the current state of this Folder.
	 *
	 * @param linkUUIDs if true, modifications to a UUID will be performed in both the original and the copy
	 * @return
	 */
	@Override
    Folder createCopy(boolean linkUUIDs);

	/**
	 *
	 * @return true if Folder has sub-folders or if it can not be determined if it has, false if it has no sub folders. This method always operates on the current content of the Folder (original is ignored)
	 */
	boolean hasSubFolders();

	int getOwnerID();
}
