/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */
package com.openexchange.usm.api.datatypes;

import java.io.Serializable;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.usm.api.session.Session;

/**
 * Special DataType that implements an Array of another DataType
 * 
 * @author afe
 *
 */
public class ArrayDataType implements DataType<Object[]> {
	private static final Object[] EMPTY_ARRAY = new Object[0];

	protected final DataType<?> _elementType;

	public ArrayDataType(DataType<?> elementType) {
		_elementType = elementType;
	}

	@Override
    public void addToJSONArray(Session session, JSONArray parent, Object value) throws JSONException {
		parent.put(generateJSONArray(session, (Object[]) value));
	}

	@Override
    public Object[] checkValue(Object o) {
		if (o == null)
			return EMPTY_ARRAY;
		if (o instanceof Object[]) {
			Object[] array = (Object[]) o;
			for (int i = 0; i < array.length; i++) {
				array[i] = _elementType.checkValue(array[i]);
			}
			return array;
		}
		try {
			return new Object[] { _elementType.checkValue(o) };
		} catch (IllegalArgumentException e) {
			throw new IllegalArgumentException(o.getClass().getName() + " is neither " + getName()
					+ " or its element type");
		}
	}

	@Override
    public Object[] createCopy(Object value) {
		Object[] source = (Object[]) value;
		if (source.length == 0)
			return source;
		Object[] result = new Object[source.length];
		for (int i = 0; i < result.length; i++) {
			result[i] = _elementType.createCopy(source[i]);
		}
		return result;
	}

    @Override
    public Serializable serialize(Object value) {
        Object[] source = (Object[]) value;
        if (source.length == 0)
            return source;
        Object[] result = new Object[source.length];
        for (int i = 0; i < result.length; i++) {
            result[i] = _elementType.serialize(source[i]);
        }
        return result;
    }

    @Override
    public Object[] deserialize(Serializable value) {
        Object[] source = (Object[]) value;
        if (source.length == 0)
            return source;
        Object[] result = new Object[source.length];
        for (int i = 0; i < result.length; i++) {
            result[i] = _elementType.deserialize((Serializable) source[i]);
        }
        return result;
    }

	@Override
    public Object[] createNewObject() {
		return EMPTY_ARRAY;
	}

	@Override
    public Object[] extractFromJSONArray(Session session, JSONArray array, int index) throws JSONException {
		if (array.isNull(index))
			return createNewObject();
		return extractFromJSONArray(session, array.getJSONArray(index));
	}

	@Override
    public Object[] extractFromJSONObject(Session session, JSONObject jsonObject, String field) throws JSONException {
		if (jsonObject.isNull(field))
			return createNewObject();
		return extractFromJSONArray(session, jsonObject.getJSONArray(field));
	}

	@Override
    public String getName() {
		return "Array(" + _elementType.getName() + ')';
	}

	@Override
    public void storeInJSONObject(Session session, JSONObject parent, String field, Object value) throws JSONException {
		parent.put(field, generateJSONArray(session, (Object[]) value));
	}

	private JSONArray generateJSONArray(Session session, Object[] value) throws JSONException {
		JSONArray array = new JSONArray();
		for (Object o : value)
			_elementType.addToJSONArray(session, array, o);
		return array;
	}

	private Object[] extractFromJSONArray(Session session, JSONArray array) throws JSONException {
		Object[] result = new Object[array.length()];
		for (int i = 0; i < result.length; i++)
			result[i] = _elementType.extractFromJSONArray(session, array, i);
		return result;
	}

	@Override
    public boolean isEqual(Object v1, Object v2) {
		Object[] value1 = (Object[]) v1;
		Object[] value2 = (Object[]) v2;
		if (value1.length != value2.length)
			return false;
		for (int i = 0; i < value1.length; i++) {
			if (!_elementType.isEqual(value1[i], value2[i]))
				return false;
		}
		return true;
	}

	@Override
    public boolean isDefaultValue(Object object) {
		Object[] value = (Object[]) object;
		return value.length == 0;
	}

	public DataType<?> getElementType() {
		return _elementType;
	}
}
