/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.api.database;

import java.sql.Connection;
import java.util.List;
import com.openexchange.usm.api.exceptions.USMStorageException;

/**
 * Service that provides access to OX database or alternative database
 * 
 * @author afe
 */
public interface DatabaseAccess {

    /**
     * Provides an EncapsulatedConnection to store custom data. Every Connection that is retrieved from the DatabaseAccess has to be closed
     * later. The DatabaseAccess may returns an <code>EncapsulatedConnection</code> that contains most (if not all, depending on the JDK
     * version) methods of a <code>java.sql.Connection</code>. For the access to the database a context id has to be specified.
     * 
     * @param context
     * @return
     * @throws StorageAccessException if the Connection can not be retrieved
     */
    EncapsulatedConnection getWritable(int contextID) throws StorageAccessException;

    /**
     * As getWritable(), but returns a Connection that can be used only for read access.
     * 
     * @param contextID
     * @return
     * @throws StorageAccessException if the Connection can not be retrieved
     */
    EncapsulatedConnection getReadOnly(int contextID) throws StorageAccessException;

    /**
     * Generates a sequence in the database. Implementations can handle this differently depending on the database. (ex. create a table and
     * a procedure for retrieving ids from this table, create a sequence, etc.)
     * 
     * @param contextID
     * @param sequenceName
     * @throws USMStorageException if an error occurred while creating the sequence
     * @throws StorageAccessException if the Connection can not be retrieved
     */
    void generateSequence(int contextID, String sequenceName) throws USMStorageException, StorageAccessException;

    /**
     * Gets the next id from a sequence in the database.
     * 
     * @param contextID
     * @param sequenceName
     * @return
     * @throws USMStorageException if an error occurred when retrieving the next id or no next id is returned
     * @throws StorageAccessException if the Connection can not be retrieved
     */
    int getNextIdFromSequence(int contextID, String sequenceName) throws USMStorageException, StorageAccessException;

    /**
     * Returns the name of the function used for retrieving a String representation of a hexadecimal value in the used DB.
     * 
     * @return
     */
    String retrieveHexFunctionName();

    /**
     * Returns the name of the function used for retrieving a hexadecimal representation of a String value in the used DB.
     * 
     * @return
     */
    String retrieveUnHexFunctionName();

    /**
     * @return List of all currently available/used Context-IDs in the system
     * @throws StorageAccessException if the list could not be generated/retrieved
     */
    List<Integer> getAllContextIds() throws StorageAccessException;

    /**
     * @param contextId OX context-id
     * @return Array of all context-ids that are in the same schema as the given context-id
     * @throws StorageAccessException if the array could not be generated/retrieved
     */
    int[] getContextsInSameSchema(int contextId) throws StorageAccessException;
    
    
    boolean supportsOnDuplicateKeyUpdateStatement();
    
    boolean supportsInsertIgnoreStatement();
    
    /**
     * Releases a previously requested read/write SQL Connection so that it can be reused.
     * 
     * @param connection
     * @throws StorageAccessException if an error occurred while releasing the Connection
     */
    void backWritable(int contextID, Connection connection) throws StorageAccessException;

    /**
     * Releases a previously requested read-only SQL Connection so that it can be reused.
     * 
     * @param connection
     * @throws StorageAccessException if an error occurred while releasing the Connection
     */
    void backReadOnly(int contextID, Connection connection) throws StorageAccessException;
}
