/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.api.configuration;

/**
 * Global interface where all configuration keys for the various USM parts are stored, along with their description. An interface with
 * String constants is used to allow for extensibility without having to change the API while still providing a common place for all
 * configuration options.
 * 
 * @author afe
 */
public class ConfigurationProperties {

    // --- USM ---
    
    /**
     * Read USM access rights from configuration.
     */
    public static final String USM_INTERFACE_CONFIG_PROPERTY = "com.openexchange.usm.access_check_module";

    public static final String USM_INTERFACE_CONFIG_DEFAULT = "";

    /**
     * Determines the limit for inactivity of an USM session after which the persistent session data is removed (i.e. slow sync is required
     * afterwards). This value is given in days, the default value is 60 (approx. 2 months).
     */
    public static final String USM_SESSION_STORAGE_TIME_LIMIT_PROPERTY = "com.openexchange.usm.session_storage_time_limit";

    public static final int USM_SESSION_STORAGE_TIME_LIMIT_DEFAULT = 60; // 60 days

    /**
     * 
     */
    public static final String USM_SESSION_CACHE_CHECK_INTERVAL_PROPERTY = "com.openexchange.usm.session_cache_check_interval";

    public static final int USM_SESSION_CACHE_CHECK_INTERVAL_DEFAULT = 60000; // 1 minute

    /**
     * 
     */
    public static final String USM_SESSION_CACHE_CHECK_INTERVAL_LOG_INFO_PROPERTY = "com.openexchange.usm.session_cache_check_interval_log_info";

    public static final int USM_SESSION_CACHE_CHECK_INTERVAL_LOG_INFO_DEFAULT = 300000; // 5 minutes

    /**
     * Specifies a timeout (in seconds) after which USM sessions that are not accessed will be removed from memory. Removed sessions will be
     * automatically restored from DB on next access, but since server changes couldn't be traced while the session was not in memory, the
     * client has to perform an (incremental) sync for each folder (it will be notified that each folder changed regardless of any real
     * server changes). Default: 43200s = 12h
     */
    public static final String USM_CACHE_SESSION_TIMEOUT_PROPERTY = "com.openexchange.usm.cache.session.timeout";

    public static final int USM_CACHE_SESSION_TIMEOUT_DEFAULT = 43200;

    /**
     * Specifies a timeout (in seconds) after which cached SyncStates that are not accessed will be removed from memory. Removed SyncStates
     * will be automatically restored from DB on next access. Default: 3600s = 1h
     */
    public static final String USM_CACHE_SYNC_STATE_TIMEOUT_PROPERTY = "com.openexchange.usm.cache.syncstate.timeout";

    public static final int USM_CACHE_SYNC_STATE_TIMEOUT_DEFAULT = 3600;

    /**
     * Used by the OXAjaxAccess bundle to determine base url for all calls to the OX server
     */
    public static final String OX_ACCESS_URL_PROPERTY = "com.openexchange.usm.ox.url";

    /**
     * Default value for the base url of the OX server. TODO Replace the value for development with some useful default for client
     * installations (e.g. localhost)
     */
    public static final String OX_ACCESS_URL_DEFAULT = "http://localhost/ajax/";

    /**
     * Specifies the overall maximum number of connections the USM may use for requests to the OX server.
     */
    public static final String OX_ACCESS_MAX_CONNECTIONS_PROPERTY = "com.openexchange.usm.ox.max_connections";

    /**
     * Default value for the overall maximum number of connections the USM may use for requests to the OX server.
     */
    public static final int OX_ACCESS_MAX_CONNECTIONS_DEFAULT = 100;

    /**
     * Specifies the time (in milliseconds) after which idle connections to the OX server will be closed.
     */
    public static final String OX_ACCESS_CONNECTION_TIMEOUT_PROPERTY = "com.openexchange.usm.ox.idle_connection_timeout";

    /**
     * Default value for the time (in milliseconds) after which idle connections to the OX server will be closed.
     */
    public static final int OX_ACCESS_CONNECTION_TIMEOUT_DEFAULT = 60000;

    /**
     * Specifies if the HttpClient should perform stale checking of Connections before using them (increases security, lowers performance).
     */
    public static final String OX_ACCESS_PERFORM_STALE_CHECKING_PROPERTY = "com.openexchange.usm.ox.perform_stale_checking";

    /**
     * Default value for stale checking of Connections before using them (increases security, lowers performance).
     */
    public static final boolean OX_ACCESS_PERFORM_STALE_CHECKING_DEFAULT = true;
    
    /**
     * Specifies for Connections whether to stay signed in (avoids expiration of running sessions to the OX server).
     */
    public static final String OX_ACCESS_STAY_SIGNED_IN_PROPERTY = "com.openexchange.usm.ox.stay_signed_in";

    /**
     * Default value for Connections whether to stay signed in (avoids expiration of running sessions to the OX server).
     */
    public static final boolean OX_ACCESS_STAY_SIGNED_IN_DEFAULT = true;
    
    /**
     * If this property is set to true, usm sets the host header in the login request to the one received from client. To be able to do this, a system property sun.net.http.allowRestrictedHeaders is also set to true
     */
    public static final String OX_ACCESS_PASS_HOST_HEADER_FROM_CLIENT = "com.openexchange.usm.pass_client_host_header";

    /**
     * Default value for com.openexchange.usm.pass_client_host_header
     */
    public static final boolean OX_ACCESS_PASS_HOST_HEADER_FROM_CLIENT_DEFAULT = true;

    /**
     * This property defines a list of OX error codes that will trigger a re-login and re-execution of the request instead of reporting the error to the caller
     */
    public static final String OX_ACCESS_RELOGIN_ERROR_CODES_PROPERTY = "com.openexchange.usm.ox.relogin_error_codes";

    /**
     * Default value for com.openexchange.usm.ox.relogin_error_codes
     */
    public static final String OX_ACCESS_RELOGIN_ERROR_CODES_DEFAULT = "SES-0202,SES-0203,SES-0206";

    /**
     * This property defines a list of OX error codes that signify non-existing PIM items
     */
    public static final String OX_ACCESS_MISSING_OBJECT_ERROR_CODES_PROPERTY = "com.openexchange.usm.ox.missing_object_error_codes";

    /**
     * Default value for com.openexchange.usm.ox.missing_object_error_codes
     */
    /* As a suggestion from Bug 36808 this has been made configurable */
    public static final String OX_ACCESS_MISSING_OBJECT_ERROR_CODES_DEFAULT = "MSG-0032,OX-0001,CON-0125,TSK-0019";

    /**
     * This property defines how long a synchronization request will wait for exclusive access to the object (specific folder or folder hierarchy) that
     * it wants to synchronize.
     */
    public static final String SYNC_LOCK_TIMEOUT_PROPERTY = "com.openexchange.usm.session.sync.lock.timeout";

    /**
     * Default value for com.openexchange.usm.session.sync.lock.timeout
     */
    public static final int SYNC_LOCK_TIMEOUT_DEFAULT = 10000;
    
    /**
     * Used by the slow and incremental syncer to limit the number of retries if a concurrent modification is detected while performing a
     * synchronization with a client.
     */
    public static final String SYNC_CONCURRENT_MODIFICATION_MAX_RETRIES_PROPERTY = "com.openexchange.usm.session.sync.concurrent_modification_retries";

    /**
     * Default value for the max. number of retries of syncs that failed due to concurrent modifications on the server.
     */
    public static final int SYNC_CONCURRENT_MODIFICATION_MAX_RETRIES_DEFAULT = 3;

    /**
     * Defines the delay that is used when waiting for changes in email folders (Session.waitForChanges()) after which a pull on email
     * folders is performed. The delay can either be specified as a fixed value in milliseconds which defines the maximum time to wait (if
     * the waitForChanges waits for a smaller interval, the pull will be performed at the end of the wait) or as a percentage (integer
     * ending with a "%"), which indicates a portion of the given wait interval after which the pull will be performed.<br>
     * Note that the effective delay will be modified by the com.openexchange.usm.session.sync.email_pull_min_delay property which may
     * increase the delay or prohibit a pull completely.
     */
    public static final String SYNC_EMAIL_PULL_DELAY_PROPERTY = "com.openexchange.usm.session.sync.email_pull_delay";

    /**
     * Default value for the email pull delay, the pull will be performed after half the time given to waitForChanges has passed.
     */
    public static final String SYNC_EMAIL_PULL_DELAY_DEFAULT = "50%";

    /**
     * Defines the minimum delay that is required when waiting for changes in email folders (Session.waitForChanges()) after which a pull on
     * email folders will be performed. The value specified is in milliseconds. The effective delay will be the maximum of this property and
     * the result of the computation based on the com.openexchange.usm.session.sync.email_pull_delay property and the value specified as
     * parameter to the waitForChanges()-call. If the effective value is greater than the parameter to the waitForChanges()-call, no pull
     * will be performed.<br>
     * To disable pulling completely, set the value of this property to a negative value.
     */
    public static final String SYNC_EMAIL_PULL_MIN_DELAY_PROPERTY = "com.openexchange.usm.session.sync.email_pull_min_delay";

    /**
     * Default value (1 minute) for the minimum wait interval required before an email pull is performed.
     */
    public static final int SYNC_EMAIL_PULL_MIN_DELAY_DEFAULT = 60000;

    /**
     * Defines a limit on the number of sync states that are stored in the database and local memory cache for each folder and the folder
     * hierarchy itself (per session/client). If a client connects to the USM with a given sync state, all older states are automatically
     * removed, the new state is computed, stored in the database and sent to the client. As long a client uses an old synckey, new keys
     * keep getting generated, but none deleted. If the limit specified here is reached, the USM deletes the oldest sync state on a client
     * connect that is not the one the client just used. Note that the value of this property imposes an upper limit to the number of sync
     * states stored in the cache and database. If a client behaves as expected, only 2 states will be stored (the one used by the client to
     * initiate synchronization in case the synchronization result does not reach the client, and the new generated one that contains the
     * base state for the next synchronization). The lower limit for this property is 2, since at least 2 sync states are required to
     * account for communication failures.
     */
    public static final String SYNC_MAX_STATES_IN_DB_PROPERTY = "com.openexchange.usm.session.sync.max_states_in_db";

    /**
     * Default value(5) for com.openexchange.usm.session.sync.max_states_in_db.
     */
    public static final int SYNC_MAX_STATES_IN_DB_DEFAULT = 5;

    /**
     * This property defines how long server data should be cached after a sync has been completed (i.e. all changes have been reported to
     * the client). A value of 0 specifies "no caching". If a new sync is executed within the time frame and the data is still available in
     * the cache, the cached data is used instead of retrieving the current data from the OX server.
     */
    public static final String SYNC_CACHE_TIME_LIMIT_COMPLETE_PROPERTY = "com.openexchange.usm.session.sync.cache.time_limit.complete";

    /**
     * Default value(60000 ms. == 1 minute) for com.openexchange.usm.session.sync.cache.time_limit.normal.
     */
    public static final int SYNC_CACHE_TIME_LIMIT_COMPLETE_DEFAULT = 60000; // 1 minute

    /**
     * This property defines how long server data should be cached after an incomplete sync has been executed (i.e. NOT all changes have
     * been reported to the client). A value of 0 specifies "no caching". If an additional (follow-up) sync is executed within the time
     * frame and the data is still available in the cache, the cached data is used instead of retrieving the current data from the OX
     * server.
     */
    public static final String SYNC_CACHE_TIME_LIMIT_INCOMPLETE_PROPERTY = "com.openexchange.usm.session.sync.cache.time_limit.incomplete";

    /**
     * Default value(1200000 ms. == 20 minutes) for com.openexchange.usm.session.sync.cache.time_limit.incomplete.
     */
    public static final int SYNC_CACHE_TIME_LIMIT_INCOMPLETE_DEFAULT = 1200000; // 20 minutes

    public static final String OBJECTS_IN_FOLDER_SYNC_LIMIT_PROPERTY = "com.openexchange.usm.max_objects_per_folder";

    public static final int OBJECTS_IN_FOLDER_SYNC_LIMIT_DEFAULT = 65535;

    public static final String ATTACHMENT_COUNT_LIMIT_PER_OBJECT_PROPERTY = "com.openexchange.usm.max_attachment_count_per_object";

    public static final int ATTACHMENT_COUNT_LIMIT_PER_OBJECT_DEFAULT = 100;

    public static final String ATTACHMENT_SIZE_LIMIT_PER_OBJECT_PROPERTY = "com.openexchange.usm.max_attachment_size_per_object";

    public static final long ATTACHMENT_SIZE_LIMIT_PER_OBJECT_DEFAULT = 50L * 1024L * 1024L;

    /**
     * Defines a timeout after which unused temporary files uploaded by a protocol will be automatically removed
     */
    public static final String TEMP_FILES_TIMEOUT_PROPERTY = "com.openexchange.usm.temp_files.timeout";

    public static final int TEMP_FILES_TIMEOUT_DEFAULT = 7200000; // 2 hours

    // Database connection properties (only used in stand-alone mode)

    public static final String MYSQL_DATABASE_HOST_PROPERTY = "com.openexchange.usm.database.mysql.host";

    public static final String MYSQL_DATABASE_PORT_PROPERTY = "com.openexchange.usm.database.mysql.port";

    public static final String MYSQL_DATABASE_NAME_PROPERTY = "com.openexchange.usm.database.mysql.dbname";

    public static final String MYSQL_DATABASE_USER_PROPERTY = "com.openexchange.usm.database.mysql.user";

    public static final String MYSQL_DATABASE_PASSWORD_PROPERTY = "com.openexchange.usm.database.mysql.password";

    public static final String MYSQL_DATABASE_HOST_PROPERTY_DEFAULT = "localhost";

    public static final String MYSQL_DATABASE_PORT_PROPERTY_DEFAULT = "3306";

    public static final String MYSQL_DATABASE_NAME_PROPERTY_DEFAULT = "test";

    public static final String MYSQL_DATABASE_USER_PROPERTY_DEFAULT = "";

    public static final String MYSQL_DATABASE_PASSWORD_PROPERTY_DEFAULT = "";

    public static final String HSQL_CONNECTION_PROPERTY = "com.openexchange.usm.database.hsql.url";

    public static final String HSQL_CONNECTION_DEFAULT = "jdbc:hsqldb:file:db/hsqldb";

    public static final String HSQL_USER_PROPERTY = "com.openexchange.usm.database.hsql.user";

    public static final String HSQL_USER_DEFAULT = "sa";

    public static final String HSQL_PASSWORD_PROPERTY = "com.openexchange.usm.database.hsql.password";

    public static final String HSQL_PASSWORD_DEFAULT = "";
    
    // --- JSON/OLOX2 ---

    // TODO Create properties for this value ! Why this strange default value ?
    public static final long START_DATE_MINUS_DEFAULT = 311040000L;// 36*1000*24*360 = 3.6 days

    // TODO Create properties for this value ! Why this strange default value ?
    public static final long END_DATE_PLUS_DEFAULT = 3110400L;// 360*24*360 = 51.84 minutes

    public static final String JSON_SERVLET_ALIAS_PROPERTY = "com.openexchange.usm.json.alias";

    public static final String JSON_SERVLET_ALIAS_DEFAULT = "/usm-json";

    public static final String JSON_SERVLET_PING_MIN_INTERVAL = "com.openexchange.usm.json.ping.min_interval";

    public static final int JSON_SERVLET_PING_MIN_INTERVAL_DEFAULT = 5000;

    public static final String JSON_SERVLET_PING_MAX_INTERVAL = "com.openexchange.usm.json.ping.max_interval";

    public static final int JSON_SERVLET_PING_MAX_INTERVAL_DEFAULT = 360000;

    public static final String JSON_SERVLET_PING_MAX_REQUEST_PER_SECOND = "com.openexchange.usm.json.ping.max_requests_per_second";

    public static final int JSON_SERVLET_PING_MAX_REQUEST_PER_SECOND_DEFAULT = 100;

    public static final String JSON_SERVLET_SYNC_MAX_LIMIT_PROPERTY = "com.openexchange.usm.json.sync.limit.objects";

    public static final int JSON_SERVLET_SYNC_MAX_LIMIT_DEFAULT = 0;

    public static final String JSON_SERVLET_EMAIL_SIZE_LIMIT_PROPERTY = "com.openexchange.usm.json.sync.limit.email_size";

    public static final int JSON_SERVLET_EMAIL_SIZE_LIMIT_DEFAULT = 2097152; // 2 MByte

    public static final String JSON_SERVLET_PIM_ATTACHMENT_COUNT_LIMIT_PROPERTY = "com.openexchange.usm.json.sync.limit.pim_attachment_count";

    public static final int JSON_SERVLET_PIM_ATTACHMENT_COUNT_LIMIT_DEFAULT = 5;

    public static final String JSON_SERVLET_MAX_EMAIL_SIZE_PROPERTY = "com.openexchange.usm.json.max_email_size";

    public static final int JSON_SERVLET_MAX_EMAIL_SIZE_DEFAULT = 50 * 1024 * 1024; // 50 MByte

    public static final String JSON_SERVLET_MAX_INLINE_ATTACHMENT_SIZE_PROPERTY = "com.openexchange.usm.json.max_inline_attachment_size";

    public static final int JSON_SERVLET_MAX_INLINE_ATTACHMENT_SIZE_DEFAULT = 1024 * 1024; // 1 MByte

    /**
     * Defines the lower time filter limit in days. Appointments and mails which are in the past beyond this limit will not be synchronised
     * from/to the client.
     */
    public static final String JSON_SERVLET_TIMEFILTER_LIMIT_PAST_PROPERTY = "com.openexchange.usm.json.sync.timefilter_limit_past";

    public static final int JSON_SERVLET_TIMEFILTER_LIMIT_PAST_DEFAULT = 365;

    /**
     * Defines the upper time filter limit in days. Appointments and mails which are in the future beyond this limit will not be
     * synchronised from/to the client.
     */
    public static final String JSON_SERVLET_TIMEFILTER_LIMIT_FUTURE_PROPERTY = "com.openexchange.usm.json.sync.timefilter_limit_future";

    public static final int JSON_SERVLET_TIMEFILTER_LIMIT_FUTURE_DEFAULT = 730; // 2 * 365
    
    
    /**
     * Defines the lower time filter limit in days. Appointments which are in the past beyond this limit will not be synchronised
     * from/to the client. Overwrites global appointment & mail time filter. 
     */
    public static final String JSON_SERVLET_TIMEFILTER_CALENDAR_LIMIT_PAST_PROPERTY = "com.openexchange.usm.json.sync.calendar.timefilter_limit_past";

    public static final int JSON_SERVLET_TIMEFILTER_CALENDAR_LIMIT_PAST_DEFAULT = 365;

    /**
     * Defines the upper time filter limit in days. Appointments which are in the future beyond this limit will not be
     * synchronised from/to the client. Overwrites global appointment & mail time filter. 
     */
    public static final String JSON_SERVLET_TIMEFILTER_CALENDAR_LIMIT_FUTURE_PROPERTY = "com.openexchange.usm.json.sync.calendar.timefilter_limit_future";

    public static final int JSON_SERVLET_TIMEFILTER_CALENDAR_LIMIT_FUTURE_DEFAULT = 730; // 2 * 365
    

    public static final String JSON_SERVLET_ACCESS_CHECK_TEST_INTERVAL_PROPERTY = "com.openexchange.usm.json.access_check.test_interval";

    public static final int JSON_SERVLET_ACCESS_CHECK_TEST_INTERVAL_DEFAULT = 0;

    public static final String JSON_SERVLET_ACCESS_CHECK_MAX_ACCESSES_PROPERTY = "com.openexchange.usm.json.access_check.max_accesses";

    public static final int JSON_SERVLET_ACCESS_CHECK_MAX_ACCESSES_DEFAULT = 0;

    public static final String JSON_SERVLET_ACCESS_CHECK_REFUSAL_INTERVAL_PROPERTY = "com.openexchange.usm.json.access_check.refusal_interval";

    public static final int JSON_SERVLET_ACCESS_CHECK_REFUSAL_INTERVAL_DEFAULT = 0;

    /**
     * Defines the interval after which usm should poll the inbox if no changes have arrived.
     */
    public static final String JSON_SERVLET_INBOX_POLLING_INTERVAL_PROPERTY = "com.openexchange.usm.json.sync.inbox_polling_interval";

    public static final int JSON_SERVLET_INBOX_POLLING_INTERVAL_DEFAULT = 600000; // 10 min
    
    // --- SyncML ---
    
    public static final String SYNCML_SERVLET_ALIAS_PROPERTY = "com.openexchange.usm.syncml.alias";

    public static final String SYNCML_SERVLET_ALIAS_DEFAULT = "/usm-syncml";

    public static final String SYNCML_SYNC_TIMEOUT_PROPERTY = "com.openexchange.usm.syncml.timeout";

    public static final int SYNCML_SYNC_TIMEOUT_DEFAULT = 600000; // 10 minutes

    public static final String SYNCML_FORMATTED_SERVER_ANCHORS_PROPERTY = "com.openexchange.usm.syncml.formatted_server_anchors";

    public static final boolean SYNCML_FORMATTED_SERVER_ANCHORS_DEFAULT = false;

    public static final String SYNCML_DEFAULT_MAX_MESSAGE_SIZE_PROPERTY = "com.openexchange.usm.syncml.default_max_message_size";

    public static final int SYNCML_DEFAULT_MAX_MESSAGE_SIZE_DEFAULT = 10000;

    public static final String SYNCML_SYNC_ALL_FOLDERS_PROPERTY = "com.openexchange.usm.syncml.sync_all_folders";

    public static final boolean SYNCML_SYNC_ALL_FOLDERS_DEFAULT = false;

    public static final String SYNCML_REDIRECT_URLS_PROPERTY = "com.openexchange.usm.syncml.redirect_urls";

    public static final boolean SYNCML_REDIRECT_URLS_DEFAULT = true;

    public static final String SYNCML_DEBUG_LOG_PROPERTY = "com.openexchange.usm.syncml.debug_log";

    public static final String SYNCML_DEBUG_LOG_DEFAULT = null;
}
