/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.session.sync;
import com.openexchange.usm.api.database.StorageAccessException;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.exceptions.USMStorageException;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.session.dataobject.DataObjectSet;

public interface ContentSyncerStorage {

    /**
     * Retrieves the data currently stored on the OX server. Stores all objects retrieved in the provided DataObjectSet, returns false if
     * the read was incomplete (i.e. not all new objects have been read from the OX server) and therefore must not be stored in the
     * OXDataCache
     * 
     * @param resultSet DataObjectSet in which the current folder content will be stored
     * @return true if the read was complete, false if the result was incomplete and must not be stored in any caches
     * @throws USMException
     */
    boolean getCurrentServerData(DataObjectSet resultSet) throws USMException;

    /**
     * Stores the given DataObjects associated with the timestamp. It returns the possibly modified timestamp, under which the data was
     * actually stored (if the provided timestamp was too old or the same as an already existing timestamp).
     * 
     * @param timestamp
     * @param data
     * @param updatesPending true if the stored data does not represent the current server state, i.e. the folder (hierarchy) should not be
     *            marked as "in sync" (for waitForChanges)
     * @return
     */
    long storeServerData(long timestamp, DataObjectSet data, boolean updatesPending) throws USMException;


    /**
     * @return {@link OXDataCacheID} for the OX server storage (a specific folder or the folder hierarchy for a session)
     */
    OXDataCacheID getCacheID();

    /**
     * Tries to find a matching client object for a given server object. This method is called if no client object with the same ID as the
     * given server object could be found. The method must either return one of the objects in clientDataObjects or null if none of the
     * objects matches based on other fields (e.g. standard_folder_type).
     * 
     * @param clientDataObjects set of available client objects to match
     * @param serverObject server object to match against
     * @return matching client object or null if none matches
     */
    DataObject findMatchingClientObjectForServerObject(DataObjectSet clientDataObjects, DataObject serverObject);
}
