/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.session.storage;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import com.openexchange.usm.api.cache.SyncStateCache;
import com.openexchange.usm.api.contenttypes.common.ContentType;
import com.openexchange.usm.api.database.CachedDBEntry;
import com.openexchange.usm.api.database.StorageAccessException;
import com.openexchange.usm.api.exceptions.DeserializationFailedException;
import com.openexchange.usm.api.exceptions.USMIllegalStateException;
import com.openexchange.usm.api.exceptions.USMStorageException;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.assets.CompleteSessionID;
import com.openexchange.usm.api.session.storage.PersistentSyncStateStorage;
import com.openexchange.usm.session.dataobject.DataObjectSet;
import com.openexchange.usm.session.impl.SessionImpl;
import com.openexchange.usm.session.impl.USMSessionManagementErrorCodes;
import com.openexchange.usm.util.Toolkit;

/**
 * DB DataObject storage.
 * 
 * @author ldo
 */
public class SyncStateStorage {

    private final SessionImpl _session;

    private final Map<String, long[]> _timestamps = new HashMap<String, long[]>();

    private final SyncStateCache _syncStateCache;

    private static final Log LOG = LogFactory.getLog(SyncStateStorage.class);

    public final static String FOLDER_HIERARCHY_ID = "";

    /**
     * Initializes a new {@link SyncStateStorage}.
     * 
     * @param session
     */
    public SyncStateStorage(SessionImpl session) {
        _session = session;
        _syncStateCache = session.getSessionManager().getSyncStateCacheProvider().getCache(session.getCompleteSessionID());
    }

    /**
     * Store the specified DataObjectSet.
     * 
     * @param objectID
     * @param timestamp
     * @param objectSet
     * @param timestampToKeep
     * @return
     * @throws StorageAccessException
     * @throws USMStorageException
     */
    public long put(String objectID, long timestamp, DataObjectSet objectSet, long timestampToKeep) throws StorageAccessException, USMStorageException {
        Serializable[][] serializedObjects = serializeObjects(objectSet);
        long[] timestamps = getTimestamps(objectID);
        long newTS = timestamp;
        long oldTS = timestampToKeep;
        if (timestamps != null && timestamps.length > 0) {
            long newestTimestamp = getNewestTimestamp(timestamps);
            Serializable[][] newestData = getCachedData(objectID, newestTimestamp);
            if (!hasStateChanged(newestTimestamp, newestData, objectSet, _session)) {
                if (LOG.isDebugEnabled())
                    LOG.debug(_session + " Storing in DB cache not necessary, no changes for " + objectID + ", timestamp " + timestamp + "->" + newestTimestamp + ", keeping " + timestampToKeep);
                return newestTimestamp;
            }
            newTS = Math.max(newestTimestamp + 1L, newTS);
            int timestampLimit = _session.getSessionManager().getMaxSyncStatesInDB();
            oldTS = updateLocalTimestamps(objectID, timestampLimit, timestampToKeep, newTS, timestamps);
        } else {
            storeLocalTimestamps(objectID, new long[] { newTS });
        }

        byte[] data = compress(serializedObjects);
        // SyncStateStorageStatistics.collectCompressionStatistics(serializedObjects, data);
        boolean tooManyTimestamps = oldTS != timestampToKeep;
        String dbData = Toolkit.encodeBase64(data);
        if (LOG.isDebugEnabled()) {
            if (tooManyTimestamps)
                LOG.debug(_session + " Storing in DB cache '" + objectID + "' for " + newTS + ", keeping " + timestampToKeep + ", removing older than " + oldTS);
            else
                LOG.debug(_session + " Storing in DB cache '" + objectID + "' for " + newTS + ", removing older than " + oldTS);
        }

        // store in db
        long newDbTS = getStorage().store(_session, objectID, dbData, oldTS, newTS, timestampToKeep);
        if (newDbTS != newTS) {
            replaceLocalTimestamp(getLocalTimestamps(objectID), newTS, newDbTS);
        }

        // store in cache
        _syncStateCache.put(_session.getCompleteSessionID(), objectID, newDbTS, serializedObjects);

        // update bean info
        _session.getSessionManager().syncStateSavedToDatabase();

        return newTS;
    }

    private static void replaceLocalTimestamp(long[] localTimestamps, long oldTS, long newTS) {
        if (localTimestamps == null)
            return;
        for (int i = localTimestamps.length - 1; i >= 0; i--) {
            if (localTimestamps[i] == oldTS) {
                localTimestamps[i] = newTS;
                return;
            }
        }
    }

    /**
     * Re-maps the already stored object IDs.
     * 
     * @param oldObjectID
     * @param newObjectID
     * @throws StorageAccessException
     * @throws USMStorageException
     */
    public void remapStates(String oldObjectID, String newObjectID) throws StorageAccessException, USMStorageException {
        if (LOG.isDebugEnabled())
            LOG.debug(_session + " Remapping states for object :  " + oldObjectID + " to new object id: " + newObjectID);
        remapLocalTimestamps(oldObjectID, newObjectID);
        _syncStateCache.remapStates(getSessionID(), oldObjectID, newObjectID);

        // restore the data in DB
        getStorage().remapStates(_session, oldObjectID, newObjectID);
    }

    /**
     * Special method that returns the newest timestamp for internal USM methods that do not know of the current sync key / timestamp.
     * 
     * @param objectID
     * @return
     * @throws StorageAccessException
     * @throws USMStorageException
     */
    public long getNewestTimestamp(String objectID) throws StorageAccessException, USMStorageException {
        long[] timestamps = getLocalTimestamps(objectID);
        if (timestamps == null || timestamps.length == 0) {
            readSyncStatesFromDB(objectID);
            timestamps = getLocalTimestamps(objectID);
        }
        return (timestamps == null || timestamps.length == 0) ? 0L : timestamps[timestamps.length - 1];
    }

    /**
     * Removes all DB entries for the given user and device that have object IDs as the ones provided as parameters
     * 
     * @param user
     * @param device
     * @param objectIDs
     * @throws USMStorageException
     */
    public void remove(String... objectIDs) throws USMStorageException {
        if (LOG.isDebugEnabled())
            LOG.debug(_session + " Removing from DB cache " + Arrays.toString(objectIDs));
        removeLocalTimestamps(objectIDs);
        _syncStateCache.removeSyncStates(getSessionID(), objectIDs);
        getStorage().remove(_session, objectIDs);
    }

    /**
     * Removes all DB entries for the given user and device that have object IDs other than the ones provided as parameters
     * 
     * @param objectIDs
     * @throws USMStorageException
     */
    public void retain(String... objectIDs) throws USMStorageException {
        if (LOG.isDebugEnabled())
            LOG.debug(_session + " Removing from DB cache all except " + Arrays.toString(objectIDs));

        // preparation
        Set<String> objectIDSet = new HashSet<String>();
        for (String objectId : objectIDs) {
            objectIDSet.add(objectId);
        }
        retainLocalTimestamps(objectIDSet);
        _syncStateCache.retainSyncStates(getSessionID(), objectIDSet); // TODO verify that Folder Hierarchy is not removed by accident

        // search which DB entries need to be removed
        List<String> toDelete = getStorage().getRemainingObjectIDs(_session, objectIDSet);

        // If any have been found, remove those DB entries
        if (!toDelete.isEmpty())
            remove(toDelete.toArray(new String[toDelete.size()]));
    }

    /**
     * Removes all saved states for this session
     * 
     * @throws USMStorageException
     */
    public void removeAllObjectsForSession() throws USMStorageException {
        if (LOG.isDebugEnabled())
            LOG.debug(_session + " Removing from DB cache all elements for session: " + _session.getSessionId());
        _session.getSessionManager().getSyncStateCacheProvider().removeCacheCompletely(_session.getCompleteSessionID());
        getStorage().purgeSession(_session);
    }

    /**
     * Read Sync States from DB
     * 
     * @param objectID
     * @throws StorageAccessException
     * @throws USMStorageException
     */
    private void readSyncStatesFromDB(String objectID) throws StorageAccessException, USMStorageException {
        List<CachedDBEntry> dbList = getStorage().get(_session, objectID, _session.getSessionManager().getMaxSyncStatesInDB());

        long[] timestamps = new long[dbList.size()];
        _syncStateCache.removeSyncStates(getSessionID(), objectID);
        int i = 0;
        for (CachedDBEntry entry : dbList) {
            try {
                _syncStateCache.put(getSessionID(), objectID, entry.getTimestamp(), decompress(Toolkit.decodeBase64(entry.getData())));
                timestamps[i++] = entry.getTimestamp();
            } catch (DeserializationFailedException e) {
                _session.logDeserializationError(objectID, entry.getTimestamp(), e);
            }
        }
        if (timestamps.length != i) { // In case a DeserializationFailedException occurred
            long[] deserializedTimestamps = new long[i];
            System.arraycopy(timestamps, 0, deserializedTimestamps, 0, i);
            timestamps = deserializedTimestamps;
        }
        storeLocalTimestamps(objectID, timestamps);
        _session.getSessionManager().syncStatesLoadedFromDatabase(dbList.size());
    }

    private static Serializable[][] decompress(byte[] compressedData) throws DeserializationFailedException {
        try {
            ObjectInputStream in = new ObjectInputStream(new GZIPInputStream(new ByteArrayInputStream(compressedData)));
            return (Serializable[][]) in.readObject();
        } catch (ClassCastException e) {
            throw new DeserializationFailedException(
                USMSessionManagementErrorCodes.DATA_STORAGE_CACHE_CLASS_CAST_EXCEPTION,
                "Stored object is not of correct type",
                e);
        } catch (IOException e) {
            throw new DeserializationFailedException(
                USMSessionManagementErrorCodes.DATA_STORAGE_CACHE_IO_EXCEPTION,
                "Can not retrieve object from bytes",
                e);
        } catch (ClassNotFoundException e) {
            throw new DeserializationFailedException(
                USMSessionManagementErrorCodes.DATA_STORAGE_CACHE_CLASS_NOT_FOUND_EXCEPTION,
                "Can not find the class of the serializable object",
                e);
        }
    }

    private static byte[] compress(Serializable[][] objects) {
        try {
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            ObjectOutputStream oos = new ObjectOutputStream(new GZIPOutputStream(baos));
            oos.writeObject(objects);
            oos.close();
            return baos.toByteArray();
        } catch (IOException e) {
            throw new USMIllegalStateException(
                USMSessionManagementErrorCodes.SERIALIZATION_ERROR,
                "Can not serialize data to byte array",
                e);
        }
    }

    private long[] getLocalTimestamps(String objectID) {
        synchronized (_timestamps) {
            return _timestamps.get(objectID);
        }
    }

    private void storeLocalTimestamps(String objectID, long[] newTimestamps) {
        Arrays.sort(newTimestamps);
        synchronized (_timestamps) {
            _timestamps.put(objectID, newTimestamps);
        }
    }

    private void removeLocalTimestamps(String... objectIDs) {
        synchronized (_timestamps) {
            for (String id : objectIDs)
                _timestamps.remove(id);
        }
    }

    private void remapLocalTimestamps(String oldObjectID, String newObjectID) {
        synchronized (_timestamps) {
            long[] timestamps = _timestamps.remove(oldObjectID);
            if (timestamps != null)
                _timestamps.put(newObjectID, timestamps);
        }
    }

    private void retainLocalTimestamps(Set<String> objectIDSet) {
        synchronized (_timestamps) {
            _timestamps.keySet().retainAll(objectIDSet);
        }
    }

    private long[] getTimestamps(String objectID) throws StorageAccessException, USMStorageException {
        long[] timestamps = getLocalTimestamps(objectID);
        if (timestamps != null && timestamps.length > 0)
            return timestamps;
        readSyncStatesFromDB(objectID);
        return getLocalTimestamps(objectID);
    }

    public long[] getSortedTimestamps(String objectID) throws StorageAccessException, USMStorageException {
        return getTimestamps(objectID);
    }

    private static long getNewestTimestamp(long... timestamps) {
        long newest = 0L;
        for (long timestamp : timestamps)
            newest = Math.max(newest, timestamp);
        return newest;
    }

    private static Long findOldestTimestamp(long timestampToIgnore, Set<Long> timestamps) {
        Long oldestTimestamp = null;
        for (Long timestamp : timestamps) {
            if (timestamp != timestampToIgnore && (oldestTimestamp == null || oldestTimestamp > timestamp)) {
                oldestTimestamp = timestamp;
            }
        }
        return oldestTimestamp;
    }

    private void removeOldestTimestamp(String objectID, long timestampToKeep, Set<Long> timestamps) {
        Long oldestTimestamp = findOldestTimestamp(timestampToKeep, timestamps);
        if (oldestTimestamp != null) {
            _syncStateCache.remove(getSessionID(), objectID, oldestTimestamp);
            timestamps.remove(oldestTimestamp);
        }
    }

    private long updateLocalTimestamps(String objectID, int limit, long timestampToKeep, long timestampToAdd, long[] timestamps) {
        Set<Long> ts = new HashSet<Long>();
        for (long timestamp : timestamps) {
            if (timestamp < timestampToKeep) {
                _syncStateCache.remove(getSessionID(), objectID, timestamp);
            } else {
                ts.add(timestamp);
            }
        }
        for (int i = 0; ts.size() >= limit && i < ts.size(); i++) {
            removeOldestTimestamp(objectID, timestampToKeep, ts);
        }
        long[] newTimestamps = new long[ts.size() + 1];
        int i = 0;
        for (Long timestamp : ts) {
            newTimestamps[i++] = timestamp;
        }
        newTimestamps[i] = timestampToAdd;
        storeLocalTimestamps(objectID, newTimestamps);
        Long oldestTimestamp = findOldestTimestamp(timestampToKeep, ts);
        return (oldestTimestamp == null) ? timestampToKeep : oldestTimestamp;
    }

    private static boolean hasStateChanged(long newestTimestamp, Serializable[][] serializedData, DataObjectSet objects, SessionImpl session) {
        int len1 = (serializedData == null) ? 0 : serializedData.length;
        int len2 = (objects == null) ? 0 : objects.size();
        if (len1 != len2) {
            return true;
        }
        if (len1 == 0) {
            return false;
        }
        if(objects == null)
            return serializedData == null;
        ContentType contentType = objects.iterator().next().getContentType();
        DataObject[] cachedObjects = deserializeObjects(serializedData, newestTimestamp, contentType, session);
        return !objects.isEqualTo(cachedObjects);
    }

    private static DataObject[] deserializeObjects(Serializable[][] values, long timestamp, ContentType contentType, SessionImpl session) {
        DataObject[] result = new DataObject[values.length];
        for (int i = 0; i < result.length; i++) {
            DataObject o = contentType.newDataObject(session);
            try {
                o.deserialize(timestamp, values[i]);
            } catch (DeserializationFailedException e) {
                return null;
            }
            result[i] = o;
        }
        return result;
    }

    private static Serializable[][] serializeObjects(DataObjectSet objects) {
        List<String> ids = new ArrayList<String>(objects.getIDs());
        Collections.sort(ids);
        Serializable[][] data = new Serializable[ids.size()][];
        for (int i = 0; i < ids.size(); i++) {
            data[i] = objects.get(ids.get(i)).serialize();
        }
        return data;
    }

    public Serializable[][] get(String objectID, long timestamp) throws StorageAccessException, USMStorageException {
        return getCachedData(objectID, timestamp);
    }

    private Serializable[][] getCachedData(String objectID, long timestamp) throws StorageAccessException, USMStorageException {
        Serializable[][] data = _syncStateCache.get(getSessionID(), objectID, timestamp);
        if (data != null) {
            return data;
        }
        readSyncStatesFromDB(objectID);
        return _syncStateCache.get(getSessionID(), objectID, timestamp);
    }

    public SyncStateCache getSyncStateCache() {
        return _syncStateCache;
    }

    private CompleteSessionID getSessionID() {
        return _session.getCompleteSessionID();
    }

    private PersistentSyncStateStorage getStorage() {
        return _session.getSessionManager().getSyncStateStorage();
    }
}
