/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.session.osgi;

import javax.management.ObjectName;
import org.osgi.framework.Filter;
import org.osgi.framework.ServiceReference;
import org.osgi.util.tracker.ServiceTracker;
import org.osgi.util.tracker.ServiceTrackerCustomizer;
import com.openexchange.cluster.timer.ClusterTimerService;
import com.openexchange.filemanagement.ManagedFileManagement;
import com.openexchange.groupware.settings.PreferencesItemService;
import com.openexchange.groupware.settings.tree.modules.interfaces.USM;
import com.openexchange.groupware.settings.tree.modules.interfaces.USMActive;
import com.openexchange.management.ManagementService;
import com.openexchange.timer.TimerService;
import com.openexchange.usm.api.cache.SyncStateCacheProvider;
import com.openexchange.usm.api.configuration.ConfigurationManager;
import com.openexchange.usm.api.contenttypes.common.ContentTypeManager;
import com.openexchange.usm.api.contenttypes.folder.FolderContentType;
import com.openexchange.usm.api.mapping.storage.ContextUUIDMappingStorage;
import com.openexchange.usm.api.mapping.storage.FolderIDMappingStorage;
import com.openexchange.usm.api.ox.json.OXJSONAccess;
import com.openexchange.usm.api.session.SessionManager;
import com.openexchange.usm.api.session.storage.PersistentSessionDataStorage;
import com.openexchange.usm.api.session.storage.PersistentSyncStateStorage;
import com.openexchange.usm.session.impl.SessionManagerImpl;
import com.openexchange.usm.session.jmx.USMCacheInformation;
import com.openexchange.usm.session.jmx.USMSessionInformation;
import com.openexchange.usm.session.jmx.USMSessionInformationMBean;
import com.openexchange.usm.util.AbstractUSMActivator;
import com.openexchange.usm.util.OSGiToolkit;
import com.openexchange.usm.util.TempFileStorage;

/**
 * {@link SessionManagerActivator}
 * 
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 */
public class SessionManagerActivator extends AbstractUSMActivator {

    private ObjectName objectName;

    /*
     * (non-Javadoc)
     * @see com.openexchange.usm.util.AbstractUSMActivator#getNeededServices()
     */
    @Override
    protected Class<?>[] getNeededServices() {
        return new Class<?>[] { OXJSONAccess.class, 
                                ContentTypeManager.class, 
                                FolderContentType.class, 
                                ConfigurationManager.class,
                                ContextUUIDMappingStorage.class, 
                                FolderIDMappingStorage.class, 
                                ManagementService.class, 
                                PersistentSyncStateStorage.class,
                                PersistentSessionDataStorage.class
                                };
    }

    /*
     * (non-Javadoc)
     * @see com.openexchange.usm.util.ExtendedUSMActivator#getOptionalServices()
     */
    @Override
    protected Class<?>[] getOptionalServices() {
        return new Class<?>[] { ManagedFileManagement.class, TimerService.class, SyncStateCacheProvider.class, ClusterTimerService.class};
    }

    /*
     * (non-Javadoc)
     * @see com.openexchange.usm.util.AbstractUSMActivator#register(org.osgi.framework.BundleContext)
     */
    @Override
    protected void register() throws Exception {
        OXJSONAccess jsonAccess = (OXJSONAccess) getService(OXJSONAccess.class);
        ContentTypeManager ctManager = (ContentTypeManager) getService(ContentTypeManager.class);
        FolderContentType fct = (FolderContentType) getService(FolderContentType.class);
        ConfigurationManager cfManager = (ConfigurationManager) getService(ConfigurationManager.class);
        FolderIDMappingStorage folderIdMapping = (FolderIDMappingStorage) getService(FolderIDMappingStorage.class);
        ContextUUIDMappingStorage uuidMapping = (ContextUUIDMappingStorage) getService(ContextUUIDMappingStorage.class);
        ManagementService managementService = (ManagementService) getService(ManagementService.class);
        PersistentSyncStateStorage ssStorage = (PersistentSyncStateStorage) getService(PersistentSyncStateStorage.class);
        PersistentSessionDataStorage psdStorage = (PersistentSessionDataStorage) getService(PersistentSessionDataStorage.class);
            
        final SessionManagerImpl sessionManager = new SessionManagerImpl(
            jsonAccess,
            ctManager,
            fct,
            cfManager,
            folderIdMapping,
            uuidMapping,
            ssStorage,
            psdStorage);

        registerService(SessionManager.class, sessionManager);
        LOG.info("SessionManager registered.");

        USM usm = new USM();
        context.registerService(PreferencesItemService.class, usm, null);
        LOG.info("USM Preferences Item Service activated");

        USMActive usmActive = new USMActive();
        context.registerService(PreferencesItemService.class, usmActive, null);
        LOG.info("USM Active Preferences Item Service activated");

        
        try {
            USMCacheInformation usmCacheInfo = sessionManager.getUSMCacheInformation();
            objectName = OSGiToolkit.getObjectName(USMSessionInformation.class.getName(), USMSessionInformationMBean.SESSION_DOMAIN);
            managementService.registerMBean(objectName, new USMSessionInformation(sessionManager, usmCacheInfo));
            LOG.info("JMX Bean USMCacheInformation registered.");
        } catch (Exception e) {
            LOG.error("Failed start-up of bundle com.openexchange.session.impl: " + e.getMessage(), e);
        }
        
        // track optional services
        Filter optionalServices = context.createFilter(OSGiToolkit.createServiceFilter(getOptionalServices()));
        ServiceTracker<Object, Object> tracker = new ServiceTracker<Object, Object>(context, optionalServices, new ServiceTrackerCustomizer<Object, Object>() {
                @Override
                public Object addingService(ServiceReference<Object> reference) {
                    Object o = context.getService(reference);
                    if (o instanceof ManagedFileManagement) {
                        TempFileStorage.setManagedFileManagement((ManagedFileManagement) o);
                    } else if (o instanceof TimerService) {
                        sessionManager.setTimerService((TimerService) o);
                    } else if (o instanceof SyncStateCacheProvider) {
                        sessionManager.setSyncStateCacheProvider((SyncStateCacheProvider) o);
                    } else if (o instanceof ManagementService) {
                        sessionManager.setManagementService((ManagementService) o);
                        sessionManager.setSyncStateCacheProvider((SyncStateCacheProvider) o);
                    } else if (o instanceof ClusterTimerService) {
                        sessionManager.setClusterTimerService((ClusterTimerService) o);
                    }

                    return o;
                }

                @Override
                public void modifiedService(ServiceReference<Object> reference, Object service) {
                    // nothing
                }

                @Override
                public void removedService(ServiceReference<Object> reference, Object service) {
                    Object o = context.getService(reference);
                    if (o instanceof ManagedFileManagement) {
                        TempFileStorage.setManagedFileManagement(null);
                    } else if (o instanceof TimerService) {
                        sessionManager.setTimerService(null);
                    } else if (o instanceof SyncStateCacheProvider) {
                        sessionManager.setSyncStateCacheProvider(null);
                    } else if (o instanceof ManagementService) {
                        sessionManager.setManagementService(null);
                        Object sessionManager = getService(SessionManager.class);
                        if (sessionManager instanceof SessionManagerImpl) {
                            ((SessionManagerImpl) sessionManager).setSyncStateCacheProvider(null);
                        }
                    } else if (o instanceof ClusterTimerService) {
                        sessionManager.setClusterTimerService(null);
                    }
                }
            });
        tracker.open();
    }

    /*
     * (non-Javadoc)
     * @see com.openexchange.usm.util.AbstractUSMActivator#unregister(org.osgi.framework.BundleContext)
     */
    @Override
    protected void unregister() {
        SessionManagerImpl s = (SessionManagerImpl) getRegisteredService(SessionManager.class);
        ManagementService m = (ManagementService) getService(ManagementService.class);
        if (s != null)
            s.deactivate();

        if (m != null && objectName != null) {
            try {
                m.unregisterMBean(objectName);
                LOG.info("JMX Bean USMCacheInformation unregistered.");
            } catch (Exception e) {
                LOG.error("Failed to unregisterd JMX Bean USMCacheInformation: " + e.getMessage(), e);
            } finally {
                objectName = null;
            }
        }
        LOG.info("SessionManager unregistered.");
    }
}
