/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.session.jmx;

import java.util.concurrent.atomic.AtomicInteger;
import com.openexchange.usm.session.impl.SessionManagerImpl;

/**
 * {@link USMCacheInformation}
 * 
 * @author <a href="mailto:afe@microdoc.de">Alexander Feess</a>
 */
public class USMCacheInformation {

    private final SessionManagerImpl _sessionManager;

    private final AtomicInteger _newSessions = new AtomicInteger();

    private final AtomicInteger _sessionsRemovedDueToInactivity = new AtomicInteger();

    private final AtomicInteger _sessionsRemovedByJVM = new AtomicInteger();

    private final AtomicInteger _syncStatesRemovedDueToInactivity = new AtomicInteger();

    private final AtomicInteger _syncStateSavedToDatabase = new AtomicInteger();

    private final AtomicInteger _syncStatesLoadedFromDatabase = new AtomicInteger();

    private final AtomicInteger _sessionsRemovedSinceLastLogging = new AtomicInteger();

    private final AtomicInteger _sessionsRemovedByJVMSinceLastLogging = new AtomicInteger();

    private final AtomicInteger _syncStatesRemovedSinceLastLogging = new AtomicInteger();

    public USMCacheInformation(SessionManagerImpl sessionManager) {
        _sessionManager = sessionManager;
    }

    public int getSessionsRemovedByJVMSinceLastLogging() {
        return _sessionsRemovedByJVMSinceLastLogging.getAndSet(0);
    }

    public int getSessionsRemovedSinceLastLogging() {
        return _sessionsRemovedSinceLastLogging.getAndSet(0);
    }

    public int getSyncStatesRemovedSinceLastLogging() {
        return _syncStatesRemovedSinceLastLogging.getAndSet(0);
    }

    public int getUSMSesssionCacheInactivityTimeout() {
        return _sessionManager.getSessionCacheTimeout();
    }

    public void setUSMSesssionCacheInactivityTimeout(int timeout) {
        _sessionManager.setSessionCacheTimeout(timeout);
    }

    public int getSyncStateCacheInactivityTimeout() {
        return _sessionManager.getSyncStateCacheTimeout();
    }

    public void setSyncStateCacheInactivityTimeout(int timeout) {
        _sessionManager.setSyncStateCacheTimeout(timeout);
    }

    public int getTotalUSMSessionsInMemory() {
        return _sessionManager.getSessionStorage().getSessionCount();
    }

    public int getTotalSyncStatesInMemory() {
        return _sessionManager.getSyncStateCacheProvider().getSyncStateCount();
    }

    public void clearCacheCounters() {
        _newSessions.set(0);
        _sessionsRemovedDueToInactivity.set(0);
        _sessionsRemovedByJVM.set(0);
        _syncStatesRemovedDueToInactivity.set(0);
        _syncStateSavedToDatabase.set(0);
        _syncStatesLoadedFromDatabase.set(0);
        _sessionsRemovedSinceLastLogging.set(0);
        _sessionsRemovedByJVMSinceLastLogging.set(0);
        _syncStatesRemovedSinceLastLogging.set(0);
    }

    public int getNewUSMSessionsInMemory() {
        return _newSessions.get();
    }

    public void newSessionCreated() {
        _newSessions.incrementAndGet();
    }

    public int getUSMSessionsRemovedDueToInactivity() {
        return _sessionsRemovedDueToInactivity.get();
    }

    public void sessionsRemovedDueToInactivity(int count) {
        _sessionsRemovedSinceLastLogging.addAndGet(count);
        _sessionsRemovedDueToInactivity.addAndGet(count);
    }

    public int getUSMSessionsRemovedByJVM() {
        return _sessionsRemovedByJVM.get();
    }

    public void sessionsRemovedByJVM(int jvmRemovals) {
        _sessionsRemovedByJVMSinceLastLogging.addAndGet(jvmRemovals);
        _sessionsRemovedByJVM.addAndGet(jvmRemovals);
    }

    public int getSyncStatesRemovedDueToInactivity() {
        return _syncStatesRemovedDueToInactivity.get();
    }

    public void syncStatesRemovedDueToInactivity(int removedSyncStates) {
        _syncStatesRemovedSinceLastLogging.addAndGet(removedSyncStates);
        _syncStatesRemovedDueToInactivity.addAndGet(removedSyncStates);
    }

    public int getSyncStatesLoadedFromDatabase() {
        return _syncStatesLoadedFromDatabase.get();
    }

    public void syncStatesLoadedFromDatabase(int count) {
        _syncStatesLoadedFromDatabase.addAndGet(count);
    }

    public int getSyncStatesSavedToDatabase() {
        return _syncStateSavedToDatabase.get();
    }

    public void syncStateSavedToDatabase() {
        _syncStateSavedToDatabase.incrementAndGet();
    }
}
