/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.session.cache;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import com.openexchange.usm.api.cache.SyncStateCache;
import com.openexchange.usm.api.session.assets.CompleteSessionID;

/**
 * {@link MemorySyncStateCache}. SyncStateCache for 1 USM session, which keeps all cached data in the Java heap.
 * 
 * @author <a href="mailto:afe@microdoc.de">Alexander Feess</a>
 */
public class MemorySyncStateCache implements SyncStateCache {

    private final int _maxSyncStatesPerObjectID;
    
    private final Map<String, MemorySyncStateCacheEntry[]> _internalCache = new HashMap<String, MemorySyncStateCacheEntry[]>();

    public MemorySyncStateCache(int maxSyncStatesPerObjectID) {
        super();
        _maxSyncStatesPerObjectID = maxSyncStatesPerObjectID;
    }

    /*
     * (non-Javadoc)
     * @see com.openexchange.usm.session.syncstates.SyncStateCache#put(com.openexchange.usm.session.impl.CompleteSessionID,
     * java.lang.String, long, byte[])
     */
    @Override
    public boolean put(CompleteSessionID sessionID, String objectID, long timestamp, Serializable[][] data) {
        synchronized (_internalCache) {
            MemorySyncStateCacheEntry newEntry = new MemorySyncStateCacheEntry(timestamp, data);
            MemorySyncStateCacheEntry[] oldData = _internalCache.get(objectID);
            if (oldData == null) {
                _internalCache.put(objectID, new MemorySyncStateCacheEntry[] { newEntry });
                return true;
            }
            for (MemorySyncStateCacheEntry entry : oldData) {
                if (entry.getTimestamp() == timestamp)
                    return false;
            }
            if(oldData.length >= _maxSyncStatesPerObjectID) {
                // This is an additional security check to make sure the cache doesn't grow to large. SyncStateStorage already ensures that this never happens
                long oldestTimestamp = Long.MAX_VALUE;
                int oldestIndex = -1;
                for (int i = 0; i < oldData.length; i++) {
                    MemorySyncStateCacheEntry entry = oldData[i];
                    if(entry.getTimestamp() < oldestTimestamp) {
                        oldestTimestamp = entry.getTimestamp();
                        oldestIndex = i;
                    }
                }
                if(oldestIndex >= 0) {
                    oldData[oldestIndex] = newEntry;
                    return true;
                }
            }
            MemorySyncStateCacheEntry[] newData = new MemorySyncStateCacheEntry[oldData.length + 1];
            for (int i = 0; i < oldData.length; i++)
                newData[i] = oldData[i];
            newData[oldData.length] = newEntry;
            _internalCache.put(objectID, newData);
            return true;
        }
    }

    /*
     * (non-Javadoc)
     * @see com.openexchange.usm.session.syncstates.SyncStateCache#get(com.openexchange.usm.session.impl.CompleteSessionID,
     * java.lang.String, long)
     */
    @Override
    public Serializable[][] get(CompleteSessionID sessionID, String objectID, long timestamp) {
        MemorySyncStateCacheEntry[] storedData = getStoredData(objectID);
        if (storedData != null) {
            for (MemorySyncStateCacheEntry entry : storedData) {
                if (entry.getTimestamp() == timestamp)
                    return entry.getData();
            }
        }
        return null;
    }

    /*
     * (non-Javadoc)
     * @see com.openexchange.usm.session.syncstates.SyncStateCache#remove(com.openexchange.usm.session.impl.CompleteSessionID,
     * java.lang.String, long)
     */
    @Override
    public void remove(CompleteSessionID sessionID, String objectID, long timestamp) {
        synchronized (_internalCache) {
            MemorySyncStateCacheEntry[] storedData = _internalCache.get(objectID);
            if (storedData != null) {
                for (int i = 0; i < storedData.length; i++) {
                    if (storedData[i].getTimestamp() == timestamp) {
                        if (storedData.length == 1) {
                            _internalCache.remove(objectID);
                            return;
                        }
                        MemorySyncStateCacheEntry[] newData = new MemorySyncStateCacheEntry[storedData.length - 1];
                        for (int j = 0; j < i; j++)
                            newData[j] = storedData[j];
                        for (int j = i; j < newData.length; j++)
                            newData[j] = storedData[j + 1];
                        _internalCache.put(objectID, newData);
                        return;
                    }
                }
            }
        }
    }

    /*
     * (non-Javadoc)
     * @see com.openexchange.usm.session.syncstates.SyncStateCache#removeSyncStates(com.openexchange.usm.session.impl.CompleteSessionID,
     * java.lang.String[])
     */
    @Override
    public void removeSyncStates(CompleteSessionID sessionID, String... objectIDs) {
        synchronized (_internalCache) {
            for (String objectID : objectIDs)
                _internalCache.remove(objectID);
        }
    }

    /*
     * (non-Javadoc)
     * @see com.openexchange.usm.session.syncstates.SyncStateCache#retainSyncStates(com.openexchange.usm.session.impl.CompleteSessionID,
     * java.util.Set)
     */
    @Override
    public void retainSyncStates(CompleteSessionID sessionID, Set<String> objectIDs) {
        synchronized (_internalCache) {
            for (Iterator<Entry<String, MemorySyncStateCacheEntry[]>> i = _internalCache.entrySet().iterator(); i.hasNext();) {
                Entry<String, MemorySyncStateCacheEntry[]> entry = i.next();
                if (!objectIDs.contains(entry.getKey()))
                    i.remove();
            }
        }
    }

    /*
     * (non-Javadoc)
     * @see com.openexchange.usm.session.syncstates.SyncStateCache#remapStates(com.openexchange.usm.session.impl.CompleteSessionID,
     * java.lang.String, java.lang.String)
     */
    @Override
    public void remapStates(CompleteSessionID sessionID, String oldObjectID, String newObjectID) {
        synchronized (_internalCache) {
            MemorySyncStateCacheEntry[] data = _internalCache.remove(oldObjectID);
            if (data != null)
                _internalCache.put(newObjectID, data);
        }
    }

    public int getSyncStateCount() {
        int count = 0;
        synchronized (_internalCache) {
            for (Entry<String, MemorySyncStateCacheEntry[]> entry : _internalCache.entrySet()) {
                count += entry.getValue().length;
            }
        }
        return count;
    }

    public int removeUnusedSyncStates(long limit) {
        int count = 0;
        synchronized (_internalCache) {
            for (Iterator<Entry<String, MemorySyncStateCacheEntry[]>> i = _internalCache.entrySet().iterator(); i.hasNext();) {
                Entry<String, MemorySyncStateCacheEntry[]> entry = i.next();
                int subCount = 0;
                MemorySyncStateCacheEntry[] storedData = entry.getValue();
                for (MemorySyncStateCacheEntry e : storedData) {
                    if (e.getLastAccess() < limit)
                        subCount++;
                }
                count += subCount;
                if (subCount == storedData.length) {
                    i.remove();
                } else if (subCount > 0) {
                    MemorySyncStateCacheEntry[] newData = new MemorySyncStateCacheEntry[storedData.length - subCount];
                    for (int j = 0, k = 0; j < storedData.length; j++) {
                        if (storedData[j].getLastAccess() >= limit)
                            newData[k++] = storedData[j];
                    }
                    entry.setValue(newData);
                }
            }
        }
        return count;
    }

    private MemorySyncStateCacheEntry[] getStoredData(String objectID) {
        synchronized (_internalCache) {
            return _internalCache.get(objectID);
        }
    }
}
