/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.session.dataobject;

import java.util.Map;
import com.openexchange.usm.api.contenttypes.common.ContentType;
import com.openexchange.usm.api.contenttypes.common.DefaultContentTypes;
import com.openexchange.usm.api.contenttypes.folder.FolderConstants;
import com.openexchange.usm.api.contenttypes.folder.FolderContentType;
import com.openexchange.usm.api.session.Folder;
import com.openexchange.usm.api.session.Session;
import com.openexchange.usm.api.session.assets.ChangeState;
import com.openexchange.usm.api.session.assets.FolderType;

/**
 * Base implementation of Folder based on SimpleDataObject. When the ContentType of the
 * elements is set, it automatically updates the appropriate field (by default "module",
 * but another field may be specified in the constructor).<br>
 * Attention: There is no special handling for the other direction, i.e. if the field
 * "module" is set, the ContentType of the elements is not automatically set. This is due
 * to the fact that this object has no knowledge of available ContentTypes and therefore
 * the sync-system itself has to make sure that the ContentType of elements is correctly
 * set.
 *
 * @author afe
 *
 */
public class FolderImpl extends SimpleDataObject implements Folder {

    private static int _elementsContentTypeFieldIndex = -1;

    private static int _subfoldersFieldIndex = -1;

    private static int _createdByFieldIndex = -1;

    private static int _typeFieldIndex = -1;

    private static void initializeFieldIndexes(Map<String, Integer> nameToIndexMap) {
        if (_elementsContentTypeFieldIndex == -1) {
            _elementsContentTypeFieldIndex = nameToIndexMap.get("module");
            _subfoldersFieldIndex = nameToIndexMap.get("subfolders");
            _createdByFieldIndex = nameToIndexMap.get("created_by");
            _typeFieldIndex = nameToIndexMap.get("type");
        }
    }

    private ContentType _elementsContentType;

    private long _contentTimestamp, _originalContentTimestamp;

    public FolderImpl(Session session, FolderContentType contentType) {
        super(session, contentType);
        if (!contentType.getID().equals(DefaultContentTypes.FOLDER_ID)) {
            throw new IllegalArgumentException("Illegal ContentType " + contentType.getID() + " for Folder object");
        }
        initializeFieldIndexes(_fieldNameToIndexMap);
        _fieldContent[_fieldNameToIndexMap.get(FolderConstants.SUBSCRIBED)] = true;
    }

    protected FolderImpl(FolderImpl source, boolean linkUUIDs) {
        super(source, linkUUIDs);
        _elementsContentType = source._elementsContentType;
        _contentTimestamp = source._contentTimestamp;
        _originalContentTimestamp = source._originalContentTimestamp;
    }

    @Override
    public FolderContentType getContentType() {
        return (FolderContentType) super.getContentType();
    }

    @Override
    public ContentType getElementsContentType() {
        return _elementsContentType;
    }

    @Override
    public void setElementsContentType(ContentType type) {
        if (_elementsContentType != null) {
            throw new IllegalStateException("ContentType of elements already set");
        }
        if (type != null) {
            _elementsContentType = type;
            setFieldContent(_elementsContentTypeFieldIndex, type.getID());
        }
    }

    @Override
    public String getElementsContentTypeID() {
        if (_elementsContentType != null) {
            return _elementsContentType.getID();
        }
        return convertObjectToString(getFieldContent(_elementsContentTypeFieldIndex));
    }

    @Override
    public long getContentSyncTimestamp() {
        return _contentTimestamp;
    }

    @Override
    public long getOriginalContentSyncTimestamp() {
        return _originalContentTimestamp;
    }

    @Override
    public void setContentSyncTimestamp(long timestamp) {
        _contentTimestamp = timestamp;
        updateChangeState(_contentTimestamp != _originalContentTimestamp);
    }

    @Override
    public void rollbackChanges() {
        _contentTimestamp = _originalContentTimestamp;
        super.rollbackChanges();
    }

    @Override
    public void commitChanges() {
        _originalContentTimestamp = _contentTimestamp;
        super.commitChanges();
    }

    @Override
    protected void updateChangeState(boolean newFieldContentIsNotEqual) {
        super.updateChangeState(newFieldContentIsNotEqual);
        if (_changeState == ChangeState.UNMODIFIED && _contentTimestamp != _originalContentTimestamp) {
            _changeState = ChangeState.MODIFIED;
        }
    }

    @Override
    protected void addSpecialFieldsToStringBuilder(StringBuilder sb) {
        super.addSpecialFieldsToStringBuilder(sb);
        sb.append(',');
        if (_formatToString) {
            sb.append("\n   ");
        }
        sb.append("ContentTimestamp:").append(_originalContentTimestamp);
        if (_contentTimestamp != _originalContentTimestamp) {
            sb.append("=>").append(_contentTimestamp);
        }
    }

    @Override
    public Folder createCopy(boolean linkUUIDs) {
        return new FolderImpl(this, linkUUIDs);
    }

    @Override
    public boolean hasSubFolders() {
        Object o = _fieldContent[_subfoldersFieldIndex];
        // Return true if either o is no Boolean (e.g. null), which means "unknown" or if the boolean value is true
        return !(o instanceof Boolean) || ((Boolean) o).booleanValue();
    }

    @Override
    public int getOwnerID() {
        Object o = _fieldContent[_typeFieldIndex];
        if (FolderType.SHARED.matches(o)) {
            o = _fieldContent[_createdByFieldIndex];
            if (o != null) {
                try {
                    return Integer.parseInt(o.toString());
                } catch (NumberFormatException ignored) {
                    // ignored
                }
            }
        }
        return 0;
    }

}
