/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */
package com.openexchange.usm.session.dataobject;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.apache.commons.logging.Log;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.Folder;
import com.openexchange.usm.api.session.Session;

/**
 * Compares two folders for their hierarchical order. 
 * 
 * @author ldo
 *
 */
public class FolderHierarchyComparator implements Comparator<Folder> {

	private static class InvalidFolderStructureException extends Exception {

		private static final long serialVersionUID = 1L;

	}

	public static Folder[] convertToSortedFoldersArray(List<DataObject> updates) {
		return convertToSortedFoldersArray(updates, null);
	}

	public static Folder[] convertToSortedFoldersArray(List<DataObject> updates, Log journal) {
		int size = updates.size();
		Folder[] folders = new Folder[size];
		for (int i = 0; i < size; i++) {
			DataObject o = updates.get(i);
			if (!(o instanceof Folder))
				return null;
			folders[i] = (Folder) o;
		}
		Comparator<Folder> comparator = (journal == null) ? new FolderHierarchyComparator(folders)
				: new FolderHierarchyComparator(folders, journal);
		Arrays.sort(folders, comparator);
		return folders;
	}

	private final Folder[] _allFolders;
	private final List<Folder> _firstHierarchy = new ArrayList<Folder>();
	private final List<Folder> _secondHierarchy = new ArrayList<Folder>();
	private final Map<String, Folder> _foldersByID = new HashMap<String, Folder>();

	public FolderHierarchyComparator(Folder[] sortingArray) {
		_allFolders = sortingArray;
		for (Folder f : sortingArray) {
			if (f != null) {
				String id = f.getID();
				if (id != null)
					_foldersByID.put(id, f);
			}
		}
	}

	public FolderHierarchyComparator(Folder[] sortingArray, Log journal) {
		this(sortingArray);
		if (sortingArray.length > 0)
			validateFolders(sortingArray[0].getSession(), journal);
	}

	private void validateFolders(Session session, Log journal) {
		List<Folder> hierarchy = new ArrayList<Folder>();
		try {
			for (Folder f : _allFolders) {
				hierarchy.clear();
				while (f != null) {
					if (hierarchy.contains(f)) {
						hierarchy.add(f);
						throw new InvalidFolderStructureException();
					}
					hierarchy.add(f);
					f = findParentFolder(f);
				}
			}
		} catch (InvalidFolderStructureException e) {
			StringBuilder sb = new StringBuilder();
			sb.append(session).append(" Folder hierarchy to sort contains cyclic parent reference:");
			char c = '[';
			for (Folder f : hierarchy) {
				sb.append(c).append(f.getID());
				c = ',';
			}
			sb.append(']');
			journal.warn(sb, e);
		}
	}

	@Override
    public int compare(Folder f1, Folder f2) {
		if (f1 == f2)
			return 0;
		if (f1 == null)
			return 1;
		if (f2 == null)
			return -1;
		buildHierarchy(_firstHierarchy, f1);
		buildHierarchy(_secondHierarchy, f2);
		int i1 = _firstHierarchy.size();
		int i2 = _secondHierarchy.size();
		// Walk from root folder towards the 2 folders until different folders are encountered
		while (i1 > 0 && i2 > 0) {
			Folder p1 = _firstHierarchy.get(--i1);
			Folder p2 = _secondHierarchy.get(--i2);
			if (p1 != p2) // Not the same folder, compare based on differing parent folder ids
				return compareIDs(p1, p2);
		}
		// Either i1 or i2 must be > 0 and the other must be 0, i.e. one of the 2 folders is the parent (directly or indirectly) of the other
		// The parent folder is "before" the child folder
		return (i1 > i2) ? 1 : -1;
	}

	private void buildHierarchy(List<Folder> hierarchy, Folder f) {
		for (hierarchy.clear(); f != null && !hierarchy.contains(f); f = findParentFolder(f))
			hierarchy.add(f);
	}

	private int compareIDs(Folder f1, Folder f2) {
		if (f1 == null)
			return f2 == null ? 0 : 1;
		return f2 == null ? -1 : compareIDs(f1.getID(), f2.getID());
	}

	private int compareIDs(String id1, String id2) {
		if (id1 == null)
			return id2 == null ? 0 : 1;
		return id2 == null ? -1 : id1.compareTo(id2);
	}

	private Folder findParentFolder(Folder child) {
		Folder parent = child.getParentFolder();
		if (parent != null)
			return parent;
		return _foldersByID.get(child.getParentFolderID());
	}
}
