/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.usm.ox_json.osgi;

import com.openexchange.usm.api.configuration.ConfigurationManager;
import com.openexchange.usm.api.configuration.ConfigurationProperties;
import com.openexchange.usm.api.exceptions.USMInvalidConfigurationException;
import com.openexchange.usm.api.exceptions.USMStartupException;
import com.openexchange.usm.api.ox.json.OXJSONAccess;
import com.openexchange.usm.ox_json.impl.OXJSONAccessImpl;
import com.openexchange.usm.ox_json.impl.OXJSONErrorCode;
import com.openexchange.usm.util.AbstractUSMActivator;


/**
 * {@link OXJSONAccessActivator}
 *
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 */
public class OXJSONAccessActivator extends AbstractUSMActivator {

    /*
     * (non-Javadoc)
     * @see com.openexchange.usm.util.AbstractUSMActivator#getNeededServices()
     */
    @Override
    protected Class<?>[] getNeededServices() {
        return new Class<?>[]{ConfigurationManager.class};
    }

    /*
     * (non-Javadoc)
     * @see com.openexchange.usm.util.AbstractUSMActivator#register(org.osgi.framework.BundleContext)
     */
    @Override
    protected void register() {
        ConfigurationManager cm = (ConfigurationManager) context.getService(presentServices.get(ConfigurationManager.class));
        
        try {
            String accessUrl = cm.getProperty(ConfigurationProperties.OX_ACCESS_URL_PROPERTY,
                    ConfigurationProperties.OX_ACCESS_URL_DEFAULT, true);
            int maxConnections = cm.getProperty(
                ConfigurationProperties.OX_ACCESS_MAX_CONNECTIONS_PROPERTY,
                    ConfigurationProperties.OX_ACCESS_MAX_CONNECTIONS_DEFAULT, false);
            int connectionTimeout = cm.getProperty(
                ConfigurationProperties.OX_ACCESS_CONNECTION_TIMEOUT_PROPERTY,
                    ConfigurationProperties.OX_ACCESS_CONNECTION_TIMEOUT_DEFAULT, false);
            boolean performStaleChecking = cm.getProperty(
                ConfigurationProperties.OX_ACCESS_PERFORM_STALE_CHECKING_PROPERTY,
                    ConfigurationProperties.OX_ACCESS_PERFORM_STALE_CHECKING_DEFAULT, false);
            boolean staySignedIn = cm.getProperty(
                ConfigurationProperties.OX_ACCESS_STAY_SIGNED_IN_PROPERTY,
                    ConfigurationProperties.OX_ACCESS_STAY_SIGNED_IN_DEFAULT, true);
            boolean passHostHeaderFromClient = cm.getProperty(
                ConfigurationProperties.OX_ACCESS_PASS_HOST_HEADER_FROM_CLIENT,
                    ConfigurationProperties.OX_ACCESS_PASS_HOST_HEADER_FROM_CLIENT_DEFAULT, false);
            String reloginErrorCodes = cm.getProperty(
                ConfigurationProperties.OX_ACCESS_RELOGIN_ERROR_CODES_PROPERTY,
                    ConfigurationProperties.OX_ACCESS_RELOGIN_ERROR_CODES_DEFAULT, false);
            String missingObjectErrorCodes = cm.getProperty(
                    ConfigurationProperties.OX_ACCESS_MISSING_OBJECT_ERROR_CODES_PROPERTY,
                        ConfigurationProperties.OX_ACCESS_MISSING_OBJECT_ERROR_CODES_DEFAULT, false);
            OXJSONAccess jsonAccess = new OXJSONAccessImpl(accessUrl, maxConnections, connectionTimeout, performStaleChecking, staySignedIn, passHostHeaderFromClient, reloginErrorCodes, missingObjectErrorCodes);
            registerService(OXJSONAccess.class, jsonAccess);
            LOG.info("OXJSONAccess registered.");
        } catch (USMInvalidConfigurationException e) {
            throw new USMStartupException(OXJSONErrorCode.CONFIGURATION_ERROR,
                    "Could't read OX server access properties", e);
        }
    }

    /*
     * (non-Javadoc)
     * @see com.openexchange.usm.util.AbstractUSMActivator#unregister(org.osgi.framework.BundleContext)
     */
    @Override
    protected void unregister() {
        OXJSONAccessImpl o = (OXJSONAccessImpl) getRegisteredService(OXJSONAccess.class);
        if (o != null)
            o.deactivate();
        LOG.info("OXJSONAccess unregistered.");
    }

}
